<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentMetric;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\media\Entity\Media;
use Drupal\media\Entity\MediaType;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Comprehensive tests for XrayAuditQueryTaskMediaPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditQueryTaskMediaPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'file',
    'image',
    'media',
    'field',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The media plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric\XrayAuditQueryTaskMediaPlugin
   */
  protected $plugin;

  /**
   * Test media types created for testing.
   *
   * @var \Drupal\media\Entity\MediaType[]
   */
  protected $testMediaTypes = [];

  /**
   * Test media entities created for testing.
   *
   * @var \Drupal\media\Entity\Media[]
   */
  protected $testMediaEntities = [];

  /**
   * Plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install media entity schema and configuration.
    $this->installEntitySchema('media');
    $this->installEntitySchema('file');
    $this->installConfig(['media', 'image', 'file', 'system']);
    $this->installSchema('file', ['file_usage']);

    // Get task plugin manager.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');

    // Get plugin repository service for cache testing.
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('queries_data_media');

    // Create test media types and media entities.
    $this->createTestMediaTypes();
    $this->createTestMediaEntities();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clean up test media entities.
    foreach ($this->testMediaEntities as $media) {
      if ($media) {
        $media->delete();
      }
    }

    // Clean up test media types.
    foreach ($this->testMediaTypes as $type) {
      if ($type) {
        $type->delete();
      }
    }

    // Clear all cache.
    $this->pluginRepository->clearAllCache();

    parent::tearDown();
  }

  /**
   * Creates test media types.
   *
   * RED: Write helper first, expecting 4 media types to be created.
   * GREEN: Implementation creates image, document, video, audio types.
   */
  protected function createTestMediaTypes() {
    // Create image media type with source field.
    $image = MediaType::create([
      'id' => 'image',
      'label' => 'Image',
      'source' => 'image',
    ]);
    $image->save();

    // Create source field for image type.
    $this->createMediaSourceField('image', 'field_media_image', 'image');
    $this->testMediaTypes['image'] = $image;

    // Create document media type with source field.
    $document = MediaType::create([
      'id' => 'document',
      'label' => 'Document',
      'source' => 'file',
    ]);
    $document->save();

    // Create source field for document type.
    $this->createMediaSourceField('document', 'field_media_document', 'file');
    $this->testMediaTypes['document'] = $document;

    // Create video media type with source field.
    $video = MediaType::create([
      'id' => 'video',
      'label' => 'Video',
      'source' => 'video_file',
    ]);
    $video->save();

    // Create source field for video type.
    $this->createMediaSourceField('video', 'field_media_video_file', 'file');
    $this->testMediaTypes['video'] = $video;

    // Create audio media type (with no entities - edge case).
    $audio = MediaType::create([
      'id' => 'audio',
      'label' => 'Audio',
      'source' => 'audio_file',
    ]);
    $audio->save();

    // Create source field for audio type.
    $this->createMediaSourceField('audio', 'field_media_audio_file', 'file');
    $this->testMediaTypes['audio'] = $audio;
  }

  /**
   * Creates a source field for a media type.
   *
   * @param string $media_type
   *   The media type ID.
   * @param string $field_name
   *   The field name.
   * @param string $field_type
   *   The field type (image or file).
   */
  protected function createMediaSourceField(string $media_type, string $field_name, string $field_type) {
    $field_storage = FieldStorageConfig::create([
      'field_name' => $field_name,
      'entity_type' => 'media',
      'type' => $field_type,
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => $media_type,
      'label' => 'Source field',
    ]);
    $field->save();

    // Set as source field.
    $media_type_entity = MediaType::load($media_type);
    $source = $media_type_entity->getSource();
    $source_configuration = $source->getConfiguration();
    $source_configuration['source_field'] = $field_name;
    $source->setConfiguration($source_configuration);
    $media_type_entity->save();
  }

  /**
   * Creates test media entities with various types.
   */
  protected function createTestMediaEntities() {
    // Create 4 image media entities.
    for ($i = 1; $i <= 4; $i++) {
      $media = Media::create([
        'bundle' => 'image',
        'name' => "Test Image $i",
      ]);
      $media->save();
      $this->testMediaEntities[] = $media;
    }

    // Create 3 document media entities.
    for ($i = 1; $i <= 3; $i++) {
      $media = Media::create([
        'bundle' => 'document',
        'name' => "Test Document $i",
      ]);
      $media->save();
      $this->testMediaEntities[] = $media;
    }

    // Create 2 video media entities.
    for ($i = 1; $i <= 2; $i++) {
      $media = Media::create([
        'bundle' => 'video',
        'name' => "Test Video $i",
      ]);
      $media->save();
      $this->testMediaEntities[] = $media;
    }

    // Note: No audio media entities created (testing edge case).
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Tests plugin definition has correct values.
   */
  public function testPluginDefinitionValues() {
    // Arrange.
    $definition = $this->taskPluginManager->getDefinition('queries_data_media');

    // Assert.
    $this->assertEquals('queries_data_media', $definition['id']);
    $this->assertEquals('content_metric', $definition['group']);
    $this->assertEquals(3, $definition['sort']);

    // Verify operations.
    $this->assertArrayHasKey('operations', $definition);
    $this->assertArrayHasKey('media_count_type', $definition['operations']);

    // Verify operation details.
    $media_count_type_op = $definition['operations']['media_count_type'];
    $this->assertArrayHasKey('label', $media_count_type_op);
    $this->assertArrayHasKey('description', $media_count_type_op);
    $this->assertEquals('Medias grouped by type', (string) $media_count_type_op['label']);

    // Verify dependencies.
    $this->assertArrayHasKey('dependencies', $definition);
    $this->assertIsArray($definition['dependencies']);
    $this->assertContains('media', $definition['dependencies']);
  }

  // ---------------------------------------------------------------------------
  // getDataOperationResult()
  // ---------------------------------------------------------------------------

  /**
   * Tests getDataOperationResult() routes media_count_type operation.
   */
  public function testGetDataOperationResultRoutesMediaCountTypeOperation() {
    // Act.
    $result = $this->plugin->getDataOperationResult('media_count_type');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure (ID, Label, Count).
    $this->assertCount(3, $result['header_table']);

    // Verify results contain media data.
    $this->assertNotEmpty($result['results_table']);
    $this->assertIsArray($result['results_table']);
  }

  // ---------------------------------------------------------------------------
  // mediasTypes()
  // ---------------------------------------------------------------------------

  /**
   * Tests mediasTypes() returns proper structure.
   */
  public function testMediasTypesReturnsProperStructure() {
    // Act.
    $result = $this->plugin->getDataOperationResult('media_count_type');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure (ID, Label, Count).
    $this->assertCount(3, $result['header_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests mediasTypes() includes all media types.
   */
  public function testMediasTypesIncludesAllMediaTypes() {
    // Act.
    $result = $this->plugin->getDataOperationResult('media_count_type');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Extract type IDs from results.
    $type_ids = array_column($results_table, 0);

    // We created 4 types: image, document, video, audio.
    $this->assertContains('image', $type_ids, 'Results should include image type');
    $this->assertContains('document', $type_ids, 'Results should include document type');
    $this->assertContains('video', $type_ids, 'Results should include video type');
    $this->assertContains('audio', $type_ids, 'Results should include audio type');

    // Verify we have exactly 4 types.
    $this->assertCount(4, $results_table, 'Should have exactly 4 media types');
  }

  /**
   * Tests mediasTypes() counts media correctly.
   */
  public function testMediasTypesCountsMediaCorrectly() {
    // Act.
    $result = $this->plugin->getDataOperationResult('media_count_type');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Build associative array of type => count.
    $counts = [];
    foreach ($results_table as $row) {
      $counts[$row[0]] = $row[2];
    }

    // We created:
    // - 4 image media entities
    // - 3 document media entities
    // - 2 video media entities
    // - 0 audio media entities (edge case)
    $this->assertEquals(4, $counts['image'], 'Image type should have 4 media entities');
    $this->assertEquals(3, $counts['document'], 'Document type should have 3 media entities');
    $this->assertEquals(2, $counts['video'], 'Video type should have 2 media entities');
    $this->assertEquals(0, $counts['audio'], 'Audio type should have 0 media entities');
  }

  /**
   * Tests mediasTypes() handles media type with no media entities.
   */
  public function testMediasTypesHandlesMediaTypeWithNoMedia() {
    // Act.
    $result = $this->plugin->getDataOperationResult('media_count_type');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Find audio type (has no media entities).
    $audio_found = FALSE;
    foreach ($results_table as $row) {
      if ($row[0] === 'audio') {
        $audio_found = TRUE;
        $this->assertEquals(0, $row[2], 'Audio type should have 0 count');
        $this->assertEquals('Audio', $row[1], 'Audio type should have correct label');
      }
    }

    $this->assertTrue($audio_found, 'Audio type should be in results even with zero count');
  }

  // ---------------------------------------------------------------------------
  // buildDataRenderArray()
  // ---------------------------------------------------------------------------

  /**
   * Tests buildDataRenderArray() returns standard structure.
   */
  public function testBuildDataRenderArrayReturnsStandardStructure() {
    // Arrange: Get operation data.
    $operation = 'media_count_type';
    $data = $this->plugin->getDataOperationResult($operation);

    // Act: Build render array with data.
    $render_array = $this->plugin->buildDataRenderArray($data, $operation);

    // Assert.
    $this->assertIsArray($render_array);

    // Verify render array has table.
    $this->assertArrayHasKey('table', $render_array);
    $this->assertArrayHasKey('#type', $render_array['table']);
    $this->assertArrayHasKey('#header', $render_array['table']);
    $this->assertArrayHasKey('#rows', $render_array['table']);

    // Verify it's a valid render array structure.
    $this->assertNotEmpty($render_array, 'Render array should not be empty');
  }

}
