<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentMetric;

use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Comprehensive tests for XrayAuditQueryTaskNodePlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditQueryTaskNodePluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'filter',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The node plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric\XrayAuditQueryTaskNodePlugin
   */
  protected $plugin;

  /**
   * Test content types created for testing.
   *
   * @var \Drupal\node\Entity\NodeType[]
   */
  protected $testContentTypes = [];

  /**
   * Test nodes created for testing.
   *
   * @var \Drupal\node\Entity\Node[]
   */
  protected $testNodes = [];

  /**
   * Plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install node entity schema and related schemas.
    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['node', 'filter', 'system']);

    // Get task plugin manager.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');

    // Get plugin repository service for cache testing.
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('queries_data_nodes');

    // Create test content types and nodes.
    $this->createTestContentTypes();
    $this->createTestNodes();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clean up test nodes.
    foreach ($this->testNodes as $node) {
      if ($node) {
        $node->delete();
      }
    }

    // Clean up test content types.
    foreach ($this->testContentTypes as $type) {
      if ($type) {
        $type->delete();
      }
    }

    // Clear all cache.
    $this->pluginRepository->clearAllCache();

    parent::tearDown();
  }

  /**
   * Creates test content types.
   */
  protected function createTestContentTypes() {
    // Create article content type.
    $article = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $article->save();
    $this->testContentTypes['article'] = $article;

    // Create page content type.
    $page = NodeType::create([
      'type' => 'page',
      'name' => 'Page',
    ]);
    $page->save();
    $this->testContentTypes['page'] = $page;

    // Create blog content type.
    $blog = NodeType::create([
      'type' => 'blog',
      'name' => 'Blog',
    ]);
    $blog->save();
    $this->testContentTypes['blog'] = $blog;
  }

  /**
   * Creates test nodes with various types, statuses, and languages.
   */
  protected function createTestNodes() {
    // Create 3 published articles in English.
    for ($i = 1; $i <= 3; $i++) {
      $node = Node::create([
        'type' => 'article',
        'title' => "Article $i EN",
        'langcode' => 'en',
        'status' => 1,
      ]);
      $node->save();
      $this->testNodes[] = $node;
    }

    // Create 2 unpublished articles in English.
    for ($i = 1; $i <= 2; $i++) {
      $node = Node::create([
        'type' => 'article',
        'title' => "Unpublished Article $i EN",
        'langcode' => 'en',
        'status' => 0,
      ]);
      $node->save();
      $this->testNodes[] = $node;
    }

    // Create 2 published pages in English.
    for ($i = 1; $i <= 2; $i++) {
      $node = Node::create([
        'type' => 'page',
        'title' => "Page $i EN",
        'langcode' => 'en',
        'status' => 1,
      ]);
      $node->save();
      $this->testNodes[] = $node;
    }

    // Create 1 unpublished page in English.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Unpublished Page EN',
      'langcode' => 'en',
      'status' => 0,
    ]);
    $node->save();
    $this->testNodes[] = $node;

    // Create 1 published article in Spanish.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Article ES',
      'langcode' => 'es',
      'status' => 1,
    ]);
    $node->save();
    $this->testNodes[] = $node;

    // Create 1 published blog in French.
    $node = Node::create([
      'type' => 'blog',
      'title' => 'Blog FR',
      'langcode' => 'fr',
      'status' => 1,
    ]);
    $node->save();
    $this->testNodes[] = $node;

    // Create node with multiple revisions (10 revisions).
    $node = Node::create([
      'type' => 'page',
      'title' => 'Page with many revisions',
      'langcode' => 'en',
      'status' => 1,
    ]);
    $node->save();
    $this->testNodes[] = $node;

    // Create 9 additional revisions.
    for ($i = 1; $i <= 9; $i++) {
      $node->setNewRevision(TRUE);
      $node->setTitle("Page revision $i");
      $node->save();
    }

    // Create another node with moderate revisions (5 revisions).
    $node = Node::create([
      'type' => 'article',
      'title' => 'Article with moderate revisions',
      'langcode' => 'en',
      'status' => 1,
    ]);
    $node->save();
    $this->testNodes[] = $node;

    // Create 4 additional revisions.
    for ($i = 1; $i <= 4; $i++) {
      $node->setNewRevision(TRUE);
      $node->setTitle("Article revision $i");
      $node->save();
    }
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  // ---------------------------------------------------------------------------
  // NODE COUNT OPERATIONS TESTS
  // ---------------------------------------------------------------------------

  /**
   * Tests nodeByTypes() returns proper structure.
   */
  public function testNodesCountByTypeReturnsProperStructure() {
    // Act.
    $result = $this->plugin->getDataOperationResult('node_count_type');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure has 5 columns.
    $this->assertCount(5, $result['header_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests nodeByTypes() includes all content types.
   */
  public function testNodesCountByTypeIncludesAllTypes() {
    // Act.
    $result = $this->plugin->getDataOperationResult('node_count_type');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Extract type IDs from results (excluding total row).
    $type_ids = [];
    foreach ($results_table as $key => $row) {
      if ($key !== 'total') {
        $type_ids[] = $key;
      }
    }

    // We created 3 types: article, page, blog.
    $this->assertContains('article', $type_ids, 'Results should include article type');
    $this->assertContains('page', $type_ids, 'Results should include page type');
    $this->assertContains('blog', $type_ids, 'Results should include blog type');
  }

  /**
   * Tests nodeByTypes() counts correctly by type and status.
   */
  public function testNodesCountByTypeCountsCorrectly() {
    // Act.
    $result = $this->plugin->getDataOperationResult('node_count_type');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // We created:
    // - 6 articles total (4 published EN + 1 published ES + 2 unpublished EN = 5 published, 2 unpublished)
    // - 4 pages total (2 published EN + 1 unpublished EN + 1 with revisions = 3 published, 1 unpublished)
    // - 1 blog (1 published FR)
    // Note: The node with revisions adds 1 to the count.
    $this->assertEquals(7, $results_table['article']['total'], 'Article type should have 7 nodes');
    $this->assertEquals(4, $results_table['page']['total'], 'Page type should have 4 nodes');
    $this->assertEquals(1, $results_table['blog']['total'], 'Blog type should have 1 node');

    // Verify published/unpublished counts.
    $this->assertEquals(5, $results_table['article']['published'], 'Should have 5 published articles');
    $this->assertEquals(2, $results_table['article']['unpublished'], 'Should have 2 unpublished articles');
    $this->assertEquals(3, $results_table['page']['published'], 'Should have 3 published pages');
    $this->assertEquals(1, $results_table['page']['unpublished'], 'Should have 1 unpublished page');

    // Verify total row.
    $this->assertArrayHasKey('total', $results_table);
    $this->assertEquals(12, $results_table['total']['total'], 'Total should be 12 nodes');
  }

  /**
   * Tests nodeTypesPerLanguage() returns proper structure.
   */
  public function testNodesCountByTypeAndLanguageReturnsStructure() {
    // Act.
    $result = $this->plugin->getDataOperationResult('node_count_type_langcode');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure (ID, Label, Language, Total, Published, Unpublished).
    $this->assertCount(6, $result['header_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests nodeTypesPerLanguage() groups by language.
   */
  public function testNodesCountByTypeAndLanguageGroupsByLanguage() {
    // Act.
    $result = $this->plugin->getDataOperationResult('node_count_type_langcode');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Each row should have id, label, language, total, published, unpublished.
    foreach ($results_table as $row) {
      $this->assertArrayHasKey('id', $row);
      $this->assertArrayHasKey('label', $row);
      $this->assertArrayHasKey('language', $row);
      $this->assertArrayHasKey('total', $row);
      $this->assertArrayHasKey('published', $row);
      $this->assertArrayHasKey('unpublished', $row);
    }

    // Verify we have multiple language codes.
    $langcodes = [];
    foreach ($results_table as $row) {
      $langcodes[] = $row['language'];
    }
    $langcodes = array_unique($langcodes);

    $this->assertGreaterThan(1, count($langcodes), 'Should have multiple language codes');
    $this->assertContains('en', $langcodes, 'Should include English nodes');
    $this->assertContains('es', $langcodes, 'Should include Spanish nodes');
    $this->assertContains('fr', $langcodes, 'Should include French nodes');
  }

  /**
   * Tests nodeTypesPerLanguage() sorted by type.
   */
  public function testNodesCountByTypeAndLanguageSortsByType() {
    // Act.
    $result = $this->plugin->getDataOperationResult('node_count_type_langcode');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Extract type IDs and languages.
    $entries = [];
    foreach ($results_table as $row) {
      $entries[] = [
        'type' => $row['id'],
        'lang' => $row['language'],
      ];
    }

    // Verify results contain all expected types.
    $type_ids = array_column($entries, 'type');
    $unique_types = array_unique($type_ids);

    $this->assertContains('article', $unique_types, 'Results should include article type');
    $this->assertContains('page', $unique_types, 'Results should include page type');
    $this->assertContains('blog', $unique_types, 'Results should include blog type');

    // Verify we have multiple entries (type-language combinations).
    $this->assertGreaterThan(3, count($entries), 'Should have more than 3 entries due to language combinations');
  }

  // ---------------------------------------------------------------------------
  // REVISION COUNTING TESTS
  // ---------------------------------------------------------------------------

  /**
   * Tests nodesWithExcessiveRevisions() identifies high revision nodes.
   *
   */
  public function testNodeRevisionsCountIdentifiesHighRevisionNodes() {
    // Act.
    $result = $this->plugin->getDataOperationResult('node_revisions_count');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Extract revision counts.
    $revision_counts = [];
    foreach ($results_table as $row) {
      if (isset($row['revisions']['data'])) {
        $revision_counts[] = $row['revisions']['data'];
      }
    }

    // Verify results are ordered descending.
    $sorted_counts = $revision_counts;
    rsort($sorted_counts);
    $this->assertEquals($sorted_counts, $revision_counts, 'Results should be ordered by revision count descending');

    // Verify highest count is 10 (our node with 10 revisions).
    $this->assertEquals(10, $revision_counts[0], 'Highest revision count should be 10');
  }

  /**
   * Tests nodesWithExcessiveRevisions() respects thresholds.
   *
   */
  public function testNodeRevisionsCountRespectsThresholds() {
    // Arrange: Set revision threshold in config.
    $config = $this->config('xray_audit.settings');
    $config->set('revisions_thresholds.node', 8);
    $config->save();

    // Recreate plugin to load new config (config is loaded in constructor).
    $this->plugin = $this->taskPluginManager->createInstance('queries_data_nodes');

    // Act.
    $result = $this->plugin->getDataOperationResult('node_revisions_count');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Find node with 10 revisions.
    $found_excessive = FALSE;
    foreach ($results_table as $row) {
      if (isset($row['revisions']['data']) && $row['revisions']['data'] >= 10) {
        // Should have error class.
        $this->assertArrayHasKey('class', $row['revisions'], 'Node with 10 revisions should have error class');
        $this->assertEquals('xray-audit-color-error', $row['revisions']['class']);
        $found_excessive = TRUE;
      }
    }

    $this->assertTrue($found_excessive, 'Should find at least one node exceeding threshold');
  }

  /**
   * Tests nodesWithExcessiveRevisions() includes all revision data.
   *
   */
  public function testNodeRevisionsCountIncludesAllRevisionData() {
    // Act.
    $result = $this->plugin->getDataOperationResult('node_revisions_count');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Verify each row has required fields.
    foreach ($results_table as $row) {
      $this->assertArrayHasKey('id', $row);
      $this->assertArrayHasKey('label', $row);
      $this->assertArrayHasKey('revisions', $row);

      // Verify revisions has data field.
      if (is_array($row['revisions'])) {
        $this->assertArrayHasKey('data', $row['revisions']);
      }
    }
  }

  /**
   * Tests nodesWithExcessiveRevisions() handles no revisions case.
   *
   */
  public function testNodeRevisionsCountHandlesNoRevisions() {
    // Arrange: Delete all test nodes.
    foreach ($this->testNodes as $node) {
      $node->delete();
    }
    $this->testNodes = [];

    // Act.
    $result = $this->plugin->getDataOperationResult('node_revisions_count');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Results should have default "no nodes" message.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // First row should be the "No nodes found" message.
    $first_row = reset($results_table);
    $this->assertIsArray($first_row);
  }

  // ---------------------------------------------------------------------------
  // INTEGRATION & DATA TESTS
  // ---------------------------------------------------------------------------

  /**
   * Tests plugin definition has correct values.   *
   */
  public function testPluginDefinitionValues() {
    // Arrange.
    $definition = $this->taskPluginManager->getDefinition('queries_data_nodes');

    // Assert.
    $this->assertEquals('queries_data_nodes', $definition['id']);
    $this->assertEquals('content_metric', $definition['group']);
    $this->assertEquals(1, $definition['sort']);
    $this->assertEquals(1, $definition['local_task']);

    // Verify operations.
    $this->assertArrayHasKey('operations', $definition);
    $this->assertCount(3, $definition['operations'], 'Should have exactly 3 operations');
    $this->assertArrayHasKey('node_count_type', $definition['operations']);
    $this->assertArrayHasKey('node_count_type_langcode', $definition['operations']);
    $this->assertArrayHasKey('node_revisions_count', $definition['operations']);

    // Verify operation details.
    foreach ($definition['operations'] as $operation) {
      $this->assertArrayHasKey('label', $operation);
      $this->assertArrayHasKey('description', $operation);
    }
  }

}
