<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentMetric;

use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Comprehensive tests for XrayAuditQueryTaskVocabularyPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditQueryTaskVocabularyPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'taxonomy',
    'text',
    'field',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The vocabulary plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric\XrayAuditQueryTaskVocabularyPlugin
   */
  protected $plugin;

  /**
   * Test vocabularies created for testing.
   *
   * @var \Drupal\taxonomy\Entity\Vocabulary[]
   */
  protected $testVocabularies = [];

  /**
   * Test taxonomy terms created for testing.
   *
   * @var \Drupal\taxonomy\Entity\Term[]
   */
  protected $testTerms = [];

  /**
   * Plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install taxonomy entity schema.
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('taxonomy_vocabulary');
    $this->installConfig(['taxonomy']);

    // Get task plugin manager.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');

    // Get plugin repository service for cache testing.
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('queries_data_vocabulary');

    // Create test vocabularies and terms.
    $this->createTestVocabularies();
    $this->createTestTerms();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clean up test terms.
    foreach ($this->testTerms as $term) {
      if ($term) {
        $term->delete();
      }
    }

    // Clean up test vocabularies.
    foreach ($this->testVocabularies as $vocabulary) {
      if ($vocabulary) {
        $vocabulary->delete();
      }
    }

    // Clear all cache.
    $this->pluginRepository->clearAllCache();

    parent::tearDown();
  }

  /**
   * Creates test vocabularies.
   */
  protected function createTestVocabularies() {
    // Create tags vocabulary.
    $tags = Vocabulary::create([
      'vid' => 'tags',
      'name' => 'Tags',
      'description' => 'Tags vocabulary',
    ]);
    $tags->save();
    $this->testVocabularies['tags'] = $tags;

    // Create categories vocabulary.
    $categories = Vocabulary::create([
      'vid' => 'categories',
      'name' => 'Categories',
      'description' => 'Categories vocabulary',
    ]);
    $categories->save();
    $this->testVocabularies['categories'] = $categories;

    // Create topics vocabulary.
    $topics = Vocabulary::create([
      'vid' => 'topics',
      'name' => 'Topics',
      'description' => 'Topics vocabulary',
    ]);
    $topics->save();
    $this->testVocabularies['topics'] = $topics;

    // Create empty vocabulary (no terms will be added).
    $empty = Vocabulary::create([
      'vid' => 'empty_vocab',
      'name' => 'Empty Vocabulary',
      'description' => 'Vocabulary with no terms',
    ]);
    $empty->save();
    $this->testVocabularies['empty_vocab'] = $empty;
  }

  /**
   * Creates test taxonomy terms with various vocabularies and languages.
   *
   * Test data structure:
   * - tags: 5 terms (3 in 'en', 2 in 'es')
   * - categories: 3 terms (all in 'en')
   * - topics: 2 terms (1 in 'en', 1 in 'fr')
   * - empty_vocab: 0 terms.
   */
  protected function createTestTerms() {
    // Create 3 tags in English.
    for ($i = 1; $i <= 3; $i++) {
      $term = Term::create([
        'vid' => 'tags',
        'name' => "Tag $i EN",
        'langcode' => 'en',
      ]);
      $term->save();
      $this->testTerms[] = $term;
    }

    // Create 2 tags in Spanish.
    for ($i = 1; $i <= 2; $i++) {
      $term = Term::create([
        'vid' => 'tags',
        'name' => "Tag $i ES",
        'langcode' => 'es',
      ]);
      $term->save();
      $this->testTerms[] = $term;
    }

    // Create 3 categories in English.
    for ($i = 1; $i <= 3; $i++) {
      $term = Term::create([
        'vid' => 'categories',
        'name' => "Category $i EN",
        'langcode' => 'en',
      ]);
      $term->save();
      $this->testTerms[] = $term;
    }

    // Create 1 topic in English.
    $term = Term::create([
      'vid' => 'topics',
      'name' => 'Topic EN',
      'langcode' => 'en',
    ]);
    $term->save();
    $this->testTerms[] = $term;

    // Create 1 topic in French.
    $term = Term::create([
      'vid' => 'topics',
      'name' => 'Topic FR',
      'langcode' => 'fr',
    ]);
    $term->save();
    $this->testTerms[] = $term;

    // Note: empty_vocab vocabulary has no terms created.
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  // ---------------------------------------------------------------------------
  // 2. getDataOperationResult()
  // ---------------------------------------------------------------------------

  /**
   * Tests getDataOperationResult() routes to vocabulariesCount() method.
   */
  public function testGetDataOperationResultRoutesVocabularyCountOperation() {
    // Act.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure (VID, Count).
    $this->assertCount(2, $result['header_table']);

    // Verify results contain vocabulary data.
    $this->assertNotEmpty($result['results_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests getDataOperationResult() routes to vocabulariesCountPerLanguage().
   */
  public function testGetDataOperationResultRoutesVocabularyCountPerLanguageOperation() {
    // Act.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type_langcode');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure (VID, Langcode, Count).
    $this->assertCount(3, $result['header_table']);

    // Verify results contain vocabulary data grouped by language.
    $this->assertNotEmpty($result['results_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests vocabulariesCount() returns proper structure.
   */
  public function testVocabulariesCountReturnsProperStructure() {
    // Act.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure (VID, Count).
    $this->assertCount(2, $result['header_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests vocabulariesCount() returns correct header labels.
   */
  public function testVocabulariesCountReturnsCorrectHeader() {
    // Act.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type');

    // Assert.
    $header = $result['header_table'];
    $this->assertCount(2, $header);

    // The header contains TranslatableMarkup objects, convert to strings.
    $header_strings = array_map('strval', $header);
    $this->assertEquals('VID', $header_strings[0]);
    $this->assertEquals('Count', $header_strings[1]);
  }

  /**
   * Tests vocabulariesCount() includes all vocabularies with terms.
   */
  public function testVocabulariesCountIncludesAllVocabularies() {
    // Act.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Extract vocabulary IDs from results.
    $vocabulary_ids = array_column($results_table, 0);

    // We created 3 vocabularies with terms: tags, categories, topics.
    $this->assertContains('tags', $vocabulary_ids, 'Results should include tags vocabulary');
    $this->assertContains('categories', $vocabulary_ids, 'Results should include categories vocabulary');
    $this->assertContains('topics', $vocabulary_ids, 'Results should include topics vocabulary');

    // Empty vocabulary SHOULD be included (with 0 terms) - plugin includes all vocabularies.
    $this->assertContains('empty_vocab', $vocabulary_ids, 'Results should include empty vocabulary with 0 count');
  }

  /**
   * Tests vocabulariesCount() counts terms correctly.
   */
  public function testVocabulariesCountCountsTermsCorrectly() {
    // Act.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Build associative array of vid => count.
    $counts = [];
    foreach ($results_table as $row) {
      $counts[$row[0]] = $row[1];
    }

    // We created:
    // - 5 tags (3 EN + 2 ES)
    // - 3 categories (3 EN)
    // - 2 topics (1 EN + 1 FR)
    $this->assertEquals(5, $counts['tags'], 'Tags vocabulary should have 5 terms');
    $this->assertEquals(3, $counts['categories'], 'Categories vocabulary should have 3 terms');
    $this->assertEquals(2, $counts['topics'], 'Topics vocabulary should have 2 terms');
  }

  /**
   * Tests vocabulariesCountPerLanguage() returns proper structure.
   *

   */
  public function testVocabulariesCountPerLanguageReturnsProperStructure() {
    // Act.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type_langcode');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure (VID, Langcode, Count).
    $this->assertCount(3, $result['header_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests vocabulariesCountPerLanguage() returns correct header labels.
   *

   */
  public function testVocabulariesCountPerLanguageReturnsCorrectHeader() {
    // Act.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type_langcode');

    // Assert.
    $header = $result['header_table'];
    $this->assertCount(3, $header);

    // The header contains TranslatableMarkup objects, convert to strings.
    $header_strings = array_map('strval', $header);
    $this->assertEquals('VID', $header_strings[0]);
    $this->assertEquals('Langcode', $header_strings[1]);
    $this->assertEquals('Count', $header_strings[2]);
  }

  /**
   * Tests vocabulariesCountPerLanguage() groups by vocabulary AND language.
   *

   */
  public function testVocabulariesCountPerLanguageGroupsByVocabularyAndLanguage() {
    // Act.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type_langcode');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Each row should have 3 elements: [vid, langcode, count].
    foreach ($results_table as $row) {
      $this->assertCount(3, $row, 'Each row should have 3 elements');
      $this->assertIsString($row[0], 'First element should be vid string');
      $this->assertIsString($row[1], 'Second element should be langcode string');
      $this->assertIsNumeric($row[2], 'Third element should be count numeric');
    }

    // Verify we have multiple rows (vocabulary-language combinations).
    // We expect at least 5 combinations:
    // - tags:en, tags:es
    // - categories:en
    // - topics:en, topics:fr.
    $this->assertGreaterThanOrEqual(5, count($results_table), 'Should have at least 5 vocabulary-language combinations');
  }

  /**
   * Tests vocabulariesCountPerLanguage() counts correctly per language.
   *

   */
  public function testVocabulariesCountPerLanguageCountsCorrectly() {
    // Act.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type_langcode');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Build associative array of vid:langcode => count.
    $counts = [];
    foreach ($results_table as $row) {
      $key = $row[0] . ':' . $row[1];
      $counts[$key] = $row[2];
    }

    // We created:
    // - tags:en = 3
    // - tags:es = 2
    // - categories:en = 3
    // - topics:en = 1
    // - topics:fr = 1.
    $this->assertEquals(3, $counts['tags:en'], 'Tags EN should have 3 terms');
    $this->assertEquals(2, $counts['tags:es'], 'Tags ES should have 2 terms');
    $this->assertEquals(3, $counts['categories:en'], 'Categories EN should have 3 terms');
    $this->assertEquals(1, $counts['topics:en'], 'Topics EN should have 1 term');
    $this->assertEquals(1, $counts['topics:fr'], 'Topics FR should have 1 term');
  }

  /**
   * Tests vocabulariesCountPerLanguage() handles multiple languages.
   *

   */
  public function testVocabulariesCountPerLanguageHandlesMultipleLanguages() {
    // Act.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type_langcode');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Extract unique langcodes.
    $langcodes = array_unique(array_column($results_table, 1));

    // We created terms in 3 languages: en, es, fr (may include additional system languages).
    $this->assertGreaterThanOrEqual(3, count($langcodes), 'Should have at least 3 languages');
    $this->assertContains('en', $langcodes, 'Results should include English terms');
    $this->assertContains('es', $langcodes, 'Results should include Spanish terms');
    $this->assertContains('fr', $langcodes, 'Results should include French terms');
  }

  /**
   * Tests buildDataRenderArray() uses parent method correctly.
   *
   */
  public function testBuildDataRenderArrayUsesParentMethod() {
    // Arrange.
    $data = [
      'header_table' => ['VID', 'Count'],
      'results_table' => [
        ['tags', 5],
        ['categories', 3],
      ],
    ];

    // Act.
    $render_array = $this->plugin->buildDataRenderArray($data, 'taxonomy_term_count_type');

    // Assert.
    $this->assertIsArray($render_array);
    $this->assertArrayHasKey('table', $render_array);
    $this->assertEquals('table', $render_array['table']['#type']);
    $this->assertEquals($data['header_table'], $render_array['table']['#header']);
    $this->assertEquals($data['results_table'], $render_array['table']['#rows']);
  }

  /**
   * Tests buildDataRenderArray() returns standard structure.
   *
   */
  public function testBuildDataRenderArrayReturnsStandardStructure() {
    // Arrange.
    $result = $this->plugin->getDataOperationResult('taxonomy_term_count_type');

    // Act.
    $render_array = $this->plugin->buildDataRenderArray($result, 'taxonomy_term_count_type');

    // Assert.
    $this->assertIsArray($render_array);
    $this->assertArrayHasKey('table', $render_array);
    $this->assertIsArray($render_array['table']);
    $this->assertEquals('table', $render_array['table']['#type']);
    $this->assertArrayHasKey('#header', $render_array['table']);
    $this->assertArrayHasKey('#rows', $render_array['table']);

    // Verify description is included.
    $this->assertArrayHasKey('#markup', $render_array);
  }

}
