<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentModel;

use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;
use Drupal\Core\Entity\Entity\EntityViewDisplay;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Comprehensive tests for XrayAuditEntityArchitecturePlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditEntityArchitecturePluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'link',
    'image',
    'file',
    'datetime',
    'options',
    'taxonomy',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The entity architecture plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentModel\XrayAuditEntityArchitecturePlugin
   */
  protected $plugin;

  /**
   * The entity architecture service.
   *
   * @var \Drupal\xray_audit\Services\EntityArchitectureInterface
   */
  protected $entityArchitectureService;

  /**
   * The plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install additional schemas for complex field testing.
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);
    $this->installConfig(['field', 'node', 'taxonomy', 'file', 'image']);

    // Get services.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');
    $this->entityArchitectureService = $this->container->get('xray_audit.entity_architecture');
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('entity_architecture');

    // Set up test content types and fields.
    $this->setUpTestContentTypes();
    $this->setUpTestFields();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    parent::tearDown();
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Sets up test content types.
   *
   * Creates multiple content types to test entity architecture analysis
   * across different bundles.
   */
  protected function setUpTestContentTypes(): void {
    // Create article content type.
    $this->createContentType('article', 'Article');

    // Create page content type.
    $this->createContentType('page', 'Page');

    // Create blog content type.
    $this->createContentType('blog', 'Blog');

    // Create taxonomy vocabulary for entity reference testing.
    $vocabulary = Vocabulary::create([
      'vid' => 'tags',
      'name' => 'Tags',
    ]);
    $vocabulary->save();
  }

  /**
   * Sets up test fields on content types.
   *
   * Creates various field types to comprehensively test entity architecture
   * analysis including text, link, entity reference, datetime, etc.
   */
  protected function setUpTestFields(): void {
    // TEXT FIELD: body field (simulating standard body field).
    $this->createTestField(
      'field_body_text',
      'node',
      'article',
      'text_long',
      [],
      ['required' => FALSE]
    );

    // LINK FIELD: external link field.
    $this->createTestField(
      'field_external_link',
      'node',
      'article',
      'link',
      [],
      ['required' => FALSE, 'title' => 1]
    );

    // ENTITY REFERENCE FIELD: taxonomy term reference.
    $this->createTestField(
      'field_tags',
      'node',
      'article',
      'entity_reference',
      [
        'target_type' => 'taxonomy_term',
        'cardinality' => -1,
      ],
      [
        'required' => FALSE,
        'settings' => [
          'handler' => 'default:taxonomy_term',
          'handler_settings' => [
            'target_bundles' => ['tags' => 'tags'],
          ],
        ],
      ]
    );

    // REQUIRED TEXT FIELD: to test mandatory flag.
    $this->createTestField(
      'field_required_text',
      'node',
      'article',
      'string',
      [],
      ['required' => TRUE]
    );

    // INTEGER FIELD: to test numeric field types.
    $this->createTestField(
      'field_integer',
      'node',
      'page',
      'integer',
      [],
      ['required' => FALSE]
    );

    // BOOLEAN FIELD: to test boolean field types.
    $this->createTestField(
      'field_published',
      'node',
      'page',
      'boolean',
      [],
      ['required' => FALSE]
    );

    // EMAIL FIELD: to test email field type.
    $this->createTestField(
      'field_email',
      'node',
      'blog',
      'email',
      [],
      ['required' => FALSE]
    );

    // DATETIME FIELD: to test datetime field type.
    $this->createTestField(
      'field_publish_date',
      'node',
      'blog',
      'datetime',
      ['datetime_type' => 'datetime'],
      ['required' => FALSE]
    );

    // Set up entity form displays with widgets.
    $this->setUpFormDisplays();

    // Set up entity view displays with formatters.
    $this->setUpViewDisplays();
  }

  /**
   * Sets up entity form displays with widgets.
   */
  protected function setUpFormDisplays(): void {
    // Article form display.
    $form_display = EntityFormDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'article',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $form_display->setComponent('field_body_text', [
      'type' => 'text_textarea',
      'weight' => 1,
    ]);
    $form_display->setComponent('field_external_link', [
      'type' => 'link_default',
      'weight' => 2,
    ]);
    $form_display->setComponent('field_tags', [
      'type' => 'entity_reference_autocomplete',
      'weight' => 3,
    ]);
    $form_display->setComponent('field_required_text', [
      'type' => 'string_textfield',
      'weight' => 4,
    ]);
    $form_display->save();

    // Page form display.
    $form_display_page = EntityFormDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'page',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $form_display_page->setComponent('field_integer', [
      'type' => 'number',
      'weight' => 1,
    ]);
    $form_display_page->setComponent('field_published', [
      'type' => 'boolean_checkbox',
      'weight' => 2,
    ]);
    $form_display_page->save();

    // Blog form display.
    $form_display_blog = EntityFormDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'blog',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $form_display_blog->setComponent('field_email', [
      'type' => 'email_default',
      'weight' => 1,
    ]);
    $form_display_blog->setComponent('field_publish_date', [
      'type' => 'datetime_default',
      'weight' => 2,
    ]);
    $form_display_blog->save();
  }

  /**
   * Sets up entity view displays with formatters.
   */
  protected function setUpViewDisplays(): void {
    // Article view display (default).
    $view_display = EntityViewDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'article',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $view_display->setComponent('field_body_text', [
      'type' => 'text_default',
      'weight' => 1,
      'label' => 'above',
    ]);
    $view_display->setComponent('field_external_link', [
      'type' => 'link',
      'weight' => 2,
      'label' => 'above',
    ]);
    $view_display->setComponent('field_tags', [
      'type' => 'entity_reference_label',
      'weight' => 3,
      'label' => 'above',
    ]);
    $view_display->setComponent('field_required_text', [
      'type' => 'string',
      'weight' => 4,
      'label' => 'above',
    ]);
    $view_display->save();

    // Article view display (teaser).
    $view_display_teaser = EntityViewDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'article',
      'mode' => 'teaser',
      'status' => TRUE,
    ]);
    $view_display_teaser->setComponent('field_body_text', [
      'type' => 'text_summary_or_trimmed',
      'weight' => 1,
      'label' => 'hidden',
    ]);
    $view_display_teaser->save();

    // Page view display (default).
    $view_display_page = EntityViewDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'page',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $view_display_page->setComponent('field_integer', [
      'type' => 'number_integer',
      'weight' => 1,
      'label' => 'above',
    ]);
    $view_display_page->setComponent('field_published', [
      'type' => 'boolean',
      'weight' => 2,
      'label' => 'above',
    ]);
    $view_display_page->save();

    // Blog view display (default).
    $view_display_blog = EntityViewDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'blog',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $view_display_blog->setComponent('field_email', [
      'type' => 'basic_string',
      'weight' => 1,
      'label' => 'above',
    ]);
    $view_display_blog->setComponent('field_publish_date', [
      'type' => 'datetime_default',
      'weight' => 2,
      'label' => 'above',
    ]);
    $view_display_blog->save();
  }

  /**
   * Test getDataOperationResult() with content_entity_definition operation.
   *
   */
  public function testGetDataOperationResultContentEntityDefinition() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Result should be an array.
    $this->assertIsArray($result);
    $this->assertNotEmpty($result, 'Should return entity architecture data');

    // Verify data structure contains expected elements.
    $found_article = FALSE;
    $found_page = FALSE;
    $found_blog = FALSE;

    foreach ($result as $item) {
      $this->assertArrayHasKey('content', $item);
      $content = $item['content'];

      // Check for our test content types.
      if (isset($content['entity']) && $content['entity'] === 'node') {
        if (isset($content['bundle']) && $content['bundle'] === 'article') {
          $found_article = TRUE;
        }
        if (isset($content['bundle']) && $content['bundle'] === 'page') {
          $found_page = TRUE;
        }
        if (isset($content['bundle']) && $content['bundle'] === 'blog') {
          $found_blog = TRUE;
        }
      }
    }

    $this->assertTrue($found_article, 'Should contain article content type');
    $this->assertTrue($found_page, 'Should contain page content type');
    $this->assertTrue($found_blog, 'Should contain blog content type');
  }

  /**
   * Test getDataOperationResult() returns proper data structure.
   *
   * Validates that each data item has the expected 'content' key structure.
   *
   */
  public function testGetDataOperationResultDataStructure() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Each item should have proper structure.
    $this->assertNotEmpty($result);

    foreach ($result as $item) {
      $this->assertArrayHasKey('content', $item);
      $content = $item['content'];

      // Required keys that should always exist.
      $this->assertArrayHasKey('entity', $content);
      $this->assertArrayHasKey('bundle', $content);
      $this->assertArrayHasKey('type', $content);
      $this->assertArrayHasKey('machine_name', $content);
      $this->assertArrayHasKey('label', $content);
    }
  }

  /**
   * Test getDataOperationResult() includes entity rows.
   *
   * Verifies that entity definition rows (type='entity') are included
   * before field rows.
   *
   */
  public function testGetDataOperationResultIncludesEntityRows() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Should contain entity rows with type='entity'.
    $found_entity_row = FALSE;

    foreach ($result as $item) {
      if (isset($item['content']['type']) && $item['content']['type'] === 'entity') {
        $found_entity_row = TRUE;
        // Verify entity row structure.
        $this->assertArrayHasKey('entity', $item['content']);
        $this->assertArrayHasKey('bundle', $item['content']);
        $this->assertArrayHasKey('machine_name', $item['content']);
        $this->assertArrayHasKey('label', $item['content']);
        break;
      }
    }

    $this->assertTrue($found_entity_row, 'Should contain entity definition rows');
  }

  /**
   * Test getDataOperationResult() includes field rows.
   *
   * Verifies that field definition rows (type='base_field', 'field_config', etc.)
   * are included with complete field metadata.
   *
   */
  public function testGetDataOperationResultIncludesFieldRows() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Should contain field rows.
    $found_field_row = FALSE;

    foreach ($result as $item) {
      $type = $item['content']['type'] ?? '';
      if (in_array($type, ['base_field', 'base_field_override', 'field_config'])) {
        $found_field_row = TRUE;
        // Verify field row has expected keys.
        $this->assertArrayHasKey('machine_name', $item['content']);
        $this->assertArrayHasKey('label', $item['content']);
        $this->assertArrayHasKey('data_type', $item['content']);
        break;
      }
    }

    $this->assertTrue($found_field_row, 'Should contain field definition rows');
  }

  /**
   * Test getDataOperationResult() includes custom fields.
   *
   * Verifies that custom fields created in setUp() are properly included
   * in the entity architecture data.
   *
   */
  public function testGetDataOperationResultIncludesCustomFields() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Should contain our custom test fields.
    $found_fields = [
      'field_body_text' => FALSE,
      'field_external_link' => FALSE,
      'field_tags' => FALSE,
      'field_required_text' => FALSE,
      'field_integer' => FALSE,
      'field_published' => FALSE,
      'field_email' => FALSE,
      'field_publish_date' => FALSE,
    ];

    foreach ($result as $item) {
      $machine_name = $item['content']['machine_name'] ?? '';
      if (isset($found_fields[$machine_name])) {
        $found_fields[$machine_name] = TRUE;
      }
    }

    foreach ($found_fields as $field_name => $found) {
      $this->assertTrue($found, "Should contain custom field: $field_name");
    }
  }

  /**
   * Test field data includes cardinality.
   *
   * Verifies that field cardinality (single vs unlimited) is captured.
   * Note: Cardinality is only available for base fields, not field configs.
   *
   */
  public function testFieldDataIncludesCardinality() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Find base fields with cardinality.
    // Base fields like 'title', 'status', etc. have cardinality.
    $found_cardinality = FALSE;

    foreach ($result as $item) {
      $type = $item['content']['type'] ?? '';
      if ($type === 'base_field' && isset($item['content']['cardinality'])) {
        $found_cardinality = TRUE;
        // Verify cardinality is a valid value.
        $cardinality = $item['content']['cardinality'];
        $this->assertTrue(
          is_numeric($cardinality) || $cardinality === '',
          'Cardinality should be numeric or empty'
        );
        break;
      }
    }

    $this->assertTrue($found_cardinality, 'Should find base fields with cardinality information');
  }

  /**
   * Test field data includes mandatory flag.
   *
   */
  public function testFieldDataIncludesMandatoryFlag() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: All fields should have mandatory flag.
    $found_field_with_mandatory = FALSE;

    foreach ($result as $item) {
      $machine_name = $item['content']['machine_name'] ?? '';
      $type = $item['content']['type'] ?? '';

      // Skip entity rows, only check field rows.
      if ($type !== 'entity' && strpos($machine_name, 'field_') === 0) {
        $this->assertArrayHasKey('mandatory', $item['content'], "Field $machine_name should have mandatory key");
        $this->assertContains(
          $item['content']['mandatory'],
          ['Yes', 'No'],
          "Field $machine_name mandatory flag should be Yes or No"
        );
        $found_field_with_mandatory = TRUE;
      }
    }

    $this->assertTrue($found_field_with_mandatory, 'Should find custom fields with mandatory flag');
  }

  /**
   * REFACTOR: Test field data includes computed flag.
   *
   */
  public function testFieldDataIncludesComputedFlag() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: All custom fields should have computed flag.
    foreach ($result as $item) {
      $machine_name = $item['content']['machine_name'] ?? '';
      if (strpos($machine_name, 'field_') === 0) {
        $this->assertArrayHasKey('computed', $item['content']);
        $this->assertContains($item['content']['computed'], ['Yes', 'No'], 'Computed flag should be Yes or No');
      }
    }
  }

  /**
   * Test field data includes translatable flag.
   *
   */
  public function testFieldDataIncludesTranslatableFlag() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Fields should have translatable flag.
    foreach ($result as $item) {
      $machine_name = $item['content']['machine_name'] ?? '';
      if (strpos($machine_name, 'field_') === 0) {
        $this->assertArrayHasKey('translatable', $item['content']);
        $this->assertContains($item['content']['translatable'], ['Yes', 'No'], 'Translatable flag should be Yes or No');
      }
    }
  }

  /**
   * Test field data includes read_only flag.
   *
   */
  public function testFieldDataIncludesReadOnlyFlag() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Fields should have read_only flag.
    foreach ($result as $item) {
      $machine_name = $item['content']['machine_name'] ?? '';
      if (strpos($machine_name, 'field_') === 0) {
        $this->assertArrayHasKey('read_only', $item['content']);
        $this->assertContains($item['content']['read_only'], ['Yes', 'No'], 'Read only flag should be Yes or No');
      }
    }
  }

  /**
   * Test field data includes revisionable flag.
   *
   */
  public function testFieldDataIncludesRevisionableFlag() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Fields should have revisionable flag.
    foreach ($result as $item) {
      $machine_name = $item['content']['machine_name'] ?? '';
      if (strpos($machine_name, 'field_') === 0) {
        $this->assertArrayHasKey('revisionable', $item['content']);
        $this->assertContains($item['content']['revisionable'], ['Yes', 'No'], 'Revisionable flag should be Yes or No');
      }
    }
  }

  /**
   * Test field data includes form widget information.
   *
   */
  public function testFieldDataIncludesFormWidget() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Find field with form widget (field_external_link).
    $found_widget = FALSE;

    foreach ($result as $item) {
      if (isset($item['content']['machine_name']) && $item['content']['machine_name'] === 'field_external_link') {
        $this->assertArrayHasKey('form_widget', $item['content']);
        $this->assertNotEmpty($item['content']['form_widget'], 'Field should have form widget information');
        $this->assertStringContainsString('link_default', $item['content']['form_widget'], 'Should include widget type');
        $found_widget = TRUE;
        break;
      }
    }

    $this->assertTrue($found_widget, 'Should find field with form widget');
  }

  /**
   * Test field data includes data type.
   *
   */
  public function testFieldDataIncludesDataType() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Verify specific field types.
    $field_types = [
      'field_body_text' => 'text_long',
      'field_external_link' => 'link',
      'field_tags' => 'entity_reference',
      'field_integer' => 'integer',
      'field_published' => 'boolean',
      'field_email' => 'email',
      'field_publish_date' => 'datetime',
    ];

    foreach ($result as $item) {
      $machine_name = $item['content']['machine_name'] ?? '';
      if (isset($field_types[$machine_name])) {
        $this->assertArrayHasKey('data_type', $item['content']);
        $this->assertEquals(
          $field_types[$machine_name],
          $item['content']['data_type'],
          "Field $machine_name should have correct data type"
        );
      }
    }
  }

  /**
   * Test entity reference field includes target type settings.
   *
   */
  public function testEntityReferenceFieldIncludesTargetType() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Find entity reference field (field_tags).
    $found_reference = FALSE;

    foreach ($result as $item) {
      if (isset($item['content']['machine_name']) && $item['content']['machine_name'] === 'field_tags') {
        $this->assertArrayHasKey('data_type_settings', $item['content']);
        $settings = $item['content']['data_type_settings'];
        $this->assertNotEmpty($settings, 'Entity reference field should have settings');
        $this->assertStringContainsString('taxonomy_term', $settings, 'Should include target type');
        $found_reference = TRUE;
        break;
      }
    }

    $this->assertTrue($found_reference, 'Should find entity reference field with target type');
  }

  /**
   * GREEN: Test buildDataRenderArray() includes table element.
   *
   */
  public function testBuildDataRenderArrayIncludesTable() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('content_entity_definition');

    // Act.
    $build = $this->plugin->buildDataRenderArray($data, 'content_entity_definition');

    // Assert.
    $this->assertArrayHasKey('table', $build);
    $this->assertEquals('table', $build['table']['#type']);
    $this->assertArrayHasKey('#header', $build['table']);
    $this->assertArrayHasKey('#rows', $build['table']);
    $this->assertIsArray($build['table']['#rows']);
  }

  /**
   * Test buildDataRenderArray() includes download link.
   *
   */
  public function testBuildDataRenderArrayIncludesDownloadLink() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('content_entity_definition');

    // Act.
    $build = $this->plugin->buildDataRenderArray($data, 'content_entity_definition');

    // Assert - Download button key is 'download_button' not 'download'.
    $this->assertArrayHasKey('download_button', $build);
    $this->assertEquals('link', $build['download_button']['#type']);
    $this->assertArrayHasKey('#url', $build['download_button']);
    $this->assertArrayHasKey('#title', $build['download_button']);
  }

  /**
   * Test buildDataRenderArray() includes filter element.
   *
   */
  public function testBuildDataRenderArrayIncludesFilter() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('content_entity_definition');

    // Act.
    $build = $this->plugin->buildDataRenderArray($data, 'content_entity_definition');

    // Assert.
    $this->assertArrayHasKey('filter', $build);
    $this->assertIsArray($build['filter']);
  }

  /**
   * Test buildDataRenderArray() table has correct headers.
   *
   */
  public function testBuildDataRenderArrayTableHeaders() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('content_entity_definition');

    // Act.
    $build = $this->plugin->buildDataRenderArray($data, 'content_entity_definition');

    // Assert: Verify expected headers.
    $headers = $build['table']['#header'];
    $this->assertNotEmpty($headers);

    // Expected header count (17 headers based on getHeaders() method).
    $this->assertCount(17, $headers);

    // Verify some key headers exist (they are TranslatableMarkup objects).
    $header_strings = array_map(function ($header) {
      return (string) $header;
    }, $headers);

    $this->assertContains('Entity', $header_strings);
    $this->assertContains('Bundle', $header_strings);
    $this->assertContains('Type', $header_strings);
    $this->assertContains('Machine name', $header_strings);
    $this->assertContains('Label', $header_strings);
    $this->assertContains('Data Type', $header_strings);
    $this->assertContains('Cardinality', $header_strings);
    $this->assertContains('Mandatory', $header_strings);
    $this->assertContains('Form widget', $header_strings);
  }

  /**
   * Test prepareCsvHeaders() returns correct headers.
   *
   */
  public function testPrepareCsvHeadersReturnsCorrectHeaders() {
    // Act.
    $headers = $this->invokeProtectedMethod($this->plugin, 'prepareCsvHeaders', ['content_entity_definition']);

    // Assert.
    $this->assertIsArray($headers);
    $this->assertCount(17, $headers, 'CSV should have 17 headers');

    // Convert TranslatableMarkup to strings for comparison.
    $header_strings = array_map(function ($header) {
      return (string) $header;
    }, $headers);

    $this->assertContains('Entity', $header_strings);
    $this->assertContains('Bundle', $header_strings);
    $this->assertContains('Machine name', $header_strings);
    $this->assertContains('Data Type', $header_strings);
    $this->assertContains('Cardinality', $header_strings);
    $this->assertContains('Mandatory', $header_strings);
  }

  /**
   * Test prepareCsvData() returns correct data structure.
   *
   */
  public function testPrepareCsvDataReturnsCorrectStructure() {
    // Arrange: Get some test data.
    $test_rows = [
      [
        'data' => [
          'node',
          'article',
          'field_config',
          'field_test',
          'Test Field',
          'Description',
          'string',
          'No',
          '',
          1,
          'No',
          'No',
          'Yes',
          'Yes',
          '',
          '',
          'type: string_textfield',
        ],
        'class' => [],
      ],
    ];

    // Act - prepareCsvData ignores the passed data array and calls getDataOperationResult() internally.
    $csv_data = $this->plugin->prepareCsvData('content_entity_definition', $test_rows);

    // Assert.
    $this->assertIsArray($csv_data);
    // Note: prepareCsvData calls getDataOperationResult() which returns actual entity data, not the test data.
    $this->assertNotEmpty($csv_data, 'CSV data should contain rows from actual entity definitions');
    if (!empty($csv_data)) {
      $this->assertCount(17, $csv_data[0], 'Each CSV row should have 17 columns');
    }
  }

  /**
   * Test prepareCsvData() with multiple rows.
   *
   */
  public function testPrepareCsvDataMultipleRows() {
    // Arrange: Get actual data.
    $data = $this->plugin->getDataOperationResult('content_entity_definition');
    $build = $this->plugin->buildDataRenderArray($data, 'content_entity_definition');
    $rows = $build['table']['#rows'];

    // Act - Pass the raw data, not the rendered rows.
    $csv_data = $this->invokeProtectedMethod($this->plugin, 'prepareCsvData', ['content_entity_definition', $data]);

    // Assert.
    $this->assertIsArray($csv_data);
    $this->assertNotEmpty($csv_data);
    $this->assertEquals(count($rows), count($csv_data), 'CSV data count should match row count');

    // Verify each row has correct column count.
    foreach ($csv_data as $row) {
      $this->assertCount(17, $row, 'Each CSV row should have 17 columns');
    }
  }

  /**
   * Test field with complex settings (entity reference).
   *
   * Entity reference fields should include handler and target bundle settings.
   *
   */
  public function testEntityReferenceFieldWithComplexSettings() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Find entity reference field with settings.
    $found_reference_settings = FALSE;

    foreach ($result as $item) {
      if (isset($item['content']['machine_name']) && $item['content']['machine_name'] === 'field_tags') {
        $settings = $item['content']['data_type_settings'] ?? '';
        $this->assertNotEmpty($settings, 'Entity reference should have settings');
        $this->assertStringContainsString('handler', $settings, 'Should include handler setting');
        $this->assertStringContainsString('taxonomy_term', $settings, 'Should include target type');
        $found_reference_settings = TRUE;
        break;
      }
    }

    $this->assertTrue($found_reference_settings, 'Should find entity reference with complex settings');
  }

  /**
   * Test field type classification (base_field vs field_config).
   *
   */
  public function testFieldTypeClassification() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Should have both base_field and field_config types.
    $found_base_field = FALSE;
    $found_field_config = FALSE;

    foreach ($result as $item) {
      $type = $item['content']['type'] ?? '';
      if ($type === 'base_field') {
        $found_base_field = TRUE;
      }
      if ($type === 'field_config') {
        $found_field_config = TRUE;
      }

      if ($found_base_field && $found_field_config) {
        break;
      }
    }

    $this->assertTrue($found_base_field, 'Should contain base field definitions');
    $this->assertTrue($found_field_config, 'Should contain field config definitions');
  }

  /**
   * Test base fields are included for all bundles.
   *
   * Base fields like 'title', 'status', etc. should be present.
   *
   */
  public function testBaseFieldsIncludedForAllBundles() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Should find common base fields.
    $found_title = FALSE;
    $found_status = FALSE;

    foreach ($result as $item) {
      $machine_name = $item['content']['machine_name'] ?? '';
      $entity = $item['content']['entity'] ?? '';

      if ($entity === 'node') {
        if ($machine_name === 'title') {
          $found_title = TRUE;
        }
        if ($machine_name === 'status') {
          $found_status = TRUE;
        }
      }

      if ($found_title && $found_status) {
        break;
      }
    }

    $this->assertTrue($found_title, 'Should include title base field');
    $this->assertTrue($found_status, 'Should include status base field');
  }

  /**
   * Test field descriptions are captured.
   *
   */
  public function testFieldDescriptionsAreCaptured() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Fields should have description key.
    foreach ($result as $item) {
      if (isset($item['content']['type']) && $item['content']['type'] !== 'entity') {
        $this->assertArrayHasKey('description', $item['content'], 'Fields should have description key');
      }
    }
  }

  /**
   * Test default value information is captured.
   *
   */
  public function testDefaultValueInformationCaptured() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Fields should have default_value and default_value_callback keys.
    foreach ($result as $item) {
      if (isset($item['content']['type']) && $item['content']['type'] !== 'entity') {
        $this->assertArrayHasKey('default_value', $item['content'], 'Fields should have default_value key');
        $this->assertArrayHasKey('default_value_callback', $item['content'], 'Fields should have default_value_callback key');
      }
    }
  }

  /**
   * Test data structure consistency across different field types.
   *
   * All fields should have the same keys in their content array.
   *
   */
  public function testDataStructureConsistencyAcrossFieldTypes() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Collect all keys from field rows.
    $field_keys_sets = [];

    foreach ($result as $item) {
      $type = $item['content']['type'] ?? '';
      if ($type !== 'entity' && isset($item['content']['machine_name'])) {
        $field_keys_sets[] = array_keys($item['content']);
      }
    }

    // All field rows should have same set of keys.
    if (!empty($field_keys_sets)) {
      $first_keys = $field_keys_sets[0];
      sort($first_keys);

      foreach ($field_keys_sets as $keys) {
        sort($keys);
        $this->assertEquals(
          $first_keys,
          $keys,
          'All fields should have consistent key structure'
        );
      }
    }
  }

  /**
   * Test render array weight settings.
   *
   */
  public function testRenderArrayWeightSettings() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('content_entity_definition');

    // Act.
    $build = $this->plugin->buildDataRenderArray($data, 'content_entity_definition');

    // Assert: Check weight properties.
    $this->assertArrayHasKey('#weight', $build['table']);
    $this->assertEquals(10, $build['table']['#weight']);

    $this->assertArrayHasKey('#weight', $build['filter']);
    $this->assertEquals(6, $build['filter']['#weight']);
  }

  /**
   * Test all field types have appropriate widget information.
   *
   */
  public function testAllFieldTypesHaveWidgetInformation() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Custom fields should have widget information.
    $custom_fields_with_widgets = [
      'field_body_text',
      'field_external_link',
      'field_tags',
      'field_required_text',
      'field_integer',
      'field_published',
      'field_email',
      'field_publish_date',
    ];

    foreach ($result as $item) {
      $machine_name = $item['content']['machine_name'] ?? '';
      if (in_array($machine_name, $custom_fields_with_widgets)) {
        $this->assertArrayHasKey('form_widget', $item['content']);
        $widget = $item['content']['form_widget'];
        $this->assertNotEmpty($widget, "Field $machine_name should have widget information");
        $this->assertNotEquals('-', $widget, "Field $machine_name should have actual widget configured");
      }
    }
  }

}
