<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentModel;

use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\taxonomy\Entity\Term;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Comprehensive tests for XrayAuditVocabularyPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditVocabularyPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'taxonomy',
    'text',
    'field',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The vocabulary plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentModel\XrayAuditVocabularyPlugin
   */
  protected $plugin;

  /**
   * Plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Test vocabularies created for testing.
   *
   * @var \Drupal\taxonomy\Entity\Vocabulary[]
   */
  protected $testVocabularies = [];

  /**
   * Test taxonomy terms created for testing.
   *
   * @var \Drupal\taxonomy\Entity\Term[]
   */
  protected $testTerms = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install required entity schemas for taxonomy.
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('taxonomy_vocabulary');
    $this->installConfig(['taxonomy', 'system']);

    // Get task plugin manager.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');

    // Get plugin repository service.
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');

    // Get entity type manager.
    $this->entityTypeManager = $this->container->get('entity_type.manager');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('vocabulary_content_model');

    // Create test vocabularies and terms.
    $this->createTestVocabularies();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clean up test terms first (to avoid constraint violations).
    foreach ($this->testTerms as $term) {
      if ($term) {
        $term->delete();
      }
    }

    // Clean up test vocabularies.
    foreach ($this->testVocabularies as $vocabulary) {
      if ($vocabulary) {
        $vocabulary->delete();
      }
    }

    // Clear all cache.
    $this->pluginRepository->clearAllCache();

    parent::tearDown();
  }

  /**
   * Creates test vocabularies for testing.
   */
  protected function createTestVocabularies(): void {
    // Create vocabulary with full description.
    $tags = Vocabulary::create([
      'vid' => 'tags',
      'name' => 'Tags',
      'description' => 'Free tagging vocabulary for content classification',
      'langcode' => 'en',
    ]);
    $tags->save();
    $this->testVocabularies['tags'] = $tags;

    // Create vocabulary without description.
    $categories = Vocabulary::create([
      'vid' => 'categories',
      'name' => 'Categories',
      'description' => '',
      'langcode' => 'en',
    ]);
    $categories->save();
    $this->testVocabularies['categories'] = $categories;

    // Create vocabulary with non-English language.
    $topics = Vocabulary::create([
      'vid' => 'topics',
      'name' => 'Topics',
      'description' => 'Main content topics',
      'langcode' => 'es',
    ]);
    $topics->save();
    $this->testVocabularies['topics'] = $topics;

    // Create vocabulary with long description.
    $locations = Vocabulary::create([
      'vid' => 'locations',
      'name' => 'Locations',
      'description' => 'Geographic locations including countries, states, cities, and specific addresses for content geo-tagging and location-based filtering',
      'langcode' => 'en',
    ]);
    $locations->save();
    $this->testVocabularies['locations'] = $locations;
  }

  /**
   * Creates test taxonomy terms for vocabularies.
   */
  protected function createTestTerms(): void {
    // Create terms for tags vocabulary.
    $term1 = Term::create([
      'vid' => 'tags',
      'name' => 'Drupal',
      'langcode' => 'en',
    ]);
    $term1->save();
    $this->testTerms[] = $term1;

    $term2 = Term::create([
      'vid' => 'tags',
      'name' => 'Testing',
      'langcode' => 'en',
    ]);
    $term2->save();
    $this->testTerms[] = $term2;

    // Create term for categories vocabulary.
    $term3 = Term::create([
      'vid' => 'categories',
      'name' => 'News',
      'langcode' => 'en',
    ]);
    $term3->save();
    $this->testTerms[] = $term3;
  }

  /**
   * Invokes a protected method using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name to invoke.
   * @param array $parameters
   *   Optional parameters to pass to the method.
   *
   * @return mixed
   *   The method return value.
   *
   * @throws \ReflectionException
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Tests getDataOperationResult() routes to vocabulariesDescription().
   *
   */
  public function testGetDataOperationResultRoutesVocabularyDescription(): void {
    $result = $this->plugin->getDataOperationResult('vocabulary_description');

    // Assert result is array.
    $this->assertIsArray($result, 'getDataOperationResult should return array.');

    // Assert result has expected keys.
    $this->assertArrayHasKey('header_table', $result, 'Result should have header_table key.');
    $this->assertArrayHasKey('results_table', $result, 'Result should have results_table key.');

    // Assert both are arrays.
    $this->assertIsArray($result['header_table'], 'header_table should be array.');
    $this->assertIsArray($result['results_table'], 'results_table should be array.');
  }

  /**
   * Tests vocabulariesDescription() returns correct header structure.
   *
   */
  public function testVocabulariesDescriptionHeaderStructure(): void {
    $result = $this->plugin->getDataOperationResult('vocabulary_description');

    $this->assertArrayHasKey('header_table', $result, 'Result should have header_table.');

    $header = $result['header_table'];
    $this->assertIsArray($header, 'Header should be array.');
    $this->assertCount(4, $header, 'Header should have exactly 4 columns.');

    // Verify header columns (as TranslatableMarkup objects, convert to string).
    $this->assertEquals('Id', (string) $header[0], 'First column should be Id.');
    $this->assertEquals('Label', (string) $header[1], 'Second column should be Label.');
    $this->assertEquals('Description', (string) $header[2], 'Third column should be Description.');
    $this->assertEquals('Langcode', (string) $header[3], 'Fourth column should be Langcode.');
  }

  /**
   * Tests vocabulariesDescription() returns all vocabularies.
   *
   */
  public function testVocabulariesDescriptionReturnsAllVocabularies(): void {
    $result = $this->plugin->getDataOperationResult('vocabulary_description');

    $this->assertArrayHasKey('results_table', $result, 'Result should have results_table.');

    $rows = $result['results_table'];
    $this->assertIsArray($rows, 'Results should be array.');
    $this->assertCount(4, $rows, 'Should return exactly 4 vocabularies.');

    // Extract vocabulary IDs from results.
    $vocabulary_ids = array_column($rows, 0);

    // Assert all test vocabularies are present.
    $this->assertContains('tags', $vocabulary_ids, 'Should include tags vocabulary.');
    $this->assertContains('categories', $vocabulary_ids, 'Should include categories vocabulary.');
    $this->assertContains('topics', $vocabulary_ids, 'Should include topics vocabulary.');
    $this->assertContains('locations', $vocabulary_ids, 'Should include locations vocabulary.');
  }

  /**
   * Tests vocabulariesDescription() row structure for each vocabulary.
   *
   */
  public function testVocabulariesDescriptionRowStructure(): void {
    $result = $this->plugin->getDataOperationResult('vocabulary_description');
    $rows = $result['results_table'];

    // Check each row has exactly 4 columns.
    foreach ($rows as $row) {
      $this->assertIsArray($row, 'Each row should be an array.');
      $this->assertCount(4, $row, 'Each row should have exactly 4 columns.');

      // Verify structure: [string, string, object/string, object].
      $this->assertIsString($row[0], 'First column (ID) should be string.');
      $this->assertIsString($row[1], 'Second column (Label) should be string.');
      // Description can be string or field object.
      $this->assertIsString($row[3]->value ?? $row[3], 'Fourth column (Langcode) should have value property.');
    }
  }

  /**
   * Tests vocabulariesDescription() returns correct data for tags vocabulary.
   *
   */
  public function testVocabulariesDescriptionTagsVocabularyData(): void {
    $result = $this->plugin->getDataOperationResult('vocabulary_description');
    $rows = $result['results_table'];

    // Find tags vocabulary row.
    $tags_row = NULL;
    foreach ($rows as $row) {
      if ($row[0] === 'tags') {
        $tags_row = $row;
        break;
      }
    }

    $this->assertNotNull($tags_row, 'Tags vocabulary should be in results.');

    // Verify tags vocabulary data.
    $this->assertEquals('tags', $tags_row[0], 'Tags vocabulary ID should be "tags".');
    $this->assertEquals('Tags', $tags_row[1], 'Tags vocabulary label should be "Tags".');
    $this->assertEquals('Free tagging vocabulary for content classification', (string) $tags_row[2], 'Tags vocabulary description should match.');

    // Langcode is a FieldItemList, access via value property.
    $langcode_value = is_object($tags_row[3]) && isset($tags_row[3]->value) ? $tags_row[3]->value : (string) $tags_row[3];
    $this->assertEquals('en', $langcode_value, 'Tags vocabulary langcode should be "en".');
  }

  /**
   * Tests vocabulariesDescription() handles long description text.
   *
   */
  public function testVocabulariesDescriptionLongDescription(): void {
    $result = $this->plugin->getDataOperationResult('vocabulary_description');
    $rows = $result['results_table'];

    // Find locations vocabulary row.
    $locations_row = NULL;
    foreach ($rows as $row) {
      if ($row[0] === 'locations') {
        $locations_row = $row;
        break;
      }
    }

    $this->assertNotNull($locations_row, 'Locations vocabulary should be in results.');

    // Verify long description is complete.
    $expected_description = 'Geographic locations including countries, states, cities, and specific addresses for content geo-tagging and location-based filtering';
    $this->assertEquals('locations', $locations_row[0], 'Locations vocabulary ID should be "locations".');
    $this->assertEquals('Locations', $locations_row[1], 'Locations vocabulary label should be "Locations".');
    $this->assertEquals($expected_description, (string) $locations_row[2], 'Locations vocabulary description should be complete.');
    $this->assertGreaterThan(50, strlen((string) $locations_row[2]), 'Long description should exceed 50 characters.');
  }

  /**
   * Tests vocabulariesDescription() with populated vocabularies.
   *
   */
  public function testVocabulariesDescriptionWithTerms(): void {
    // Create terms in vocabularies.
    $this->createTestTerms();

    // Get results with terms present.
    $result = $this->plugin->getDataOperationResult('vocabulary_description');
    $rows = $result['results_table'];

    // Should still return all 4 vocabularies.
    $this->assertCount(4, $rows, 'Should still return all 4 vocabularies.');

    // Verify structure is unchanged.
    foreach ($rows as $row) {
      $this->assertCount(4, $row, 'Each row should still have 4 columns.');
    }

    // Verify tags vocabulary is still present (has terms).
    $tags_row = NULL;
    foreach ($rows as $row) {
      if ($row[0] === 'tags') {
        $tags_row = $row;
        break;
      }
    }

    $this->assertNotNull($tags_row, 'Tags vocabulary should still be present.');
    $this->assertEquals('Tags', $tags_row[1], 'Tags vocabulary label should be unchanged.');
  }

  /**
   * Tests vocabulariesDescription() data structure is CSV-compatible.
   *
   */
  public function testVocabulariesDescriptionCsvCompatibility(): void {
    $result = $this->plugin->getDataOperationResult('vocabulary_description');

    // Verify structure is CSV-compatible (simple 2D array).
    $this->assertIsArray($result['header_table'], 'Header should be array.');
    $this->assertIsArray($result['results_table'], 'Results should be array.');

    // Verify header is flat array of strings.
    foreach ($result['header_table'] as $header_item) {
      // TranslatableMarkup can be cast to string.
      $this->assertIsString((string) $header_item, 'Each header item should be string-castable.');
    }

    // Verify each row is flat array.
    foreach ($result['results_table'] as $row) {
      $this->assertIsArray($row, 'Each row should be array.');
      $this->assertCount(4, $row, 'Each row should have exactly 4 columns.');

      // Verify each cell can be converted to string (CSV requirement).
      foreach ($row as $cell) {
        $this->assertIsString((string) $cell, 'Each cell should be string-castable for CSV export.');
      }
    }
  }

}
