<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\Layout;

use Drupal\Core\Render\RenderContext;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;
use Drupal\block\Entity\Block;

/**
 * Comprehensive tests for XrayAuditBlockLayoutPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditBlockLayoutPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'block',
    'path_alias',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The block layout plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\Layout\XrayAuditBlockLayoutPlugin
   */
  protected $plugin;

  /**
   * Test blocks.
   *
   * @var \Drupal\block\Entity\Block[]
   */
  protected $blocks;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('block');
    $this->installConfig(['system', 'block']);

    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');
    $this->plugin = $this->taskPluginManager->createInstance('block_layout');

    // Create test blocks.
    $this->createTestBlocks();
  }

  /**
   * Creates test blocks with various configurations.
   */
  protected function createTestBlocks() {
    // Block 1: Enabled block in default theme.
    $this->blocks['block1'] = Block::create([
      'id' => 'test_block_1',
      'theme' => 'stark',
      'region' => 'content',
      'plugin' => 'system_branding_block',
      'status' => TRUE,
      'weight' => 0,
      'settings' => [
        'label' => 'Test Block 1',
      ],
    ]);
    $this->blocks['block1']->save();

    // Block 2: Disabled block.
    $this->blocks['block2'] = Block::create([
      'id' => 'test_block_2',
      'theme' => 'stark',
      'region' => 'sidebar_first',
      'plugin' => 'system_powered_by_block',
      'status' => FALSE,
      'weight' => 0,
      'settings' => [
        'label' => 'Test Block 2',
      ],
    ]);
    $this->blocks['block2']->save();

    // Block 3: Block with visibility conditions.
    $this->blocks['block3'] = Block::create([
      'id' => 'test_block_3',
      'theme' => 'stark',
      'region' => 'footer',
      'plugin' => 'system_menu_block:tools',
      'status' => TRUE,
      'weight' => 0,
      'settings' => [
        'label' => 'Test Block 3',
      ],
      'visibility' => [
        'request_path' => [
          'id' => 'request_path',
          'pages' => '/admin/*',
          'negate' => FALSE,
        ],
      ],
    ]);
    $this->blocks['block3']->save();
  }

  /**
   * Helper to invoke protected methods.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Gets data wrapped in render context.
   */
  protected function getDataWithRenderContext(string $operation): array {
    $renderer = $this->container->get('renderer');
    $data = [];
    $renderer->executeInRenderContext(new RenderContext(), function () use ($operation, &$data) {
      $data = $this->plugin->getDataOperationResult($operation);
    });
    return $data;
  }

  /**
   * Tests getDataOperationResult() returns correct header.
   *
   */
  public function testGetDataOperationResultReturnsCorrectHeader() {
    $result = $this->getDataWithRenderContext('block_configuration');

    // Header: Id, Label, Theme, Region, Status, Visibility Conditions, Link.
    $this->assertCount(7, $result['header_table']);
  }

  /**
   * Tests getDataOperationResult() includes all blocks.
   *
   */
  public function testGetDataOperationResultIncludesAllBlocks() {
    $result = $this->getDataWithRenderContext('block_configuration');

    $this->assertIsArray($result['results_table']);
    $this->assertGreaterThanOrEqual(3, count($result['results_table']));

    $block_ids = array_keys($result['results_table']);
    $this->assertContains('test_block_1', $block_ids);
    $this->assertContains('test_block_2', $block_ids);
    $this->assertContains('test_block_3', $block_ids);
  }

  /**
   * Tests getDataOperationResult() formats block data correctly.
   *
   */
  public function testGetDataOperationResultFormatsBlockDataCorrectly() {
    $result = $this->getDataWithRenderContext('block_configuration');

    foreach ($result['results_table'] as $block_data) {
      $this->assertArrayHasKey('id', $block_data);
      $this->assertArrayHasKey('title', $block_data);
      $this->assertArrayHasKey('theme', $block_data);
      $this->assertArrayHasKey('region', $block_data);
      $this->assertArrayHasKey('status', $block_data);
      $this->assertArrayHasKey('visibility', $block_data);
      $this->assertArrayHasKey('link', $block_data);
    }
  }

  /**
   * Tests getDataOperationResult() shows block status.
   *
   */
  public function testGetDataOperationResultHandlesEnabledDisabledStatus() {
    $result = $this->getDataWithRenderContext('block_configuration');

    // Verify that status field is present and contains valid values.
    $this->assertNotEmpty($result['results_table']);

    foreach ($result['results_table'] as $block_data) {
      $this->assertArrayHasKey('status', $block_data);
      $status = (string) $block_data['status'];
      // Status should be either Enabled or Disabled.
      $this->assertTrue(
        stripos($status, 'Enabled') !== FALSE || stripos($status, 'Disabled') !== FALSE,
        'Status should be either Enabled or Disabled, got: ' . $status
      );
    }
  }

  /**
   * Tests getDataOperationResult() labels default and admin themes.
   *
   */
  public function testGetDataOperationResultLabelsDefaultAndAdminThemes() {
    // Set stark as default theme.
    $this->config('system.theme')->set('default', 'stark')->save();

    $result = $this->getDataWithRenderContext('block_configuration');

    $block_data = $result['results_table']['test_block_1'];
    $this->assertStringContainsString('(default)', $block_data['theme']);
  }

  /**
   * Tests formatVisibilityConditions() creates proper render array.
   *
   */
  public function testFormatVisibilityConditionsCreatesProperRenderArray() {
    $block = $this->blocks['block3'];
    $conditions = $block->getVisibilityConditions();

    $result = $this->invokeProtectedMethod($this->plugin, 'formatVisibilityConditions', [$conditions]);

    $this->assertIsArray($result);
    $this->assertEquals('item_list', $result['#theme']);
    $this->assertArrayHasKey('#items', $result);
    $this->assertIsArray($result['#items']);
  }

  /**
   * Tests formatVisibilityConditions() handles block with no conditions.
   */
  public function testFormatVisibilityConditionsHandlesNoConditions() {
    $block = $this->blocks['block1'];
    $conditions = $block->getVisibilityConditions();

    $result = $this->invokeProtectedMethod($this->plugin, 'formatVisibilityConditions', [$conditions]);

    $this->assertIsArray($result);
    $this->assertCount(1, $result['#items']);
    $this->assertStringContainsString('None', (string) $result['#items'][0]);
  }

  /**
   * Tests getConditionSummaries() returns condition details.
   */
  public function testGetConditionSummariesReturnsConditionDetails() {
    $block = $this->blocks['block3'];
    $conditions = $block->getVisibilityConditions();

    $result = $this->invokeProtectedMethod($this->plugin, 'getConditionSummaries', [$conditions]);

    $this->assertIsArray($result);
    $this->assertArrayHasKey('request_path', $result);
  }

  /**
   * Tests getHeaders() returns correct headers.
   */
  public function testGetHeadersReturnsCorrectHeaders() {
    $result = $this->invokeProtectedMethod($this->plugin, 'getHeaders');

    $this->assertIsArray($result);
    $this->assertArrayHasKey('id', $result);
    $this->assertArrayHasKey('label', $result);
    $this->assertArrayHasKey('theme', $result);
    $this->assertArrayHasKey('region', $result);
    $this->assertArrayHasKey('status', $result);
    $this->assertArrayHasKey('visibility_conditions', $result);
    $this->assertArrayHasKey('link', $result);
  }

  /**
   * Tests prepareCsvHeaders() removes link column.
   */
  public function testPrepareCsvHeadersRemovesLinkColumn() {
    $result = $this->invokeProtectedMethod($this->plugin, 'prepareCsvHeaders', ['block_configuration']);

    $this->assertIsArray($result);
    $this->assertArrayNotHasKey('link', $result);
    $this->assertArrayHasKey('id', $result);
    $this->assertArrayHasKey('visibility_conditions', $result);
  }

  /**
   * Tests prepareCsvData() formats data for CSV export.
   *
   */
  public function testPrepareCsvDataFormatsDataForCsvExport() {
    $table_content = [
      'test_block_1' => [
        'id' => 'test_block_1',
        'title' => 'Test Block 1',
        'link' => 'link_object',
        'visibility' => 'rendered_visibility',
        'plane_visibility' => ['condition1', 'condition2'],
        'status' => 'Enabled',
      ],
    ];

    $result = $this->invokeProtectedMethod($this->plugin, 'prepareCsvData', ['block_configuration', $table_content]);

    $this->assertIsArray($result);
    $this->assertArrayHasKey('test_block_1', $result);
    $this->assertArrayNotHasKey('link', $result['test_block_1']);
    $this->assertArrayNotHasKey('plane_visibility', $result['test_block_1']);
    $this->assertStringContainsString('condition1', $result['test_block_1']['visibility']);
    $this->assertEquals('1', $result['test_block_1']['status']);
  }

  /**
   * Tests buildDataRenderArray() adds CSV download button.
   *
   */
  public function testBuildDataRenderArrayAddsCsvDownloadLink() {
    $data = $this->getDataWithRenderContext('block_configuration');
    $render = $this->plugin->buildDataRenderArray($data, 'block_configuration');

    $this->assertIsArray($render);
    $this->assertArrayHasKey('download_button', $render);
    $this->assertEquals(9, $render['download_button']['#weight']);
  }

}
