<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\Packages;

use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Comprehensive tests for XrayAuditModulesPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditModulesPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The modules plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\Packages\XrayAuditModulesPlugin
   */
  protected $plugin;

  /**
   * The plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install system configuration.
    $this->installConfig(['system']);

    // Get services.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('modules');
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clear cache between tests to ensure clean state.
    if ($this->pluginRepository) {
      $cid = 'modules:modules_report';
      $this->container->get('cache.default')->delete($cid);
    }
    parent::tearDown();
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Helper method to create a mock Extension object.
   *
   * @param string $name
   *   Module machine name.
   * @param string $path
   *   Module path.
   * @param array $info
   *   Module info array.
   * @param string|null $origin
   *   Module origin ('core' or null).
   *
   * @return \Drupal\Core\Extension\Extension
   *   Mock Extension object.
   */
  protected function createMockExtension(string $name, string $path, array $info, ?string $origin = NULL) {
    $extension = $this->getMockBuilder('Drupal\Core\Extension\Extension')
      ->disableOriginalConstructor()
      ->getMock();

    $extension->method('getPath')->willReturn($path);
    $extension->info = $info;
    if ($origin !== NULL) {
      $extension->origin = $origin;
    }

    return $extension;
  }

  /**
   * Tests getDataOperationResult() with 'all_package' operation.
   */
  public function testGetDataOperationResultAllPackage() {
    // Act.
    $result = $this->plugin->getDataOperationResult('all_package');

    // Assert.
    $this->assertIsArray($result);

    // Check expected groups exist (at minimum, 'core' should exist).
    if (!empty($result)) {
      // Validate structure of returned data.
      foreach ($result as $group => $group_data) {
        $this->assertIsArray($group_data);
        $this->assertArrayHasKey('number_installed_modules', $group_data);
        $this->assertArrayHasKey('number_enabled_modules', $group_data);
        $this->assertArrayHasKey('modules', $group_data);
        $this->assertIsArray($group_data['modules']);
      }
    }
  }

  /**
   * Tests getDataOperationResult() with 'contrib_package' operation.
   */
  public function testGetDataOperationResultContribPackage() {
    // Act.
    $result = $this->plugin->getDataOperationResult('contrib_package');

    // Assert: Same data should be returned for both operations.
    $this->assertIsArray($result);

    // Verify structure matches all_package (same underlying data).
    if (!empty($result)) {
      foreach ($result as $group => $group_data) {
        $this->assertIsArray($group_data);
        $this->assertArrayHasKey('number_installed_modules', $group_data);
        $this->assertArrayHasKey('number_enabled_modules', $group_data);
        $this->assertArrayHasKey('modules', $group_data);
      }
    }
  }

  /**
   * Tests buildDataRenderArray() with 'all_package' operation.
   */
  public function testBuildDataRenderArrayAllPackage() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('all_package');

    // Act.
    $build = $this->plugin->buildDataRenderArray($data, 'all_package');

    // Assert.
    $this->assertIsArray($build);

    if (!empty($data)) {
      // Should have summary element.
      $this->assertArrayHasKey('summary', $build);

      // Should have download button (new CSV refactoring).
      $this->assertArrayHasKey('download_button', $build);
      $this->assertEquals('link', $build['download_button']['#type'] ?? '');

      // Should have group elements (core, contrib, custom, etc.).
      foreach (['core', 'contrib', 'custom', 'profile', 'other'] as $group) {
        if (isset($data[$group])) {
          $this->assertArrayHasKey($group, $build);
          $this->assertEquals('details', $build[$group]['#theme'] ?? '');
        }
      }
    }
  }

  /**
   * Tests buildDataRenderArray() with 'contrib_package' operation.
   */
  public function testBuildDataRenderArrayContribPackage() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('contrib_package');

    // Act.
    $build = $this->plugin->buildDataRenderArray($data, 'contrib_package');

    // Assert.
    $this->assertIsArray($build);

    if (!empty($data) && isset($data['contrib'])) {
      // Should have download button (new CSV refactoring).
      $this->assertArrayHasKey('download_button', $build);
      $this->assertEquals('link', $build['download_button']['#type'] ?? '');

      // Should have subgroup elements (root, site, profile, unknown).
      $possible_subgroups = ['root', 'site', 'profile', 'unknown'];
      $has_subgroup = FALSE;
      foreach ($possible_subgroups as $subgroup) {
        if (isset($build[$subgroup])) {
          $has_subgroup = TRUE;
          $this->assertEquals('details', $build[$subgroup]['#theme'] ?? '');
        }
      }
    }
  }

  /**
   * Tests buildDataRenderAllModulesReport() structure.
   */
  public function testBuildDataRenderAllModulesReportStructure() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('all_package');

    // Act.
    $build = $this->plugin->buildDataRenderAllModulesReport($data);

    // Assert.
    $this->assertIsArray($build);

    if (!empty($data)) {
      // Verify each group has proper render structure.
      foreach ($data as $type => $group_data) {
        $this->assertArrayHasKey($type, $build);
        $this->assertEquals('details', $build[$type]['#theme'] ?? '');
        $this->assertArrayHasKey('#title', $build[$type]);
        $this->assertArrayHasKey('#children', $build[$type]);
        $this->assertArrayHasKey('tables', $build[$type]['#children']);
        $this->assertEquals('table', $build[$type]['#children']['tables']['#theme'] ?? '');
      }
    }
  }

  /**
   * Tests buildDataRenderAllModulesReport() includes summary.
   */
  public function testBuildDataRenderAllModulesReportHasSummary() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('all_package');

    // Act.
    $build = $this->plugin->buildDataRenderAllModulesReport($data);

    // Assert.
    if (!empty($data)) {
      $this->assertArrayHasKey('summary', $build);
      $this->assertEquals('details', $build['summary']['#theme'] ?? '');
      $this->assertArrayHasKey('#children', $build['summary']);
      $this->assertArrayHasKey('tables', $build['summary']['#children']);

      // Verify summary has expected structure.
      $summary_table = $build['summary']['#children']['tables'];
      $this->assertEquals('table', $summary_table['#theme'] ?? '');
      $this->assertArrayHasKey('#header', $summary_table);
      $this->assertArrayHasKey('#rows', $summary_table);
      $this->assertIsArray($summary_table['#rows']);
    }
  }

  /**
   * Tests buildDataRenderAllModulesReport() includes download button.
   */
  public function testBuildDataRenderAllModulesReportHasDownloadLink() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('all_package');

    // Act.
    $build = $this->plugin->buildDataRenderAllModulesReport($data);

    // Assert.
    if (!empty($data)) {
      $this->assertArrayHasKey('download_button', $build);
      $this->assertEquals('link', $build['download_button']['#type'] ?? '');
      $this->assertArrayHasKey('#url', $build['download_button']);
      $this->assertArrayHasKey('#title', $build['download_button']);
      // The download button render array structure is defined by createRenderableLink().
      // It should have basic link properties.
    }
  }

  /**
   * Tests buildDataRenderAllModulesReport() groups modules correctly.
   */
  public function testBuildDataRenderAllModulesReportGroupsCoreContribCustom() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('all_package');

    // Act.
    $build = $this->plugin->buildDataRenderAllModulesReport($data);

    // Assert: Check expected groups are present.
    if (!empty($data)) {
      $expected_groups = ['core', 'contrib', 'custom', 'profile', 'other'];
      foreach ($expected_groups as $group) {
        if (isset($data[$group])) {
          $this->assertArrayHasKey($group, $build);
        }
      }

      // Core modules should always exist in Drupal.
      $this->assertArrayHasKey('core', $build);
    }
  }

  /**
   * Tests buildDataRenderContribModulesReport() structure.
   */
  public function testBuildDataRenderContribModulesReportStructure() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('contrib_package');

    // Act.
    $build = $this->plugin->buildDataRenderContribModulesReport($data);

    // Assert.
    $this->assertIsArray($build);

    if (!empty($data) && isset($data['contrib'])) {
      // Should have download button (new CSV refactoring).
      $this->assertArrayHasKey('download_button', $build);

      // Check subgroup structures.
      $possible_subgroups = ['root', 'site', 'profile', 'unknown'];
      foreach ($possible_subgroups as $subgroup) {
        if (isset($build[$subgroup])) {
          $this->assertEquals('details', $build[$subgroup]['#theme'] ?? '');
          $this->assertArrayHasKey('#title', $build[$subgroup]);
          $this->assertArrayHasKey('#children', $build[$subgroup]);
          $this->assertArrayHasKey('tables', $build[$subgroup]['#children']);
        }
      }
    }
  }

  /**
   * Tests buildDataRenderContribModulesReport() groups by subgroup.
   */
  public function testBuildDataRenderContribModulesReportGroupsBySubgroup() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('contrib_package');

    // Act.
    $build = $this->plugin->buildDataRenderContribModulesReport($data);

    // Assert.
    if (!empty($data) && isset($data['contrib'])) {
      // Verify modules are grouped by subgroup (root, site, profile, unknown).
      $expected_subgroups = ['root', 'site', 'profile', 'unknown'];
      $found_subgroup = FALSE;

      foreach ($expected_subgroups as $subgroup) {
        if (isset($build[$subgroup])) {
          $found_subgroup = TRUE;
          $this->assertArrayHasKey('#children', $build[$subgroup]);
          $this->assertArrayHasKey('tables', $build[$subgroup]['#children']);
          $table = $build[$subgroup]['#children']['tables'];
          $this->assertEquals('table', $table['#theme'] ?? '');
          $this->assertArrayHasKey('#rows', $table);
        }
      }
    }
  }

  /**
   * Tests buildDataRenderContribModulesReport() includes download button.
   */
  public function testBuildDataRenderContribModulesReportHasDownloadLink() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('contrib_package');

    // Act.
    $build = $this->plugin->buildDataRenderContribModulesReport($data);

    // Assert.
    $this->assertArrayHasKey('download_button', $build);
    $this->assertEquals('link', $build['download_button']['#type'] ?? '');
    $this->assertArrayHasKey('#url', $build['download_button']);
    $this->assertArrayHasKey('#title', $build['download_button']);
  }

  /**
   * Tests prepareCsvHeaders() returns proper headers.
   */
  public function testPrepareCsvHeadersAllPackage() {
    // Arrange.
    $operation = 'all_package';

    // Act.
    $headers = $this->plugin->prepareCsvHeaders($operation);

    // Assert.
    $this->assertIsArray($headers);
    $this->assertNotEmpty($headers);

    // Headers can be TranslatableMarkup or strings - both are valid.
    // The CSV download manager will handle conversion to strings.
    $this->assertCount(10, $headers, 'all_package operation should have 10 headers');
  }

  /**
   * Tests prepareCsvData() returns flattened data for CSV export.
   */
  public function testPrepareCsvDataAllPackage() {
    // Arrange.
    $operation = 'all_package';

    // Act.
    $csv_data = $this->plugin->prepareCsvData($operation, []);

    // Assert.
    $this->assertIsArray($csv_data);

    // If we have data, verify it's properly formatted for CSV.
    if (!empty($csv_data)) {
      foreach ($csv_data as $row) {
        $this->assertIsArray($row);
        // Each row should be an array of scalar values suitable for CSV.
        foreach ($row as $value) {
          $this->assertTrue(is_scalar($value) || is_null($value));
        }
      }
    }
  }

}
