<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\Packages;

use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Comprehensive tests for XrayAuditThemesPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditThemesPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The themes plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\Packages\XrayAuditThemesPlugin
   */
  protected $plugin;

  /**
   * The theme handler service.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * The plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install system configuration.
    $this->installConfig(['system']);

    // Get services.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');
    $this->themeHandler = $this->container->get('theme_handler');
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('themes');
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Tests getDataOperationResult() with 'themes' operation returns array.
   *
   * The returned array should have 'core' and 'not_core' groups containing
   * theme data. Unlike ModulesPlugin, this method does NOT use caching.
   */
  public function testGetDataOperationResultThemesOperation() {
    // Act.
    $result = $this->plugin->getDataOperationResult('themes');

    // Assert.
    $this->assertIsArray($result);

    // Check expected groups exist.
    // Note: In minimal test environment, at least core themes should exist.
    if (!empty($result)) {
      // Validate structure of returned data.
      $expected_groups = ['core', 'not_core'];
      foreach ($result as $group => $themes) {
        $this->assertContains($group, $expected_groups, "Group should be 'core' or 'not_core'");
        $this->assertIsArray($themes);

        // Validate each theme's data structure.
        foreach ($themes as $machine_name => $theme_data) {
          $this->assertIsArray($theme_data);
          $this->assertArrayHasKey('machine_name', $theme_data);
          $this->assertArrayHasKey('project', $theme_data);
          $this->assertArrayHasKey('Theme', $theme_data);
          $this->assertArrayHasKey('Default', $theme_data);
          $this->assertArrayHasKey('version', $theme_data);
          $this->assertArrayHasKey('group', $theme_data);

          // Verify group matches.
          $this->assertEquals($group, $theme_data['group']);
        }
      }
    }
  }

  /**
   * Tests buildDataRenderArray() with 'themes' operation returns render array.
   *
   * The render array should include:
   * - 'core' details element
   * - 'not_core' details element
   * - 'download' link element.
   *
   */
  public function testBuildDataRenderArrayThemesOperation() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('themes');

    // Act.
    $build = $this->plugin->buildDataRenderArray($data, 'themes');

    // Assert.
    $this->assertIsArray($build);

    if (!empty($data)) {
      // Should have 'core' and 'not_core' groups.
      if (isset($data['core'])) {
        $this->assertArrayHasKey('core', $build);
        $this->assertEquals('details', $build['core']['#theme'] ?? '');
      }

      if (isset($data['not_core'])) {
        $this->assertArrayHasKey('not_core', $build);
        $this->assertEquals('details', $build['not_core']['#theme'] ?? '');
      }

      // Should have download link.
      $this->assertArrayHasKey('download_button', $build);
      $this->assertEquals('link', $build['download_button']['#type'] ?? '');
    }
  }

  /**
   * Tests buildDataRenderArray() structure has required elements.
   *
   * Verifies:
   * - Details elements have #theme, #title, #description, #children
   * - Tables have #theme, #header, #rows
   * - Download link has proper structure.
   *
   */
  public function testBuildDataRenderArrayStructure() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('themes');

    // Act.
    $build = $this->plugin->buildDataRenderArray($data, 'themes');

    // Assert.
    if (!empty($data)) {
      // Check details structure for each group.
      foreach (['core', 'not_core'] as $group) {
        if (isset($build[$group])) {
          $this->assertEquals('details', $build[$group]['#theme']);
          $this->assertArrayHasKey('#title', $build[$group]);
          $this->assertArrayHasKey('#description', $build[$group]);
          $this->assertArrayHasKey('#attributes', $build[$group]);
          $this->assertArrayHasKey('#children', $build[$group]);

          // Check table structure.
          $this->assertArrayHasKey('tables', $build[$group]['#children']);
          $table = $build[$group]['#children']['tables'];
          $this->assertEquals('table', $table['#theme']);
          $this->assertArrayHasKey('#header', $table);
          $this->assertArrayHasKey('#rows', $table);
          $this->assertIsArray($table['#rows']);

          // Verify header has 5 columns:
          // Machine name, Project, Theme, Used, Version.
          $this->assertCount(5, $table['#header']);
        }
      }

      // Check download link structure.
      $this->assertArrayHasKey('download_button', $build);
      $this->assertEquals('link', $build['download_button']['#type']);
      $this->assertArrayHasKey('#url', $build['download_button']);
      $this->assertArrayHasKey('#title', $build['download_button']);
      $this->assertArrayHasKey('#attributes', $build['download_button']);
    }
  }

  /**
   * Tests getThemeStatus() returns array with proper structure.
   *
   * Protected method that retrieves and structures theme data.
   * Should return array grouped by 'core' and 'not_core'.
   *
   */
  public function testGetThemeStatusReturnsArray() {
    // Act: Call protected method via reflection.
    $result = $this->invokeProtectedMethod($this->plugin, 'getThemeStatus');

    // Assert.
    $this->assertIsArray($result, 'getThemeStatus() should return an array');

    // Verify structure if data exists.
    if (!empty($result)) {
      // Should have 'core' and/or 'not_core' keys.
      $has_valid_groups = isset($result['core']) || isset($result['not_core']);
      $this->assertTrue($has_valid_groups, 'Result should have core or not_core groups');

      // Validate theme data structure.
      foreach ($result as $group => $themes) {
        $this->assertContains($group, ['core', 'not_core']);
        $this->assertIsArray($themes);

        foreach ($themes as $machine_name => $theme_data) {
          $this->assertArrayHasKey('machine_name', $theme_data);
          $this->assertArrayHasKey('project', $theme_data);
          $this->assertArrayHasKey('Theme', $theme_data);
          $this->assertArrayHasKey('Default', $theme_data);
          $this->assertArrayHasKey('version', $theme_data);
          $this->assertArrayHasKey('group', $theme_data);
        }
      }
    }
  }

  /**
   * Tests getThemeStatus() groups themes into 'core' and 'not_core'.
   *
   * Core themes: package contains 'core' (case-insensitive).
   * Not core themes: all others.
   *
   */
  public function testGetThemeStatusGroupsCoreAndNotCore() {
    // Act.
    $result = $this->invokeProtectedMethod($this->plugin, 'getThemeStatus');

    // Assert.
    if (!empty($result)) {
      // Verify only valid group keys exist.
      foreach (array_keys($result) as $group) {
        $this->assertContains(
          $group,
          ['core', 'not_core'],
          'Themes should only be grouped as core or not_core'
        );
      }

      // If core group exists, verify themes have 'core' in package.
      if (isset($result['core'])) {
        $this->assertIsArray($result['core']);
        foreach ($result['core'] as $theme_data) {
          $this->assertEquals('core', $theme_data['group']);
        }
      }

      // If not_core group exists, verify themes don't have 'core' in package.
      if (isset($result['not_core'])) {
        $this->assertIsArray($result['not_core']);
        foreach ($result['not_core'] as $theme_data) {
          $this->assertEquals('not_core', $theme_data['group']);
        }
      }
    }
  }

  /**
   * Tests getThemeStatus() identifies default frontend theme.
   *
   * The default theme should have 'Default' = 'Yes'.
   * Identified via $themeHandler->getDefault().
   *
   */
  public function testGetThemeStatusIdentifiesDefaultTheme() {
    // Arrange: Get current default theme.
    $default_theme_name = $this->themeHandler->getDefault();

    // Act.
    $result = $this->invokeProtectedMethod($this->plugin, 'getThemeStatus');

    // Assert.
    if (!empty($result) && !empty($default_theme_name)) {
      $found_default = FALSE;

      // Search through all themes to find the default.
      foreach ($result as $group => $themes) {
        foreach ($themes as $theme_data) {
          if ($theme_data['Theme'] == $default_theme_name) {
            $found_default = TRUE;
            // Verify it's marked as default.
            $this->assertEquals(
              'Yes',
              (string) $theme_data['Default'],
              "Default theme '{$default_theme_name}' should be marked as 'Yes'"
            );
          }
        }
      }

      $this->assertTrue($found_default, 'Default theme should be present in results');
    }
  }

  /**
   * Tests getThemeStatus() identifies admin theme.
   *
   * The admin theme should have 'Default' = 'Yes'.
   * Identified via config 'system.theme'.admin.
   *
   */
  public function testGetThemeStatusIdentifiesAdminTheme() {
    // Arrange: Get current admin theme from config.
    $admin_theme_machine = $this->container
      ->get('config.factory')
      ->get('system.theme')
      ->get('admin');

    // Act.
    $result = $this->invokeProtectedMethod($this->plugin, 'getThemeStatus');

    // Assert.
    if (!empty($result) && !empty($admin_theme_machine)) {
      $admin_theme_name = $this->themeHandler->getName($admin_theme_machine);
      $found_admin = FALSE;

      // Search through all themes to find the admin theme.
      foreach ($result as $group => $themes) {
        foreach ($themes as $theme_data) {
          if ($theme_data['Theme'] == $admin_theme_name) {
            $found_admin = TRUE;
            // Verify it's marked as default (admin themes show as 'Yes').
            $this->assertEquals(
              'Yes',
              (string) $theme_data['Default'],
              "Admin theme '{$admin_theme_name}' should be marked as 'Yes'"
            );
          }
        }
      }

      $this->assertTrue($found_admin, 'Admin theme should be present in results');
    }
  }

  /**
   * Tests theme data structure has all required fields.
   *
   * Each theme should have:
   * - machine_name
   * - project
   * - Theme (display name)
   * - Default (Yes/No)
   * - version
   * - group.
   *
   */
  public function testThemeDataStructureHasRequiredFields() {
    // Act.
    $data = $this->plugin->getDataOperationResult('themes');

    // Assert.
    if (!empty($data)) {
      foreach ($data as $group => $themes) {
        foreach ($themes as $machine_name => $theme_data) {
          $required_fields = [
            'machine_name',
            'project',
            'Theme',
            'Default',
            'version',
            'group',
          ];

          foreach ($required_fields as $field) {
            $this->assertArrayHasKey(
              $field,
              $theme_data,
              "Theme data should contain '{$field}' field"
            );
          }

          // Verify data types.
          $this->assertIsString($theme_data['machine_name']);
          $this->assertIsString($theme_data['Theme']);
          $this->assertContains($theme_data['group'], ['core', 'not_core']);
        }
      }
    }
  }

  /**
   * Tests theme grouping correctly separates core vs not_core.
   *
   * Verifies:
   * - Core themes have package containing 'core' (case-insensitive)
   * - Not core themes do not.
   *
   */
  public function testThemeGroupingCoreVsNotCore() {
    // Act.
    $data = $this->plugin->getDataOperationResult('themes');

    // Assert.
    if (!empty($data)) {
      // Verify only valid groups exist.
      $valid_groups = ['core', 'not_core'];
      foreach (array_keys($data) as $group) {
        $this->assertContains($group, $valid_groups);
      }

      // Check group field matches array key.
      foreach ($data as $group => $themes) {
        foreach ($themes as $theme_data) {
          $this->assertEquals(
            $group,
            $theme_data['group'],
            "Theme group field should match array grouping"
          );
        }
      }
    }
  }

  /**
   * Tests render array has details elements for theme groups.
   *
   * Each group (core, not_core) should render as a details element
   * with title, description, and table.
   *
   */
  public function testRenderArrayHasDetailsElements() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('themes');

    // Act.
    $build = $this->plugin->buildDataRenderArray($data, 'themes');

    // Assert.
    if (!empty($data)) {
      foreach (['core', 'not_core'] as $group) {
        if (isset($build[$group])) {
          $this->assertEquals(
            'details',
            $build[$group]['#theme'],
            "{$group} should use 'details' theme"
          );

          $this->assertArrayHasKey('#title', $build[$group]);
          $this->assertArrayHasKey('#description', $build[$group]);
          $this->assertArrayHasKey('#attributes', $build[$group]);
          $this->assertArrayHasKey('#children', $build[$group]);

          // Verify attributes have proper classes.
          $this->assertArrayHasKey('class', $build[$group]['#attributes']);
          $this->assertContains('package-listing', $build[$group]['#attributes']['class']);
        }
      }
    }
  }

  /**
   * Tests render array has table elements with proper structure.
   *
   * Tables should have:
   * - #theme = 'table'
   * - #header with 5 columns
   * - #rows with theme data.
   *
   */
  public function testRenderArrayHasTableElements() {
    // Arrange.
    $data = $this->plugin->getDataOperationResult('themes');

    // Act.
    $build = $this->plugin->buildDataRenderArray($data, 'themes');

    // Assert.
    if (!empty($data)) {
      foreach (['core', 'not_core'] as $group) {
        if (isset($build[$group]['#children']['tables'])) {
          $table = $build[$group]['#children']['tables'];

          $this->assertEquals('table', $table['#theme']);
          $this->assertArrayHasKey('#header', $table);
          $this->assertArrayHasKey('#rows', $table);

          // Verify header has 5 columns.
          $this->assertCount(5, $table['#header'], 'Table should have 5 columns');

          // Verify rows are arrays.
          $this->assertIsArray($table['#rows']);

          // If rows exist, verify structure.
          if (!empty($table['#rows'])) {
            foreach ($table['#rows'] as $row) {
              $this->assertIsArray($row);
              // Each row should have required fields.
              $this->assertArrayHasKey('machine_name', $row);
              $this->assertArrayHasKey('Theme', $row);
            }
          }
        }
      }
    }
  }

}
