<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\SiteStructure;

use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;
use Drupal\system\Entity\Menu;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\Core\Url;

/**
 * Comprehensive tests for XrayAuditNavigationArchitecturePlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditNavigationArchitecturePluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'link',
    'menu_link_content',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The navigation plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\SiteStructure\XrayAuditNavigationArchitecturePlugin
   */
  protected $plugin;

  /**
   * Test menus.
   *
   * @var \Drupal\system\Entity\Menu[]
   */
  protected $menus;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('menu_link_content');
    $this->installConfig(['system', 'menu_link_content']);

    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');
    $this->plugin = $this->taskPluginManager->createInstance('navigation_architecture');

    // Create test menu structure.
    $this->createTestMenus();
  }

  /**
   * Creates test menus with hierarchical structure.
   */
  protected function createTestMenus() {
    // Create main menu.
    $this->menus['main'] = Menu::load('main');
    if (!$this->menus['main']) {
      $this->menus['main'] = Menu::create([
        'id' => 'main',
        'label' => 'Main navigation',
      ]);
      $this->menus['main']->save();
    }

    // Create menu items in hierarchy.
    // Level 1 item.
    $link1 = MenuLinkContent::create([
      'title' => 'Home',
      'link' => ['uri' => 'internal:/'],
      'menu_name' => 'main',
      'enabled' => TRUE,
      'weight' => 0,
    ]);
    $link1->save();

    // Level 1 item.
    $link2 = MenuLinkContent::create([
      'title' => 'About',
      'link' => ['uri' => 'internal:/about'],
      'menu_name' => 'main',
      'enabled' => TRUE,
      'weight' => 1,
    ]);
    $link2->save();

    // Level 2 item (child of About).
    $link3 = MenuLinkContent::create([
      'title' => 'Team',
      'link' => ['uri' => 'internal:/about/team'],
      'menu_name' => 'main',
      'parent' => 'menu_link_content:' . $link2->uuid(),
      'enabled' => TRUE,
      'weight' => 0,
    ]);
    $link3->save();

    // Disabled item.
    $link4 = MenuLinkContent::create([
      'title' => 'Disabled Link',
      'link' => ['uri' => 'internal:/disabled'],
      'menu_name' => 'main',
      'enabled' => FALSE,
      'weight' => 2,
    ]);
    $link4->save();
  }

  /**
   * Helper to invoke protected methods.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Tests getDataOperationResult() with menu operation returns structure.
   *
   */
  public function testGetDataOperationResultMenuOperationReturnsStructure() {
    $result = $this->plugin->getDataOperationResult('menu');

    $this->assertIsArray($result);
    $this->assertArrayHasKey('menu', $result);
    $this->assertArrayHasKey('items', $result);
    $this->assertArrayHasKey('item_number', $result);
    $this->assertArrayHasKey('level_max', $result);
    $this->assertArrayHasKey('show_parent_reference', $result);
  }

  /**
   * Tests getDataOperationResult() defaults to main menu.
   *
   */
  public function testGetDataOperationResultDefaultsToMainMenu() {
    $result = $this->plugin->getDataOperationResult('menu');

    $this->assertEquals('main', $result['menu']);
  }

  /**
   * Tests getDataOperationResult() includes menu items.
   *
   */
  public function testGetDataOperationResultIncludesMenuItems() {
    $result = $this->plugin->getDataOperationResult('menu');

    $this->assertIsArray($result['items']);
    // Should have menu items (exact count may vary based on system).
    $this->assertGreaterThanOrEqual(3, $result['item_number']);
  }

  /**
   * Tests getDataOperationResult() calculates max level.
   *
   */
  public function testGetDataOperationResultCalculatesMaxLevel() {
    $result = $this->plugin->getDataOperationResult('menu');

    $this->assertIsInt($result['level_max']);
    $this->assertGreaterThanOrEqual(1, $result['level_max']);
  }

  /**
   * Tests buildDataRenderArray() includes menu selector form.
   *
   * This test verifies that the form is actually rendered using FormBuilder,
   * not just a placeholder. The form should be a proper render array with
   * form elements, not just a container with markup.
   *
   */
  public function testBuildDataRenderArrayIncludesMenuSelectorForm() {
    $data = $this->plugin->getDataOperationResult('menu');
    $build = $this->plugin->buildDataRenderArray($data, 'menu');

    // Assert form key exists.
    $this->assertIsArray($build);
    $this->assertArrayHasKey('form', $build);

    // Assert it's a proper form render array, not a placeholder.
    // Forms built with FormBuilder have these characteristics:
    // 1. They have #form_id.
    // 2. They have form elements (not just #markup).
    // 3. They have #build_id indicating they were processed by FormBuilder.
    $this->assertArrayHasKey('#form_id', $build['form'],
      'Form should have #form_id indicating it was built by FormBuilder'
    );

    // Verify it's specifically the MenuSelectorForm.
    $this->assertEquals('xray_audit_menu_selector_form', $build['form']['#form_id'],
      'Form should be the MenuSelectorForm'
    );

    // Verify form has menu selection field.
    $this->assertArrayHasKey('menu', $build['form'],
      'Form should have menu selection field'
    );

    // Verify form has show_parent_reference checkbox.
    $this->assertArrayHasKey('show_parent_reference', $build['form'],
      'Form should have show_parent_reference checkbox'
    );

    // Verify form has actions.
    $this->assertArrayHasKey('actions', $build['form'],
      'Form should have actions (submit button)'
    );
  }

  /**
   * Tests menu selector form has correct menu options.
   *
   * Verifies that the form's menu dropdown contains "All menus" option
   * plus all available menus from the NavigationArchitecture service.
   *
   */
  public function testMenuSelectorFormHasCorrectMenuOptions() {
    $data = $this->plugin->getDataOperationResult('menu');
    $build = $this->plugin->buildDataRenderArray($data, 'menu');

    // Get expected menu list from service.
    $navigation_architecture = $this->container->get('xray_audit.navigation_architecture');
    $service_menus = $navigation_architecture->getMenuList();

    // Verify form has menu field with correct options.
    $this->assertArrayHasKey('menu', $build['form']);
    $this->assertArrayHasKey('#options', $build['form']['menu'],
      'Menu field should have #options array'
    );

    // Verify "all" option exists.
    $this->assertArrayHasKey('all', $build['form']['menu']['#options'],
      'Menu options should include "all" option'
    );

    // Verify all service menus are included.
    foreach ($service_menus as $menu_id => $menu_label) {
      $this->assertArrayHasKey($menu_id, $build['form']['menu']['#options'],
        "Menu options should include menu '$menu_id'"
      );
    }

    // Verify total count matches.
    $this->assertCount(count($service_menus) + 1, $build['form']['menu']['#options'],
      'Menu options should have all service menus plus "all" option'
    );
  }

  /**
   * Tests buildDataRenderArray() includes download button.
   *
   */
  public function testBuildDataRenderArrayIncludesDownloadButton() {
    $data = $this->plugin->getDataOperationResult('menu');

    if (!empty($data['items']) && !empty($data['menu'])) {
      $build = $this->plugin->buildDataRenderArray($data, 'menu');

      $this->assertArrayHasKey('download_button', $build);
      $this->assertEquals('link', $build['download_button']['#type'] ?? '');
      $this->assertArrayHasKey('#url', $build['download_button']);
      $this->assertArrayHasKey('#title', $build['download_button']);
    }
    else {
      $this->markTestSkipped('No menu data available for testing download button');
    }
  }

  /**
   * Tests buildDataRenderArray() creates headers based on max level.
   *
   */
  public function testBuildDataRenderArrayCreatesHeadersBasedOnMaxLevel() {
    $data = $this->plugin->getDataOperationResult('menu');

    if (!empty($data['items']) && $data['level_max'] > 0) {
      $build = $this->plugin->buildDataRenderArray($data, 'menu');

      $headers = $build['main_table']['#header'];

      // Should have level columns (L1, L2, etc.) plus fixed columns.
      $this->assertArrayHasKey('level', $headers);
      $this->assertArrayHasKey('enabled', $headers);
      $this->assertArrayHasKey('link', $headers);
      $this->assertArrayHasKey('target', $headers);
      $this->assertArrayHasKey('operation', $headers);

      // Should have L1 at minimum.
      $this->assertArrayHasKey('l_1', $headers);
    }
    else {
      $this->markTestSkipped('No menu items with levels for testing headers');
    }
  }

  /**
   * Tests prepareCsvHeaders() returns correct header structure.
   *
   */
  public function testPrepareCsvHeadersReturnsCorrectStructure() {
    $headers = $this->plugin->prepareCsvHeaders('menu');

    $this->assertIsArray($headers);
    $this->assertGreaterThanOrEqual(5, count($headers));

    // Verify key headers exist.
    $header_strings = array_map('strval', $headers);
    $this->assertContains('Level', $header_strings);
    $this->assertContains('Enabled', $header_strings);
    $this->assertContains('Link URL', $header_strings);
    $this->assertContains('Link Title', $header_strings);
    $this->assertContains('Link Description', $header_strings);
  }

  /**
   * Tests prepareCsvHeaders() includes dynamic level columns.
   *
   */
  public function testPrepareCsvHeadersIncludesDynamicLevelColumns() {
    $headers = $this->plugin->prepareCsvHeaders('menu');

    // Should include L1, L2, etc. based on menu depth.
    $header_strings = array_map('strval', $headers);

    // At minimum should have L1.
    $has_level_column = FALSE;
    foreach ($header_strings as $header) {
      if (preg_match('/^L\d+$/', $header)) {
        $has_level_column = TRUE;
        break;
      }
    }

    $this->assertTrue($has_level_column, 'Should have at least one level column (L1, L2, etc.)');
  }

  /**
   * Tests prepareCsvData() returns flattened data correctly.
   *
   */
  public function testPrepareCsvDataReturnsFlattenedData() {
    $data = $this->plugin->getDataOperationResult('menu');

    $csv_data = $this->plugin->prepareCsvData('menu', $data);

    $this->assertIsArray($csv_data);

    if (!empty($data['items'])) {
      $this->assertNotEmpty($csv_data);
      // Each row should be an array of values.
      $this->assertIsArray($csv_data[0]);
      // Should have multiple columns.
      $this->assertGreaterThanOrEqual(5, count($csv_data[0]));
    }
  }

  /**
   * Tests getHeaders() returns correct header structure.
   *
   */
  public function testGetHeadersReturnsCorrectStructure() {
    $headers = $this->plugin->getHeaders('menu');

    $this->assertIsArray($headers);
    $this->assertArrayHasKey('level', $headers);
    $this->assertArrayHasKey('enabled', $headers);
    $this->assertArrayHasKey('link', $headers);
    $this->assertArrayHasKey('target', $headers);
    $this->assertArrayHasKey('operation', $headers);
  }

  /**
   * Tests buildDataRenderArray() shows parent references when enabled.
   *
   * When show_parent_reference is TRUE, level columns should show the full
   * breadcrumb trail from root to that level (e.g., "Home > About > Team").
   *
   */
  public function testBuildDataRenderArrayShowsParentReferencesWhenEnabled() {
    // Create test data with hierarchical menu structure.
    $test_data = [
      'menu' => 'main',
      'level_max' => 3,
      'item_number' => 3,
      'show_parent_reference' => TRUE,
      'items' => [
        [
          'level' => 1,
          'title' => 'Home',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home'],
          'edit_link' => '',
        ],
        [
          'level' => 2,
          'title' => 'About',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home', 2 => 'About'],
          'edit_link' => '',
        ],
        [
          'level' => 3,
          'title' => 'Team',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home', 2 => 'About', 3 => 'Team'],
          'edit_link' => '',
        ],
      ],
    ];

    $build = $this->plugin->buildDataRenderArray($test_data, 'menu');

    $this->assertArrayHasKey('main_table', $build);
    $rows = $build['main_table']['#rows'];

    // First row (level 1: Home) - no parent columns should be shown.
    $this->assertEquals('Home', $rows[0]['l_1'],
      'Level 1 item should show its own title in L1 column'
    );
    $this->assertEquals('', $rows[0]['l_2'],
      'Level 1 item should have empty L2 column'
    );

    // Second row (level 2: About) - L1 should show breadcrumb.
    $l1_value = is_array($rows[1]['l_1']) ? $rows[1]['l_1']['data'] : $rows[1]['l_1'];
    $this->assertEquals('Home', $l1_value,
      'Level 2 item should show parent breadcrumb "Home" in L1 column'
    );
    $this->assertEquals('About', $rows[1]['l_2'],
      'Level 2 item should show its own title "About" in L2 column'
    );

    // Third row (level 3: Team) - L1 and L2 should show parent titles.
    $l1_value = is_array($rows[2]['l_1']) ? $rows[2]['l_1']['data'] : $rows[2]['l_1'];
    $l2_value = is_array($rows[2]['l_2']) ? $rows[2]['l_2']['data'] : $rows[2]['l_2'];

    $this->assertEquals('Home', $l1_value,
      'Level 3 item should show level 1 parent "Home" in L1 column'
    );
    $this->assertEquals('About', $l2_value,
      'Level 3 item should show level 2 parent "About" in L2 column (not cumulative breadcrumb)'
    );
    $this->assertEquals('Team', $rows[2]['l_3'],
      'Level 3 item should show its own title "Team" in L3 column'
    );
  }

  /**
   * Tests buildDataRenderArray() hides parent references when disabled.
   *
   * When show_parent_reference is FALSE (default), level columns should NOT
   * show parent references in gray cells - those cells should be empty.
   *
   */
  public function testBuildDataRenderArrayHidesParentReferencesWhenDisabled() {
    // Create test data with hierarchical menu structure.
    $test_data = [
      'menu' => 'main',
      'level_max' => 3,
      'item_number' => 3,
      'show_parent_reference' => FALSE,
      'items' => [
        [
          'level' => 1,
          'title' => 'Home',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home'],
          'edit_link' => '',
        ],
        [
          'level' => 2,
          'title' => 'About',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home', 2 => 'About'],
          'edit_link' => '',
        ],
        [
          'level' => 3,
          'title' => 'Team',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home', 2 => 'About', 3 => 'Team'],
          'edit_link' => '',
        ],
      ],
    ];

    $build = $this->plugin->buildDataRenderArray($test_data, 'menu');

    $this->assertArrayHasKey('main_table', $build);
    $rows = $build['main_table']['#rows'];

    // Second row (level 2: About) - L1 should be empty (no breadcrumb).
    $l1_value = is_array($rows[1]['l_1']) ? $rows[1]['l_1']['data'] : $rows[1]['l_1'];
    $this->assertEquals('', $l1_value,
      'Level 2 item should have empty L1 column when show_parent_reference is FALSE'
    );

    // Third row (level 3: Team) - L1 and L2 should be empty (no breadcrumbs).
    $l1_value = is_array($rows[2]['l_1']) ? $rows[2]['l_1']['data'] : $rows[2]['l_1'];
    $l2_value = is_array($rows[2]['l_2']) ? $rows[2]['l_2']['data'] : $rows[2]['l_2'];

    $this->assertEquals('', $l1_value,
      'Level 3 item should have empty L1 column when show_parent_reference is FALSE'
    );
    $this->assertEquals('', $l2_value,
      'Level 3 item should have empty L2 column when show_parent_reference is FALSE'
    );
  }

  /**
   * Tests CSV export includes parent references when enabled.
   *
   * When show_parent_reference is TRUE, CSV export should include the full
   * breadcrumb trail in level columns, matching the table display.
   *
   */
  public function testCsvExportIncludesParentReferencesWhenEnabled() {
    $test_data = [
      'menu' => 'main',
      'level_max' => 3,
      'item_number' => 3,
      'show_parent_reference' => TRUE,
      'items' => [
        [
          'level' => 1,
          'title' => 'Home',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home'],
          'edit_link' => '',
        ],
        [
          'level' => 2,
          'title' => 'About',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home', 2 => 'About'],
          'edit_link' => '',
        ],
        [
          'level' => 3,
          'title' => 'Team',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home', 2 => 'About', 3 => 'Team'],
          'edit_link' => '',
        ],
      ],
    ];

    $csv_data = $this->plugin->prepareCsvData('menu', $test_data);

    $this->assertNotEmpty($csv_data);
    $this->assertCount(3, $csv_data);

    // First row (level 1: Home).
    // CSV structure: [Menu, L1, L2, L3, Level, Enabled, Link URL, Link Title, Link Description].
    $this->assertEquals('Home', $csv_data[0][1], 'CSV row 1, L1 should be "Home"');
    $this->assertEquals('', $csv_data[0][2], 'CSV row 1, L2 should be empty');

    // Second row (level 2: About).
    $this->assertEquals('Home', $csv_data[1][1], 'CSV row 2, L1 should show parent "Home"');
    $this->assertEquals('About', $csv_data[1][2], 'CSV row 2, L2 should be "About"');

    // Third row (level 3: Team).
    $this->assertEquals('Home', $csv_data[2][1], 'CSV row 3, L1 should show level 1 parent "Home"');
    $this->assertEquals('About', $csv_data[2][2],
      'CSV row 3, L2 should show level 2 parent "About" (not cumulative breadcrumb)'
    );
    $this->assertEquals('Team', $csv_data[2][3], 'CSV row 3, L3 should be "Team"');
  }

  /**
   * Tests CSV export excludes parent references when disabled.
   *
   * When show_parent_reference is FALSE, CSV export should have empty strings
   * in parent level columns, matching the table display.
   *
   */
  public function testCsvExportExcludesParentReferencesWhenDisabled() {
    $test_data = [
      'menu' => 'main',
      'level_max' => 3,
      'item_number' => 3,
      'show_parent_reference' => FALSE,
      'items' => [
        [
          'level' => 1,
          'title' => 'Home',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home'],
          'edit_link' => '',
        ],
        [
          'level' => 2,
          'title' => 'About',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home', 2 => 'About'],
          'edit_link' => '',
        ],
        [
          'level' => 3,
          'title' => 'Team',
          'link' => Url::fromRoute('<front>'),
          'enabled' => TRUE,
          'target' => [],
          'levels' => [1 => 'Home', 2 => 'About', 3 => 'Team'],
          'edit_link' => '',
        ],
      ],
    ];

    $csv_data = $this->plugin->prepareCsvData('menu', $test_data);

    $this->assertNotEmpty($csv_data);

    // Second row (level 2: About) - L1 should be empty.
    $this->assertEquals('', $csv_data[1][1],
      'CSV row 2, L1 should be empty when show_parent_reference is FALSE'
    );

    // Third row (level 3: Team) - L1 and L2 should be empty.
    $this->assertEquals('', $csv_data[2][1],
      'CSV row 3, L1 should be empty when show_parent_reference is FALSE'
    );
    $this->assertEquals('', $csv_data[2][2],
      'CSV row 3, L2 should be empty when show_parent_reference is FALSE'
    );
  }

}
