<?php

namespace Drupal\Tests\xray_audit\Kernel\Services;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Tests the EntityArchitecture service.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class EntityArchitectureTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'file',
    'image',
    'media',
    'xray_audit',
  ];

  /**
   * The entity architecture service under test.
   *
   * @var \Drupal\xray_audit\Services\EntityArchitecture
   */
  protected $entityArchitecture;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install additional schemas.
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);

    // Get the service under test.
    $this->entityArchitecture = $this->container->get('xray_audit.entity_architecture');
  }

  /**
   * Tests getContentEntitiesInfo() returns content entities.
   */
  public function testGetContentEntitiesInfo() {
    // Act.
    $entities = $this->entityArchitecture->getContentEntitiesInfo();

    // Assert: Should contain at least user and node entities.
    $this->assertIsArray($entities);
    $this->assertArrayHasKey('user', $entities);
    $this->assertArrayHasKey('node', $entities);

    // Assert: Each entity has required keys.
    $this->assertArrayHasKey('definition', $entities['node']);
    $this->assertArrayHasKey('label', $entities['node']);
    $this->assertArrayHasKey('machine_name', $entities['node']);
    $this->assertArrayHasKey('bundles', $entities['node']);
  }

  /**
   * Tests getContentEntitiesInfo() entity structure.
   */
  public function testGetContentEntitiesInfoStructure() {
    // Arrange: Create a content type.
    $this->createContentType('test_type', 'Test Type');

    // Act.
    $entities = $this->entityArchitecture->getContentEntitiesInfo();

    // Assert: Node entity should have our test bundle.
    $this->assertArrayHasKey('test_type', $entities['node']['bundles']);
    // Verify machine_name is set correctly.
    $this->assertEquals('test_type', $entities['node']['bundles']['test_type']['machine_name']);
    // Verify label key exists (label value structure may vary).
    $this->assertArrayHasKey('label', $entities['node']['bundles']['test_type']);
  }

  /**
   * Tests getEntityFieldData() with base fields.
   */
  public function testGetEntityFieldDataWithBaseFields() {
    // Arrange: Create a content type.
    $this->createContentType('article', 'Article');

    // Act: Get field data for node article bundle.
    $field_data = $this->entityArchitecture->getEntityFieldData('node', 'article');

    // Assert: Should contain base fields.
    $this->assertIsArray($field_data);
    $this->assertArrayHasKey('nid', $field_data);
    $this->assertArrayHasKey('title', $field_data);
    $this->assertArrayHasKey('status', $field_data);
    $this->assertArrayHasKey('created', $field_data);
    $this->assertArrayHasKey('changed', $field_data);

    // Assert: Field data structure.
    $this->assertEquals('node', $field_data['nid']['entity']);
    $this->assertEquals('article', $field_data['nid']['bundle']);
    $this->assertArrayHasKey('field_type', $field_data['nid']);
    $this->assertArrayHasKey('type', $field_data['nid']);
  }

  /**
   * Tests getEntityFieldData() with custom text field.
   */
  public function testGetEntityFieldDataWithTextField() {
    // Arrange: Create content type and text field.
    $this->createContentType('page', 'Page');
    $this->createTestField('field_test_text', 'node', 'page', 'string', [], []);

    // Act.
    $field_data = $this->entityArchitecture->getEntityFieldData('node', 'page');

    // Assert: Custom field should be present.
    $this->assertArrayHasKey('field_test_text', $field_data);
    $field = $field_data['field_test_text'];

    $this->assertEquals('node', $field['entity']);
    $this->assertEquals('page', $field['bundle']);
    $this->assertEquals('field_test_text', $field['name']);
    $this->assertEquals('string', $field['type']);
    $this->assertNotNull($field['field_type']);
  }

  /**
   * Tests getEntityFieldData() with entity reference field.
   */
  public function testGetEntityFieldDataWithEntityReferenceField() {
    // Arrange: Create content types.
    $this->createContentType('article', 'Article');
    $this->createContentType('page', 'Page');

    // Create entity reference field.
    $this->createTestField(
      'field_ref_node',
      'node',
      'article',
      'entity_reference',
      ['target_type' => 'node'],
      ['handler_settings' => ['target_bundles' => ['page' => 'page']]]
    );

    // Act.
    $field_data = $this->entityArchitecture->getEntityFieldData('node', 'article');

    // Assert.
    $this->assertArrayHasKey('field_ref_node', $field_data);
    $field = $field_data['field_ref_node'];

    $this->assertEquals('entity_reference', $field['type']);
    $this->assertIsArray($field['settings']);
    $this->assertArrayHasKey('target_type', $field['settings']);
    $this->assertEquals('node', $field['settings']['target_type']);
  }

  /**
   * Tests getEntityFieldData() with multi-value field.
   */
  public function testGetEntityFieldDataWithMultiValueField() {
    // Arrange.
    $this->createContentType('test_multi', 'Test Multi');

    // Create multi-value field (unlimited cardinality).
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_multi_text',
      'entity_type' => 'node',
      'type' => 'string',
      'cardinality' => -1,
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'test_multi',
      'label' => 'Multi Text',
    ]);
    $field->save();

    // Act.
    $field_data = $this->entityArchitecture->getEntityFieldData('node', 'test_multi');

    // Assert.
    $this->assertArrayHasKey('field_multi_text', $field_data);
    // Cardinality might be null for FieldConfig (not BaseFieldDefinition).
    $this->assertArrayHasKey('cardinality', $field_data['field_multi_text']);
  }

  /**
   * Tests getEntityFieldData() with required field.
   */
  public function testGetEntityFieldDataWithRequiredField() {
    // Arrange.
    $this->createContentType('test_required', 'Test Required');

    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_required',
      'entity_type' => 'node',
      'type' => 'string',
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'test_required',
      'label' => 'Required Field',
      'required' => TRUE,
    ]);
    $field->save();

    // Act.
    $field_data = $this->entityArchitecture->getEntityFieldData('node', 'test_required');

    // Assert.
    $this->assertArrayHasKey('field_required', $field_data);
    $this->assertTrue($field_data['field_required']['is_required']);
  }

  /**
   * Tests getEntityFieldData() with translatable field.
   */
  public function testGetEntityFieldDataWithTranslatableField() {
    // Arrange.
    $this->createContentType('test_trans', 'Test Translatable');

    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_trans',
      'entity_type' => 'node',
      'type' => 'string',
      'translatable' => TRUE,
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'test_trans',
      'label' => 'Translatable Field',
      'translatable' => TRUE,
    ]);
    $field->save();

    // Act.
    $field_data = $this->entityArchitecture->getEntityFieldData('node', 'test_trans');

    // Assert.
    $this->assertArrayHasKey('field_trans', $field_data);
    $this->assertTrue($field_data['field_trans']['is_translatable']);
  }

  /**
   * Tests getEntityFieldData() with field with default value.
   */
  public function testGetEntityFieldDataWithDefaultValue() {
    // Arrange.
    $this->createContentType('test_default', 'Test Default');

    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_default',
      'entity_type' => 'node',
      'type' => 'string',
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'test_default',
      'label' => 'Field with Default',
      'default_value' => [['value' => 'default_value']],
    ]);
    $field->save();

    // Act.
    $field_data = $this->entityArchitecture->getEntityFieldData('node', 'test_default');

    // Assert.
    $this->assertArrayHasKey('field_default', $field_data);
    $this->assertNotEmpty($field_data['field_default']['default_value']);
  }

  /**
   * Tests getDataForViewDisplayArchitecture() returns display configuration.
   */
  public function testGetDataForViewDisplayArchitecture() {
    // Arrange.
    $this->createContentType('display_test', 'Display Test');

    // Act.
    $display_data = $this->entityArchitecture->getDataForViewDisplayArchitecture('node', 'display_test');

    // Assert: Should return fields and optionally displays.
    $this->assertIsArray($display_data);
    $this->assertArrayHasKey('fields', $display_data);

    // Assert: Fields should contain base fields.
    $this->assertArrayHasKey('title', $display_data['fields']);

    // Displays key exists only if there are view displays configured.
    if (isset($display_data['displays'])) {
      $this->assertIsArray($display_data['displays']);
    }
  }

  /**
   * Tests getDataForViewDisplayArchitecture() with custom field.
   */
  public function testGetDataForViewDisplayArchitectureWithCustomField() {
    // Arrange.
    $this->createContentType('custom_display', 'Custom Display');
    $this->createTestField('field_custom', 'node', 'custom_display', 'string');

    // Act.
    $display_data = $this->entityArchitecture->getDataForViewDisplayArchitecture('node', 'custom_display');

    // Assert.
    $this->assertArrayHasKey('field_custom', $display_data['fields']);
    $field = $display_data['fields']['field_custom'];

    // Should have widget info populated.
    $this->assertArrayHasKey('widget_form_default', $field);
  }

  /**
   * Tests getDataForViewDisplayArchitecture() returns empty for invalid bundle.
   */
  public function testGetDataForViewDisplayArchitectureEmptyForInvalidBundle() {
    // Act: Try with non-existent bundle.
    $display_data = $this->entityArchitecture->getDataForViewDisplayArchitecture('node', 'invalid_bundle_xyz');

    // Assert: Should return empty or minimal data structure.
    $this->assertIsArray($display_data);
  }

  /**
   * Tests getDataForEntityFieldArchitecture() returns comprehensive data.
   */
  public function testGetDataForEntityFieldArchitecture() {
    // Arrange: Create content type with field.
    $this->createContentType('full_test', 'Full Test');
    $this->createTestField('field_full', 'node', 'full_test', 'string');

    // Act.
    $data = $this->entityArchitecture->getDataForEntityFieldArchitecture();

    // Assert: Should return array of field data.
    $this->assertIsArray($data);
    $this->assertNotEmpty($data);

    // Find our test bundle in the data.
    $found_bundle = FALSE;
    foreach ($data as $item) {
      if (isset($item['content']['bundle']) && $item['content']['bundle'] === 'full_test') {
        $found_bundle = TRUE;
        break;
      }
    }
    $this->assertTrue($found_bundle, 'Test bundle should be in returned data');
  }

  /**
   * Tests field data includes all required properties.
   */
  public function testFieldDataIncludesRequiredProperties() {
    // Arrange.
    $this->createContentType('props_test', 'Props Test');

    // Act.
    $field_data = $this->entityArchitecture->getEntityFieldData('node', 'props_test');

    // Assert: Check title field has all required properties.
    $title_field = $field_data['title'];

    $required_keys = [
      'entity',
      'bundle',
      'field_type',
      'type',
      'name',
      'label',
      'computed',
      'cardinality',
      'is_required',
      'is_base_field',
      'is_read_only',
      'is_translatable',
      'is_revisionable',
    ];

    foreach ($required_keys as $key) {
      $this->assertArrayHasKey($key, $title_field, "Field data should contain '{$key}' property");
    }
  }

  /**
   * Tests that computed fields are properly identified.
   */
  public function testComputedFieldsIdentified() {
    // Arrange.
    $this->createContentType('computed_test', 'Computed Test');

    // Act.
    $field_data = $this->entityArchitecture->getEntityFieldData('node', 'computed_test');

    // Assert: Check that computed property exists for changed field.
    if (isset($field_data['changed'])) {
      $this->assertArrayHasKey('computed', $field_data['changed']);
      // The computed property exists, value may vary by Drupal version.
      $this->assertIsBool($field_data['changed']['computed']);
    }
    else {
      // If changed field doesn't exist, test passes.
      $this->assertTrue(TRUE, 'Changed field not present in this Drupal version');
    }
  }

  /**
   * Tests field with custom settings.
   */
  public function testFieldWithCustomSettings() {
    // Arrange.
    $this->createContentType('settings_test', 'Settings Test');

    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_with_settings',
      'entity_type' => 'node',
      'type' => 'string',
      'settings' => [
        'max_length' => 255,
      ],
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'settings_test',
      'label' => 'Field With Settings',
      'settings' => [
        'custom_setting' => 'custom_value',
      ],
    ]);
    $field->save();

    // Act.
    $field_data = $this->entityArchitecture->getEntityFieldData('node', 'settings_test');

    // Assert.
    $this->assertArrayHasKey('field_with_settings', $field_data);
    $this->assertArrayHasKey('settings', $field_data['field_with_settings']);
    $this->assertIsArray($field_data['field_with_settings']['settings']);
  }

  /**
   * Tests multiple content types are processed correctly.
   */
  public function testMultipleContentTypesProcessed() {
    // Arrange: Create multiple content types.
    $this->createContentType('type_one', 'Type One');
    $this->createContentType('type_two', 'Type Two');
    $this->createContentType('type_three', 'Type Three');

    // Act.
    $data = $this->entityArchitecture->getDataForEntityFieldArchitecture();

    // Assert: All types should be in the data.
    $found_types = [];
    foreach ($data as $item) {
      if (isset($item['content']['bundle'])) {
        $found_types[] = $item['content']['bundle'];
      }
    }

    $this->assertContains('type_one', $found_types);
    $this->assertContains('type_two', $found_types);
    $this->assertContains('type_three', $found_types);
  }

  /**
   * Tests service handles empty entity type gracefully.
   */
  public function testServiceHandlesEmptyEntityTypeGracefully() {
    // Act & Assert: Should not throw exception with non-content entity.
    $field_data = $this->entityArchitecture->getEntityFieldData('user', 'user');

    // User entity should return field data.
    $this->assertIsArray($field_data);
    $this->assertNotEmpty($field_data);
  }

}
