<?php

namespace Drupal\Tests\xray_audit\Kernel\Services;

use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Tests the EntityUseParagraph service.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class EntityUseParagraphTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'xray_audit',
  ];

  /**
   * The entity use paragraph service under test.
   *
   * @var \Drupal\xray_audit\Services\EntityUseParagraph
   */
  protected $entityUseParagraph;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Get the service under test.
    $this->entityUseParagraph = $this->container->get('xray_audit.entity_use_paragraph');
  }

  /**
   * Tests countEntityUses() result structure when data exists.
   */
  public function testCountEntityUsesResultStructure() {
    // Act.
    $result = $this->entityUseParagraph->countEntityUses();

    // Assert: If result has data, check structure.
    if (!empty($result)) {
      $first_result = reset($result);
      $this->assertArrayHasKey('entity', $first_result);
      $this->assertArrayHasKey('bundle', $first_result);
      $this->assertArrayHasKey('label', $first_result);
      $this->assertArrayHasKey('count', $first_result);
      $this->assertArrayHasKey('site', $first_result);

      $this->assertEquals('paragraph', $first_result['entity']);
    }
    else {
      // No paragraph bundles configured, which is valid.
      $this->assertIsArray($result);
    }
  }

  /**
   * Tests countEntityUses() uses paragraph usage map.
   *
   */
  public function testCountEntityUsesUsesParagraphUsageMap() {
    // Act: Call countEntityUses.
    $result = $this->entityUseParagraph->countEntityUses();

    // Assert: Method executed successfully.
    $this->assertIsArray($result);

    // The method internally calls paragraphUsageMap->getTreeMap().
    // If no exception was thrown, the service integration works.
    $this->assertTrue(TRUE, 'countEntityUses executed without error');
  }

  /**
   * Data provider for parent entity types.
   *
   * @return array
   *   Test cases.
   */
  public function parentEntityTypeProvider() {
    return [
      'node' => ['node', 'article'],
      'user' => ['user', NULL],
      'taxonomy_term' => ['taxonomy_term', 'tags'],
    ];
  }

  /**
   * Tests countEntityUses() handles empty paragraph usage.
   *
   */
  public function testCountEntityUsesEmptyParagraphUsage() {
    // Act: Even with no paragraphs configured, should return array.
    $result = $this->entityUseParagraph->countEntityUses();

    // Assert.
    $this->assertIsArray($result);
    // Result may be empty if no paragraph bundles exist.
    $this->assertTrue(TRUE, 'Method handles empty paragraph usage');
  }

  /**
   * Tests service integration with paragraph usage map service.
   *
   */
  public function testParagraphUsageMapIntegration() {
    // Assert: The service has ParagraphUsageMap injected.
    $reflection = new \ReflectionClass($this->entityUseParagraph);
    $property = $reflection->getProperty('paragraphUsageMap');
    $property->setAccessible(TRUE);

    $usageMap = $property->getValue($this->entityUseParagraph);

    // Verify it's not null and is the expected type.
    $this->assertNotNull($usageMap);
    $this->assertIsObject($usageMap);
  }

}
