<?php

namespace Drupal\Tests\xray_audit\Kernel\Services;

use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;
use Drupal\Core\Url;

/**
 * Tests the PluginRepository service.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class PluginRepositoryTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'xray_audit',
  ];

  /**
   * The plugin repository service under test.
   *
   * @var \Drupal\xray_audit\Services\PluginRepository
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Get the service under test.
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');
  }

  /**
   * Tests getGroupPluginDefinitions() returns group plugin definitions.
   *
   */
  public function testGetGroupPluginDefinitions() {
    // Act.
    $definitions = $this->pluginRepository->getGroupPluginDefinitions();

    // Assert.
    $this->assertIsArray($definitions);
    $this->assertNotEmpty($definitions, 'Should have at least one group plugin');

    // Check structure of first definition.
    $first_definition = reset($definitions);
    $this->assertArrayHasKey('id', $first_definition);
    $this->assertArrayHasKey('label', $first_definition);
  }

  /**
   * Tests getTaskPluginDefinitions() returns task plugin definitions.
   *
   */
  public function testGetTaskPluginDefinitions() {
    // Act.
    $definitions = $this->pluginRepository->getTaskPluginDefinitions();

    // Assert.
    $this->assertIsArray($definitions);
    $this->assertNotEmpty($definitions, 'Should have at least one task plugin');

    // Check structure of first definition.
    $first_definition = reset($definitions);
    $this->assertArrayHasKey('id', $first_definition);
    $this->assertArrayHasKey('label', $first_definition);
  }

  /**
   * Tests getGroupPageUrl() generates correct URL.
   *
   */
  public function testGetGroupPageUrl() {
    // Arrange: Get a real group plugin ID.
    $definitions = $this->pluginRepository->getGroupPluginDefinitions();
    $this->assertNotEmpty($definitions, 'Need at least one group plugin for this test');

    $first_definition = reset($definitions);
    $group_id = $first_definition['id'];

    // Act.
    $url = $this->pluginRepository->getGroupPageUrl($group_id);

    // Assert.
    $this->assertInstanceOf(Url::class, $url);
    $this->assertEquals('xray_audit.group_page', $url->getRouteName());
    $this->assertEquals($group_id, $url->getRouteParameters()['group_plugin']);
  }

  /**
   * Tests getTaskPageOperationFromIdOperation() generates URL.
   *
   */
  public function testGetTaskPageOperationFromIdOperation() {
    // Act.
    $url = $this->pluginRepository->getTaskPageOperationFromIdOperation('test_operation');

    // Assert.
    $this->assertInstanceOf(Url::class, $url);
    $this->assertEquals('xray_audit.task_page.test_operation', $url->getRouteName());
  }

  /**
   * Tests getTaskPageOperationFromIdOperation() with query parameters.
   *
   */
  public function testGetTaskPageOperationFromIdOperationWithQuery() {
    // Arrange.
    $query_options = ['param1' => 'value1', 'param2' => 'value2'];

    // Act.
    $url = $this->pluginRepository->getTaskPageOperationFromIdOperation('test_op', $query_options);

    // Assert.
    $this->assertInstanceOf(Url::class, $url);
    $this->assertEquals('xray_audit.task_page.test_op', $url->getRouteName());
    $this->assertEquals($query_options, $url->getOption('query'));
  }

  /**
   * Tests explodeFromParameterTaskOperation() parses correctly.
   *
   */
  public function testExplodeFromParameterTaskOperation() {
    // Arrange.
    $task_operation = 'task_plugin_id-operation_name';

    // Act.
    $result = $this->pluginRepository->explodeFromParameterTaskOperation($task_operation);

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('task_plugin', $result);
    $this->assertArrayHasKey('operation', $result);
    $this->assertEquals('task_plugin_id', $result['task_plugin']);
    $this->assertEquals('operation_name', $result['operation']);
  }

  /**
   * Tests getInstancePluginGroup() returns group plugin instance.
   *
   */
  public function testGetInstancePluginGroup() {
    // Arrange: Get a real group plugin ID.
    $definitions = $this->pluginRepository->getGroupPluginDefinitions();
    $this->assertNotEmpty($definitions, 'Need at least one group plugin');

    $first_definition = reset($definitions);
    $group_id = $first_definition['id'];

    // Act.
    $plugin = $this->pluginRepository->getInstancePluginGroup($group_id);

    // Assert.
    $this->assertNotNull($plugin);
    $this->assertInstanceOf('\Drupal\xray_audit\Plugin\XrayAuditGroupPluginInterface', $plugin);
  }

  /**
   * Tests getInstancePluginTask() returns task plugin instance.
   *
   */
  public function testGetInstancePluginTask() {
    // Arrange: Get a real task plugin ID.
    $definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $this->assertNotEmpty($definitions, 'Need at least one task plugin');

    $first_definition = reset($definitions);
    $task_id = $first_definition['id'];

    // Act.
    $plugin = $this->pluginRepository->getInstancePluginTask($task_id);

    // Assert.
    $this->assertNotNull($plugin);
    $this->assertInstanceOf('\Drupal\xray_audit\Plugin\XrayAuditTaskPluginInterface', $plugin);
  }

  /**
   * Tests getAllPluginGroupRouteInfo() returns route information.
   *
   */
  public function testGetAllPluginGroupRouteInfo() {
    // Act.
    $route_info = $this->pluginRepository->getAllPluginGroupRouteInfo();

    // Assert.
    $this->assertIsArray($route_info);
    $this->assertNotEmpty($route_info, 'Should have route info for group plugins');

    // Check structure of first route info.
    $first_route = reset($route_info);
    $this->assertArrayHasKey('id', $first_route);
    $this->assertArrayHasKey('url', $first_route);
    $this->assertArrayHasKey('route_name', $first_route);
    $this->assertArrayHasKey('title', $first_route);

    // Check URL format.
    $this->assertStringStartsWith('/admin/reports/xray-audit/', $first_route['url']);
  }

  /**
   * Tests getOnePluginGroupRouteInfo() returns specific group route info.
   *
   */
  public function testGetOnePluginGroupRouteInfo() {
    // Arrange: Get a real group plugin ID.
    $all_route_info = $this->pluginRepository->getAllPluginGroupRouteInfo();
    $this->assertNotEmpty($all_route_info);

    $group_ids = array_keys($all_route_info);
    $test_group_id = reset($group_ids);

    // Act.
    $route_info = $this->pluginRepository->getOnePluginGroupRouteInfo($test_group_id);

    // Assert.
    $this->assertIsArray($route_info);
    $this->assertEquals($test_group_id, $route_info['id']);
    $this->assertArrayHasKey('url', $route_info);
    $this->assertArrayHasKey('route_name', $route_info);
  }

  /**
   * Tests getAllPluginTaskRouteInfo() returns task route information.
   *
   */
  public function testGetAllPluginTaskRouteInfo() {
    // Act.
    $route_info = $this->pluginRepository->getAllPluginTaskRouteInfo();

    // Assert.
    $this->assertIsArray($route_info);

    // If there are task plugins with operations, check structure.
    if (!empty($route_info)) {
      $first_task_routes = reset($route_info);
      $this->assertIsArray($first_task_routes);

      if (!empty($first_task_routes)) {
        $first_operation_route = reset($first_task_routes);
        $this->assertArrayHasKey('operation_id', $first_operation_route);
        $this->assertArrayHasKey('url', $first_operation_route);
        $this->assertArrayHasKey('route_name', $first_operation_route);
        $this->assertArrayHasKey('title', $first_operation_route);
      }
    }
  }

  /**
   * Tests task plugin definitions have required structure.
   *
   */
  public function testTaskPluginDefinitionsStructure() {
    // Act.
    $definitions = $this->pluginRepository->getTaskPluginDefinitions();

    // Assert: Should have at least one task plugin.
    $this->assertNotEmpty($definitions);

    // Assert: Each definition should be an array.
    foreach ($definitions as $plugin_id => $definition) {
      $this->assertIsString($plugin_id);
      $this->assertIsArray($definition);

      // Skip empty definitions (may be deriver placeholders or other plugins).
      if (empty($definition)) {
        continue;
      }

      // Task plugins typically have group and operations.
      // Plugin structure may vary, so check optionally.
      if (isset($definition['group'])) {
        $this->assertIsString($definition['group']);
      }
      if (isset($definition['operations'])) {
        $this->assertIsArray($definition['operations']);
      }
    }
  }

  /**
   * Tests route info uses correct URL pattern.
   *
   */
  public function testGroupRouteInfoUrlPattern() {
    // Act.
    $route_info = $this->pluginRepository->getAllPluginGroupRouteInfo();

    // Assert: All URLs should follow pattern.
    foreach ($route_info as $group_id => $info) {
      // URL should replace underscores with dashes.
      $expected_url_part = str_replace('_', '-', $group_id);
      $this->assertStringContainsString($expected_url_part, $info['url']);
    }
  }

  /**
   * Data provider for task operation test.
   *
   * @return array
   *   Test cases.
   */
  public function taskOperationProvider() {
    return [
      'valid_simple' => [
        'task-operation',
        ['task_plugin' => 'task', 'operation' => 'operation'],
      ],
      'valid_with_underscores' => [
        'my_task_plugin-my_operation',
        ['task_plugin' => 'my_task_plugin', 'operation' => 'my_operation'],
      ],
      'invalid_no_dash' => [
        // Intentionally missing dash separator to test invalid format.
        'taskoperation',
        NULL,
      ],
      'invalid_empty' => [
        '',
        NULL,
      ],
      'invalid_only_dash' => [
        '-',
        // Returns array with empty strings (explode on single dash).
        ['task_plugin' => '', 'operation' => ''],
      ],
      'invalid_too_many_dashes' => [
        'task-op-extra',
        NULL,
      ],
    ];
  }

}
