<?php

namespace Drupal\Tests\xray_audit\Unit;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\Tests\UnitTestCase;
use Drupal\xray_audit\Services\CsvDownloadManagerInterface;
use Drupal\xray_audit\XrayAuditTaskCsvDownloadTrait;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * Tests for XrayAuditTaskCsvDownloadTrait.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditTaskCsvDownloadTraitTest extends UnitTestCase {

  /**
   * Mock CSV download service.
   *
   * @var \Drupal\xray_audit\Services\CsvDownloadManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $csvDownloadService;

  /**
   * Test implementation of the trait.
   *
   * @var \Drupal\Tests\xray_audit\Unit\TestCsvDownloadClass
   */
  protected $traitObject;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock CSV download service.
    $this->csvDownloadService = $this->createMock(CsvDownloadManagerInterface::class);

    // Set up container.
    $container = new ContainerBuilder();
    $container->set('xray_audit.csv_download_manager', $this->csvDownloadService);
    \Drupal::setContainer($container);

    // Create test object that uses the trait.
    $this->traitObject = new TestCsvDownloadClass();
  }

  /**
   * Tests handleCsv with additional data parameter.
   *
   */
  public function testHandleCsvWithAdditionalDataParameters(): void {
    $operation = 'test_operation';
    $additional_data = [
      ['key1' => 'value1'],
      ['key2' => 'value2'],
    ];

    $this->csvDownloadService->expects($this->once())
      ->method('downloadCsv')
      ->willReturn(TRUE);

    $this->csvDownloadService->expects($this->once())
      ->method('createCsv')
      ->willReturn(new StreamedResponse());

    $this->traitObject->handleCsv($operation, $additional_data);

    // Verify data was passed to prepareCsvData.
    $passed_data = $this->traitObject->getPassedData();
    $this->assertIsArray($passed_data);
    $this->assertEquals($additional_data, $passed_data);
  }

  /**
   * Tests createRenderableLink with default parameters.
   *
   */
  public function testCreateRenderableLinkWithDefaults(): void {
    $operation = 'test_operation';
    $url = Url::fromRoute('test.route');
    $link = new Link('Default Text', $url);

    $this->csvDownloadService->expects($this->once())
      ->method('createLink')
      ->with($operation)
      ->willReturn($link);

    $result = $this->traitObject->createRenderableLink($operation);

    $this->assertIsArray($result);
    $this->assertArrayHasKey('#type', $result);
  }

  /**
   * Tests createRenderableLink with both title and extra options.
   *
   */
  public function testCreateRenderableLinkWithTitleAndExtraOptions(): void {
    $operation = 'test_operation';
    $title = 'Export Data';
    $extra_options = [
      'attributes' => ['target' => '_blank'],
    ];

    $url = $this->getMockBuilder(Url::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['mergeOptions'])
      ->getMock();

    $url->expects($this->exactly(2))
      ->method('mergeOptions');

    $link = $this->getMockBuilder(Link::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['getUrl', 'setText', 'toRenderable'])
      ->getMock();

    $link->expects($this->once())
      ->method('getUrl')
      ->willReturn($url);

    $link->expects($this->once())
      ->method('setText')
      ->with($title);

    $link->expects($this->once())
      ->method('toRenderable')
      ->willReturn([
        '#type' => 'link',
        '#title' => $title,
        '#url' => $url,
      ]);

    $this->csvDownloadService->expects($this->once())
      ->method('createLink')
      ->with($operation)
      ->willReturn($link);

    $result = $this->traitObject->createRenderableLink($operation, $title, $extra_options);

    $this->assertIsArray($result);
    $this->assertArrayHasKey('#type', $result);
  }

  /**
   * Tests handleCsv with multiple operations.
   *
   * @dataProvider providerOperations
   */
  public function testHandleCsvWithMultipleOperations(string $operation): void {
    $this->csvDownloadService->expects($this->once())
      ->method('downloadCsv')
      ->willReturn(TRUE);

    $this->csvDownloadService->expects($this->once())
      ->method('createCsv')
      ->with(
        $this->anything(),
        $this->anything(),
        $operation
      )
      ->willReturn(new StreamedResponse());

    $this->traitObject->handleCsv($operation);

    $this->assertEquals($operation, $this->traitObject->getLastOperation());
  }

  /**
   * Data provider for operation tests.
   *
   * @return array
   *   Test scenarios with different operations.
   */
  public static function providerOperations(): array {
    return [
      'modules operation' => ['modules'],
      'themes operation' => ['themes'],
      'views operation' => ['views'],
      'content_types operation' => ['content_types'],
      'database_tables operation' => ['database_tables'],
    ];
  }

}

/**
 * Test class that implements the trait for testing purposes.
 */
class TestCsvDownloadClass {
  use XrayAuditTaskCsvDownloadTrait {
    getCsvDownloadService as public;
    handleCsv as public;
    createRenderableLink as public;
  }

  /**
   * Track if prepareCsvHeaders was called.
   *
   * @var bool
   */
  private $prepareHeadersCalled = FALSE;

  /**
   * Track if prepareCsvData was called.
   *
   * @var bool
   */
  private $prepareDataCalled = FALSE;

  /**
   * Store passed data for verification.
   *
   * @var array
   */
  private $passedData = [];

  /**
   * Store last operation.
   *
   * @var string
   */
  private $lastOperation = '';

  /**
   * Mock implementation of getHeaders.
   *
   * @param string $operation
   *   Operation.
   *
   * @return array
   *   Headers.
   */
  protected function getHeaders(string $operation): array {
    return ['Header1', 'Header2', 'Header3'];
  }

  /**
   * {@inheritdoc}
   */
  protected function prepareCsvHeaders(string $operation): array {
    $this->prepareHeadersCalled = TRUE;
    $this->lastOperation = $operation;
    return ['Header1', 'Header2', 'Header3'];
  }

  /**
   * {@inheritdoc}
   */
  protected function prepareCsvData(string $operation, array $data): array {
    $this->prepareDataCalled = TRUE;
    $this->lastOperation = $operation;
    $this->passedData = $data;
    return [['value1', 'value2', 'value3']];
  }

  /**
   * Mock implementation of getRows.
   *
   * @param string $operation
   *   Operation.
   *
   * @return array
   *   Rows.
   */
  protected function getRows(string $operation): array {
    return [['value1', 'value2', 'value3']];
  }

  /**
   * Check if prepareCsvHeaders was called.
   *
   * @return bool
   *   TRUE if called.
   */
  public function wasPrepareHeadersCalled(): bool {
    return $this->prepareHeadersCalled;
  }

  /**
   * Check if prepareCsvData was called.
   *
   * @return bool
   *   TRUE if called.
   */
  public function wasPrepareDataCalled(): bool {
    return $this->prepareDataCalled;
  }

  /**
   * Get passed data.
   *
   * @return array
   *   The passed data.
   */
  public function getPassedData(): array {
    return $this->passedData;
  }

  /**
   * Get last operation.
   *
   * @return string
   *   The last operation.
   */
  public function getLastOperation(): string {
    return $this->lastOperation;
  }

}
