<?php

namespace Drupal\xray_audit_insight\Plugin;

use Drupal\Component\Plugin\Definition\PluginDefinitionInterface;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for xray_audit_insight_plugin plugins.
 */
abstract class XrayAuditInsightPluginBase extends PluginBase implements XrayAuditInsightPluginInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The module settings.
   */
  const MODULE_SETTINGS = 'xray_audit_insight.settings';

  /**
   * Plugin manager xray.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $pluginManagerTaskXray;

  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Render.
   *
   * @var \Drupal\Core\Render\Renderer
   */
  protected $renderer;

  /**
   * Module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Site settings.
   *
   * @var \Drupal\Core\Site\Settings
   */
  protected $siteSettings;

  /**
   * Task plugin id.
   *
   * Tha task plugin used to build the insight.
   *
   * @var string
   */
  protected $taskPluginId;

  /**
   * Plugin repository.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditPluginRepository
   */
  protected $pluginRepository;

  /**
   * Operation.
   *
   * The Operation used to build the insight.
   *
   * @var string
   */
  protected $operation;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instanced_object = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );
    $instanced_object->pluginManagerTaskXray = $container->get('plugin_manager.xray_audit_task');
    $instanced_object->configFactory = $container->get('config.factory');
    $instanced_object->renderer = $container->get('renderer');
    $instanced_object->moduleHandler = $container->get('module_handler');
    $instanced_object->siteSettings = $container->get('settings');
    $instanced_object->pluginRepository = $container->get('xray_audit.plugin_repository');
    return $instanced_object;
  }

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    // Cast the label to a string since it is a TranslatableMarkup object.
    return (string) $this->getValuesFromDefinition('label');
  }

  /**
   * {@inheritdoc}
   */
  public function getSort() {
    return $this->getValuesFromDefinition('sort') ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function isActive(): bool {
    if (!$this->areDependenciesActive()) {
      return FALSE;
    }

    /** @var \Drupal\Core\Config\ConfigFactoryInterface $config_factory */
    $config_factory = $this->configFactory;

    $config_object = $config_factory->get(static::MODULE_SETTINGS);
    if (!$config_object instanceof ImmutableConfig) {
      return TRUE;
    }

    $excluded_insights = $config_object->get('excluded_insights') ?? [];
    $plugin_id = $this->getPluginId();
    return !(isset($excluded_insights[$plugin_id]) && $excluded_insights[$plugin_id] == "1");
  }

  /**
   * Check it the dependencies are active.
   *
   * In some cases, only is needed to check if
   * the plugin task is enabled.
   *
   * @return bool
   *   If false, the insight will not be executed.
   */
  protected function areDependenciesActive(): bool {
    // If not set the properties taskPluginId and operation, return false.
    // In any case, this method can be override in the child class.
    if (!isset($this->taskPluginId) || !isset($this->operation)) {
      return FALSE;
    }
    return (bool) $this->getTaskPluginDefinition($this->taskPluginId, $this->operation);
  }

  /**
   * Get values from plugin definition.
   *
   * @param string $value_parameter
   *   Value parameter.
   * @param string|null $key_value
   *   Key value.
   *
   * @return mixed|null
   *   The value or null.
   */
  protected function getValuesFromDefinition(string $value_parameter, string $key_value = '') {
    if (is_array($this->pluginDefinition) && isset($this->pluginDefinition[$value_parameter])) {
      if ($key_value !== '') {
        return $this->pluginDefinition[$value_parameter][$key_value] ?? NULL;
      }
      return $this->pluginDefinition[$value_parameter] ?? NULL;
    }

    if (($this->pluginDefinition instanceof PluginDefinitionInterface) && isset($this->pluginDefinition->{$value_parameter})) {
      if ($key_value !== '') {
        return $this->pluginDefinition->{$value_parameter}[$key_value] ?? NULL;
      }
      return $this->pluginDefinition->{$value_parameter} ?? NULL;
    }

    return NULL;
  }

  /**
   * Build the array to pass to Drupal report.
   *
   * @param mixed $title
   *   Title.
   * @param mixed $value
   *   Value.
   * @param mixed $description
   *   Description.
   * @param mixed $severity
   *   Severity.
   *
   * @return array
   *   The array to send to Drupal report.
   */
  protected function buildInsightForDrupalReport($title, $value, $description, $severity = REQUIREMENT_WARNING): array {
    $build = [];
    if ($title) {
      $build['title'] = 'Xray Audit: ' . $title;
    }
    if ($value) {
      $build['value'] = $value;
    }
    if ($description) {
      $build['description'] = $description;
    }
    if (is_int($severity)) {
      $build['severity'] = $severity;
    }

    return $build;
  }

  /**
   * Get the xray plugin instance.
   *
   * @param string $plugin_id
   *   Plugin id.
   * @param string $operation
   *   Operation.
   *
   * @return \Drupal\xray_audit\Plugin\XrayAuditTaskPluginInterface|null
   *   Return the plugin instance.
   */
  protected function getInstancedPlugin(string $plugin_id, string $operation): ?XrayAuditTaskPluginInterface {
    if ($this->getTaskPluginDefinition($plugin_id, $operation) === NULL) {
      return NULL;
    }

    $plugin_manger = $this->pluginManagerTaskXray;
    /** @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginInterface $instance */
    $instance = $plugin_manger->createInstance($plugin_id);
    return $instance;
  }

  /**
   * Get the task plugin definition.
   *
   * @param string $plugin_id
   *   Plugin id.
   * @param string $operation
   *   Operation.
   *
   * @return array|null
   *   Return the plugin definition.
   */
  protected function getTaskPluginDefinition(string $plugin_id, string $operation): ?array {
    $plugin_manger = $this->pluginManagerTaskXray;
    $plugin_definitions = $plugin_manger->getDefinitions();
    if (!isset($plugin_definitions[$plugin_id]) || !isset($plugin_definitions[$plugin_id]['operations'][$operation])) {
      return NULL;
    }
    return $plugin_definitions[$plugin_id];
  }

  /**
   * Get the path report.
   *
   * @param string $plugin_id
   *   Plugin id.
   * @param string $operation
   *   Operation.
   *
   * @return string|null
   *   Return the path report.
   */
  protected function getPathReport(string $plugin_id, string $operation): ?string {
    $path_info = $this->pluginRepository->getOnePluginTaskRouteInfo($plugin_id, $operation);
    return $path_info['url'] ?? NULL;
  }

  /**
   * Build the array to pass to the insight configuration settings form.
   *
   * @return array
   *   The array to send to Settings form.
   */
  public function buildInsightForSettings(Config $config): array {
    $build = [
      '#type' => 'details',
      '#title' => $this->label() ?? '',
      '#open' => TRUE,
    ];

    $build[$this->pluginId] = [
      '#type' => 'checkbox',
      '#title' => $this->label() ? $this->t('Exclude @insight', ['@insight' => $this->label()]) : '',
      '#default_value' => $config->get('excluded_insights.' . $this->pluginId),
    ];

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function submitInsightSettings(FormStateInterface $form_state, Config $config) {
    $value = $form_state->getValue($this->pluginId);
    $config->set('excluded_insights.' . $this->pluginId, $value);
  }

}
