<?php

namespace Drupal\xray_audit\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\xray_audit\Services\PluginRepositoryInterface;
use Symfony\Component\Routing\Route;

/**
 * Checks access for operations based on module dependencies.
 */
class OperationAccessCheck implements AccessInterface {

  /**
   * Plugin repository.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Constructs an OperationAccessCheck object.
   *
   * @param \Drupal\xray_audit\Services\PluginRepositoryInterface $plugin_repository
   *   The plugin repository service.
   */
  public function __construct(PluginRepositoryInterface $plugin_repository) {
    $this->pluginRepository = $plugin_repository;
  }

  /**
   * Checks access based on operation dependencies.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route to check against.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The parametrized route.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Route $route, RouteMatchInterface $route_match) {
    // Get the operation ID from the route parameter.
    $task_operation = $route_match->getParameter('task_operation');

    if (empty($task_operation)) {
      return AccessResult::forbidden()->addCacheContexts(['route']);
    }

    // Normalize the operation ID (replace dashes with underscores).
    $task_operation = str_replace('-', '_', $task_operation);

    // Get the task plugin instance.
    $task_plugin_id = $route->getDefault('task_plugin_id');
    if (empty($task_plugin_id)) {
      // Try to extract from route name if not in defaults.
      $route_name = $route_match->getRouteName();
      if (preg_match('/^xray_audit\.task\.([^.]+)\./', $route_name, $matches)) {
        $task_plugin_id = $matches[1];
      }
    }

    if (empty($task_plugin_id)) {
      return AccessResult::forbidden()->addCacheContexts(['route']);
    }

    $task_plugin = $this->pluginRepository->getInstancePluginTask($task_plugin_id);

    if (!$task_plugin) {
      return AccessResult::forbidden()->addCacheContexts(['route']);
    }

    // Check if operation dependencies are met.
    if ($task_plugin->isOperationAvailable($task_operation)) {
      return AccessResult::allowed()->addCacheContexts(['route']);
    }

    return AccessResult::forbidden()->addCacheContexts(['route']);
  }

}
