<?php

namespace Drupal\xray_audit\Controller;

use Drupal\Core\Url;

/**
 * Trait to build items for the group.
 */
trait XrayAuditBuildTaskItemsInControllerTrait {

  /**
   * Build items for the group.
   *
   * @param string $group_definition_id
   *   Group id.
   *
   * @return array
   *   Build array.
   */
  public function buildTaskItems(string $group_definition_id): array {
    $build = [];

    $group_plugin_instance = $this->pluginRepository->getInstancePluginGroup($group_definition_id);

    if ($group_plugin_instance === NULL) {
      return ['#markup' => $this->t('No tasks are available for this group.')];
    }

    $task_definitions = $group_plugin_instance->getPluginTaskDefinitionsThisGroup();

    if (empty($task_definitions)) {
      return $build;
    }

    foreach ($task_definitions as $task_definition) {

      if (empty($task_definition['operations'])) {
        continue;
      }

      if (empty($task_definition['local_task'])) {
        $this->buildAllOperations($build, $task_definition);
        continue;
      }

      $this->buildLocalTaskCase($build, $task_definition);
    }

    return $build;
  }

  /**
   * Add items to the build array in the local task case (tabs).
   *
   * @param array $build
   *   Build array.
   * @param array $task_definition
   *   Task definition.
   */
  protected function buildLocalTaskCase(array &$build, array $task_definition) {
    // Get the task plugin instance to check operation availability.
    $task_plugin = $this->pluginRepository->getInstancePluginTask($task_definition['id']);

    // Find the first available operation (checking dependencies).
    $operation_id = NULL;
    foreach ($task_definition['operations'] as $op_id => $operation) {
      // If plugin is unavailable, use first operation (backwards compatibility).
      // If plugin is available, check if operation dependencies are met.
      if (!$task_plugin || $task_plugin->isOperationAvailable($op_id)) {
        $operation_id = $op_id;
        break;
      }
    }

    // If no available operation found, don't add the task.
    if ($operation_id === NULL) {
      return;
    }

    $build[] = $this->buildTaskItem($task_definition['label'] ?? '', $this->getTaskRouteName($task_definition['id'], $operation_id), (string) ($task_definition['description'] ?? ''));
  }

  /**
   * Add items to the build array when all operations work independently.
   *
   * @param array $build
   *   Build array.
   * @param array $task_definition
   *   Task definition.
   */
  protected function buildAllOperations(array &$build, array $task_definition) {
    // Get the task plugin instance to check operation availability.
    $task_plugin = $this->pluginRepository->getInstancePluginTask($task_definition['id']);

    foreach ($task_definition['operations'] as $operation_id => $operation) {
      if (!empty($operation['not_show'])) {
        continue;
      }

      // Check if operation dependencies are met.
      if ($task_plugin && !$task_plugin->isOperationAvailable($operation_id)) {
        continue;
      }

      $build[] = $this->buildTaskItem($operation['label'] ?? '', $this->getTaskRouteName($task_definition['id'], $operation_id), (string) ($operation['description'] ?? ''));
    }
  }

  /**
   * Get the route name of the task.
   *
   * @param string $task_id
   *   Task id.
   * @param string $operation_id
   *   Operation id.
   *
   * @return string
   *   Route name.
   */
  protected function getTaskRouteName($task_id, $operation_id): string {
    return $this->pluginRepository->getOnePluginTaskRouteInfo($task_id, $operation_id)['route_name'] ?? '';
  }

  /**
   * Build task item.
   *
   * @param string $title
   *   Title of the operation.
   * @param string $routeName
   *   Route name.
   * @param string $description
   *   Description of the operation.
   *
   * @return array
   *   Task item.
   */
  protected function buildTaskItem(string $title, string $routeName, string $description): array {
    return [
      'title' => $title,
      'url' => Url::fromRoute($routeName),
      'localized_options' => [],
      'description' => $description,
    ];
  }

}
