<?php

namespace Drupal\xray_audit\Plugin;

use Drupal\Component\Plugin\Definition\PluginDefinitionInterface;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\xray_audit\Form\SettingsForm;
use Drupal\xray_audit\Services\CsvDownloadManagerInterface;
use Drupal\xray_audit\Services\PluginRepositoryInterface;
use Drupal\xray_audit\XrayAuditTaskCsvDownloadTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for xray_audit_task_plugin plugins.
 */
abstract class XrayAuditTaskPluginBase extends PluginBase implements XrayAuditTaskPluginInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;
  use XrayAuditTaskCsvDownloadTrait;


  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Service "xray_audit.plugin_repository".
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Service "xray_audit.csv_download_manager".
   *
   * @var \Drupal\xray_audit\Services\CsvDownloadManagerInterface
   */
  protected $csvDownloadManager;

  /**
   * Service "config.factory".
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Service "module_handler".
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Configuration settings for Xray Audit.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $xrayAuditConfig;

  /**
   * Flag to track if CSV has been processed for current request.
   *
   * Prevents double CSV processing when both parent and child classes
   * call handleCsv().
   *
   * @var bool
   */
  protected bool $csvProcessed = FALSE;

  /**
   * Constructs a \Drupal\Component\Plugin\PluginBase object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   Database connection.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   Database connection.
   * @param \Drupal\xray_audit\Services\PluginRepositoryInterface $pluginRepository
   *   Service "xray_audit.plugin_repository".
   * @param \Drupal\xray_audit\Services\CsvDownloadManagerInterface $csvDownloadManager
   *   Service "xray_audit.csv_download_manager".
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Service "config.factory".
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   Service "module_handler".
   *
   * @phpstan-consistent-constructor
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database,
    LanguageManagerInterface $language_manager,
    PluginRepositoryInterface $pluginRepository,
    CsvDownloadManagerInterface $csvDownloadManager,
    ConfigFactoryInterface $config_factory,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
    $this->languageManager = $language_manager;
    $this->pluginRepository = $pluginRepository;
    $this->csvDownloadManager = $csvDownloadManager;
    $this->configFactory = $config_factory;
    $this->moduleHandler = $module_handler;
    $this->xrayAuditConfig = $this->configFactory->get(SettingsForm::SETTINGS);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('database'),
      $container->get('language_manager'),
      $container->get('xray_audit.plugin_repository'),
      $container->get('xray_audit.csv_download_manager'),
      $container->get('config.factory'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    // Cast the label to a string since it is a TranslatableMarkup object.
    return (string) $this->getValuesFromDefinition('label');
  }

  /**
   * {@inheritdoc}
   */
  public function getGroup() {
    return $this->getValuesFromDefinition('group');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    // Cast the label to a string since it is a TranslatableMarkup object.
    return (string) ($this->getValuesFromDefinition('description') ?? '');
  }

  /**
   * {@inheritdoc}
   */
  public function getSort() {
    return $this->getValuesFromDefinition('sort') ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getOperations() {
    return $this->getValuesFromDefinition('operations') ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getBatchClass(string $batch_id) {
    return $this->getValuesFromDefinition('batches', $batch_id);
  }

  /**
   * {@inheritdoc}
   */
  public function isLocalTaskCase(): bool {
    return (bool) ($this->getValuesFromDefinition('local_task') ?? FALSE);
  }

  /**
   * Checks if an operation's dependencies are met.
   *
   * @param string $operation_id
   *   The operation ID to check.
   *
   * @return bool
   *   TRUE if all dependencies are met, FALSE otherwise.
   */
  public function isOperationAvailable(string $operation_id): bool {
    $operations = $this->getOperations();

    if (!isset($operations[$operation_id])) {
      return FALSE;
    }

    $operation = $operations[$operation_id];

    // If no dependencies are defined, the operation is available.
    if (empty($operation['dependencies'])) {
      return TRUE;
    }

    // Check if all required modules are enabled.
    foreach ($operation['dependencies'] as $module) {
      if (!$this->moduleHandler->moduleExists($module)) {
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Process CSV download for the given operation.
   *
   * This method should be called by child classes that override
   * buildDataRenderArray() to ensure consistent CSV handling.
   *
   * @param string $operation
   *   The operation name.
   * @param array $data
   *   Optional data to pass to handleCsv.
   * @param array $build
   *   The build array to add the download button to (passed by reference).
   *
   * @return bool
   *   TRUE if CSV was processed, FALSE otherwise.
   */
  protected function processCsvDownload(string $operation, array $data, array &$build): bool {
    // Check if the current operation is configured for download.
    $current_operation_definition = $this->getOperations()[$operation] ?? NULL;
    $operation_supports_download = !empty($current_operation_definition['download']) && $current_operation_definition['download'] === TRUE;

    if (!$operation_supports_download) {
      return FALSE;
    }

    // Handle CSV generation and download.
    $this->handleCsv($operation, $data);

    // Add download button only once.
    if (empty($build['download_button'])) {
      $build['download_button'] = $this->createRenderableLink($operation, (string) $this->t('Download CSV'));
      $build['download_button']['#weight'] = 9;
    }

    return TRUE;
  }

  /**
   * Get values from plugin definition.
   *
   * @param string $value_parameter
   *   Value parameter.
   * @param string|null $key_value
   *   Key value.
   *
   * @return mixed|null
   *   The value or null.
   */
  protected function getValuesFromDefinition(string $value_parameter, ?string $key_value = NULL) {
    if (is_array($this->pluginDefinition) && isset($this->pluginDefinition[$value_parameter])) {
      if ($key_value) {
        return $this->pluginDefinition[$value_parameter][$key_value] ?? NULL;
      }
      return $this->pluginDefinition[$value_parameter] ?? NULL;
    }

    if (($this->pluginDefinition instanceof PluginDefinitionInterface) && isset($this->pluginDefinition->{$value_parameter})) {
      if ($key_value) {
        return $this->pluginDefinition->{$value_parameter}[$key_value] ?? NULL;
      }
      return $this->pluginDefinition->{$value_parameter} ?? NULL;
    }

    return NULL;
  }

  /**
   * Get headers for the table.
   *
   * @param string $operation
   *   Operation name.
   */
  public function getHeaders(string $operation = ''): array {
    return [];
  }

  /**
   * Get rows for the table.
   *
   * @param string $operation
   *   Operation name.
   *
   * @return array
   *   Rows.
   */
  public function getRows(string $operation = ''): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    return [
      'header_table' => $this->getHeaders($operation),
      'results_table' => $this->getRows($operation),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildDataRenderArray(array $data, string $operation = '') {

    $description = '';
    $operation_data = $this->getOperations()[$operation] ?? NULL;
    if ($operation_data) {
      $description = $operation_data['description'] ?? '';
    }

    $header = $data['header_table'] ?? [];
    $rows = $data['results_table'] ?? [];
    $extended_data_render = $data['extended_data_render'] ?? [];

    $build = [];

    if ($description) {
      $build['#markup'] = '<p>' . $description . '</p>';
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#weight' => 20,
    ] + $extended_data_render;

    // Check if the current operation is configured for download.
    $current_operation_definition = $this->getOperations()[$operation] ?? NULL;
    $operation_supports_download = !empty($current_operation_definition['download']) && $current_operation_definition['download'] === TRUE;

    if ($operation_supports_download) {
      // Only handle CSV if not already processed by child class.
      if (!$this->csvProcessed) {
        // The handleCsv method checks if a download is triggered and sends the CSV.
        // It should be called before building other parts of the page if it might exit.
        // $rows here are $data['results_table'].
        $this->handleCsv($operation, $rows);
      }

      // Add download button only once (child class may have already added it).
      if (empty($build['download_button'])) {
        // The createRenderableLink method from the trait will generate the appropriate URL
        // that handleCsv will detect.
        $build['download_button'] = $this->createRenderableLink($operation, (string) $this->t('Download CSV'));
        $build['download_button']['#weight'] = 9;
      }
    }

    return $build;
  }

}
