<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentAccessControl;

use Drupal\user\Entity\Role;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "queries_data_roles",
 *   label = @Translation("Role reports"),
 *   description = @Translation("Reports about roles and permissions."),
 *   group = "content_access_control",
 *   sort = 1,
 *   local_task = 1,
 *   operations = {
 *      "role_list" = {
 *          "label" = "Role list",
 *          "description" = "Listing all user roles in the system.",
 *           "download" = TRUE
 *      },
 *      "role_permission" = {
 *          "label" = "List of Permissions per role",
 *          "description" = "Permissions assigned to each role.",
 *           "download" = TRUE
 *       }
 *    },
 *   dependencies = {"user"}
 * )
 */
class XrayAuditQueryTaskRolesPlugin extends XrayAuditTaskPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    switch ($operation) {
      case 'role_list':
        return [
          $this->t('Id'),
          $this->t('Label'),
        ];

      case 'role_permission':
        return [
          $this->t('Role'),
          $this->t('Permissions'),
        ];
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    switch ($operation) {
      case 'role_list':
        return $this->rolesDescription();

      case 'role_permission':
        return $this->permissionsPerRole();

    }
    return [];
  }

  /**
   * Description roles.
   *
   * @return array
   *   Render array.
   */
  public function rolesDescription() {
    $resultTable = [];
    $roles = Role::loadMultiple();
    foreach ($roles as $role) {
      $resultTable[$role->get('id')] = [
        $role->get('id'),
        $role->get('label'),
      ];
    }

    ksort($resultTable);
    return $resultTable;
  }

  /**
   * Permissions and roles.
   *
   * @return array
   *   Render array.
   */
  public function permissionsPerRole() {
    $resultTable = [];
    $roles = Role::loadMultiple();
    foreach ($roles as $role) {
      $resultTable[$role->id()] = [
        $role->get('label'),
        implode(', ', $role->get('permissions')),
      ];
    }

    ksort($resultTable);
    return $resultTable;
  }

}
