<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentAccessControl;

use Drupal\user\Entity\Role;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "queries_data_user",
 *   label = @Translation("User reports"),
 *   description = @Translation("Reports about the users."),
 *   group = "content_access_control",
 *   local_task = 1,
 *   sort = 5,
 *   operations = {
 *     "user_count_role" = {
 *           "label" = "Grouped by roles",
 *           "description" = "Number of Users grouped by roles.",
 *           "download" = TRUE
 *        },
 *      "user_count_status" = {
 *          "label" = "Grouped by status",
 *          "description" = "Number of Users grouped by status.",
 *           "download" = TRUE
 *       },
 *      "user_count_activity" = {
 *          "label" = "Grouped by activity",
 *          "description" = "Number of Users grouped by activity.",
 *           "download" = TRUE
 *       }
 *    },
 *   dependencies = {"user"}
 * )
 */
class XrayAuditQueryTaskUserPlugin extends XrayAuditTaskPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    switch ($operation) {
      case 'user_count_status':
        return [$this->t('Status'), $this->t('Number of users')];

      case 'user_count_activity':
        return [$this->t('Last access'), $this->t('Number of users')];

      case 'user_count_role':
        return [$this->t('ID'), $this->t('Label'), $this->t('Count')];
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    switch ($operation) {
      case 'user_count_status':
        return $this->getUserCountByStatus();

      case 'user_count_activity':
        return $this->getUsersByActivity();

      case 'user_count_role':
        return $this->getUsersByRole();
    }
    return [];
  }

  /**
   * Get users count by status.
   */
  protected function getUserCountByStatus(): array {
    $aggregateQuery = $this->entityTypeManager->getStorage('user')->getAggregateQuery();

    $resultTable = $aggregateQuery->accessCheck(FALSE)
      ->aggregate('uid', 'count')
      ->condition('uid', 0, '!=')
      ->groupBy('status')
      ->execute();

    $total = 0;
    $rows = [];

    foreach ($resultTable as $row) {
      $status = $row['status'] == 1 ? $this->t('Active') : $this->t('Blocked');
      $rows[] = [$status, $row['uid_count']];
      $total += $row['uid_count'];
    }

    $rows[] = [$this->t('Total'), $total];

    return $rows;
  }

  /**
   * Get users by activity.
   */
  protected function getUsersByActivity(): array {
    $userStorage = $this->entityTypeManager->getStorage('user');
    $current_time = time();
    $one_month_seconds = 2592000;

    $user_count_6_m = $userStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('status', 1)
      ->condition('access', $current_time - ($one_month_seconds * 6), '>')
      ->count()
      ->execute();

    $user_count_6_12_m = $userStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('status', 1)
      ->condition('access', $current_time - ($one_month_seconds * 6), '<')
      ->condition('access', $current_time - ($one_month_seconds * 12), '>')
      ->count()
      ->execute();

    $user_count_more_12_m = $userStorage->getQuery()
      ->accessCheck(FALSE)
      ->condition('status', 1)
      ->condition('access', $current_time - ($one_month_seconds * 12), '<')
      ->count()
      ->execute();

    return [
      [$this->t('Last 6 months'), $user_count_6_m],
      [$this->t('Last 6 to 12 months'), $user_count_6_12_m],
      [$this->t('More than 12 months'), $user_count_more_12_m],
    ];
  }

  /**
   * Get users by role.
   */
  protected function getUsersByRole(): array {
    $roles = Role::loadMultiple();
    $roleCount = [];

    foreach ($roles as $role) {
      $roleCount[$role->id()] = [
        $role->id(),
        $role->label(),
        0,
      ];
    }

    ksort($roleCount);

    $query = $this->database->select('user__roles', 'ur');

    $query->addField('ur', 'roles_target_id', 'role');
    $query->addExpression('COUNT(1)', 'count');
    $query->groupBy('ur.roles_target_id');
    $query->orderBy('ur.roles_target_id');
    $results = $query->execute();

    $result = $results->fetchAll();
    foreach ($result as $row) {
      if (isset($roleCount[$row->role])) {
        $roleCount[$row->role][2] = $row->count;
      }
    }

    return array_values($roleCount);
  }

}
