<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric;

use Drupal\Component\Plugin\DependentPluginInterface;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FormatterPluginManager;
use Drupal\Core\Link;
use Drupal\field_ui\FieldUI;
use Drupal\image\ImageStyleStorageInterface;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;
use Drupal\xray_audit\Services\EntityArchitectureInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an image styles report.
 *
 * @XrayAuditTaskPlugin(
 *   id = "queries_data_image_styles",
 *   label = @Translation("Image Styles Report"),
 *   description = @Translation("Reports on the image styles used in the site"),
 *   group = "content_metric",
 *   sort = 7,
 *   local_task = 1,
 *   operations = {
 *      "image_styles_summary" = {
 *          "label" = "Image Styles Summary",
 *          "description" = "Overview of all image styles with usage indicators.",
 *          "download" = TRUE
 *       },
 *      "image_styles_display_mode_usage" = {
 *          "label" = "Image Styles in Display Modes",
 *          "description" = "Image styles directly used in entity display configurations.",
 *          "download" = TRUE
 *       },
 *      "image_styles_responsive_usage" = {
 *          "label" = "Image Styles in Responsive Image Styles",
 *          "description" = "Image styles configured within responsive image style definitions.",
 *          "download" = TRUE,
 *          "dependencies" = {"responsive_image"},
 *       },
 *      "responsive_styles_display_mode_usage" = {
 *          "label" = "Responsive Image Styles in Display Modes",
 *          "description" = "Responsive image styles used in entity display configurations.",
 *          "download" = TRUE,
 *          "dependencies" = {"responsive_image"},
 *       },
 *      "image_styles_ckeditor_usage" = {
 *          "label" = "Image Styles in CKEditor",
 *          "description" = "Image styles available in CKEditor configurations.",
 *          "download" = TRUE,
 *          "dependencies" = {"ckeditor5"}
 *       },
 *    },
 *   dependencies = {"media", "image"},
 * )
 */
class XrayAuditQueryTaskImageStylesPlugin extends XrayAuditTaskPluginBase {

  /**
   * The entity architecture service.
   *
   * @var \Drupal\xray_audit\Services\EntityArchitectureInterface
   */
  protected EntityArchitectureInterface $entityArchitecture;

  /**
   * The bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected EntityTypeBundleInfoInterface $bundleInfo;

  /**
   * The image style storage.
   *
   * @var \Drupal\image\ImageStyleStorageInterface
   */
  protected ImageStyleStorageInterface $imageStyleStorage;

  /**
   * The responsive image style storage.
   *
   * Only available when responsive_image module is enabled.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|null
   */
  protected ?EntityStorageInterface $responsiveImageStyleStorage = NULL;

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * The field formatter manager service.
   *
   * @var \Drupal\Core\Field\FormatterPluginManager
   */
  protected FormatterPluginManager $fieldFormatterManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityArchitecture = $container->get('xray_audit.entity_architecture');
    $instance->bundleInfo = $container->get('entity_type.bundle.info');
    $instance->imageStyleStorage = $instance->entityTypeManager->getStorage('image_style');

    // Only load responsive image style storage if the module is enabled.
    if ($instance->moduleHandler->moduleExists('responsive_image')) {
      $instance->responsiveImageStyleStorage = $instance->entityTypeManager->getStorage('responsive_image_style');
    }

    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->fieldFormatterManager = $container->get('plugin.manager.field.formatter');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    $cid = $this->getPluginId() . ':' . $operation;
    $data = $this->pluginRepository->getCachedData($cid);
    $cache_tags = [];

    if (!empty($data) && is_array($data)) {
      return $data;
    }

    switch ($operation) {
      case 'image_styles_display_mode_usage':
        $data = [
          'header_table' => $this->getHeaders($operation),
          'results_table' => $this->getRows($operation),
        ];
        $cache_tags = [
          'config:entity_view_display_list',
          'config:image_style_list',
        ];
        break;

      case 'image_styles_responsive_usage':
        $data = [
          'header_table' => $this->getHeaders($operation),
          'results_table' => $this->getRows($operation),
        ];
        $cache_tags = [
          'config:responsive_image_style_list',
          'config:image_style_list',
        ];
        break;

      case 'responsive_styles_display_mode_usage':
        $data = [
          'header_table' => $this->getHeaders($operation),
          'results_table' => $this->getRows($operation),
        ];
        $cache_tags = [
          'config:entity_view_display_list',
          'config:responsive_image_style_list',
        ];
        break;

      case 'image_styles_ckeditor_usage':
        $data = [
          'header_table' => $this->getHeaders($operation),
          'results_table' => $this->getRows($operation),
        ];
        $cache_tags = [
          'config:filter_format_list',
          'config:entity_view_display_list',
          'config:image_style_list',
        ];
        break;

      case 'image_styles_summary':
        $data = [
          'header_table' => $this->getHeaders($operation),
          'results_table' => $this->getRows($operation),
        ];
        $cache_tags = [
          'config:image_style_list',
          'config:entity_view_display_list',
          'config:responsive_image_style_list',
        ];
        break;
    }

    $this->pluginRepository->setCacheTagsInv($cid, $data, $cache_tags);

    return $data;
  }

  /**
   * Retrieves all image styles.
   *
   * @return array
   *   An array of image style entities ids.
   */
  protected function getImageStylesIds(): array {
    $query = $this->imageStyleStorage->getQuery();
    $query->accessCheck(FALSE);
    $ids = $query->execute();

    return $ids;
  }

  /**
   * Retrieves all image styles.
   *
   * @return array
   *   An array of responsive image style entities ids.
   */
  protected function getResponsiveImageStyleIds(): array {
    if ($this->responsiveImageStyleStorage === NULL) {
      return [];
    }

    $query = $this->responsiveImageStyleStorage->getQuery();
    $query->accessCheck(FALSE);
    $ids = $query->execute();

    return $ids;
  }

  /**
   * Gets table headers for a specific operation.
   *
   * @param string $operation
   *   The operation ID.
   *
   * @return array
   *   Array of translatable header strings.
   */
  public function getHeaders(string $operation = ''): array {
    switch ($operation) {
      case 'image_styles_display_mode_usage':
        return [
          $this->t('Image Style ID'),
          $this->t('Image Style Name'),
          $this->t('Entity Type'),
          $this->t('Bundle'),
          $this->t('Field Type'),
          $this->t('Field Name'),
          $this->t('View Mode'),
          $this->t('Formatter'),
        ];

      case 'image_styles_responsive_usage':
        return [
          $this->t('Image Style ID'),
          $this->t('Image Style Name'),
          $this->t('Responsive Image Style ID'),
          $this->t('Responsive Image Style Label'),
          $this->t('Image Mapping Type'),
          $this->t('Breakpoint'),
          $this->t('Multiplier'),
        ];

      case 'responsive_styles_display_mode_usage':
        return [
          $this->t('Responsive Image Style ID'),
          $this->t('Responsive Image Style Name'),
          $this->t('Entity Type'),
          $this->t('Bundle'),
          $this->t('Field Type'),
          $this->t('Field Name'),
          $this->t('View Mode'),
          $this->t('Formatter'),
        ];

      case 'image_styles_ckeditor_usage':
        return [
          $this->t('Image Style ID'),
          $this->t('Image Style Name'),
          $this->t('CKEditor Config ID'),
          $this->t('CKEditor Config Name'),
          $this->t('View Mode'),
        ];

      case 'image_styles_summary':
        return [
          $this->t('Image Style ID'),
          $this->t('Image Style Name'),
          $this->t('Used in Display Modes'),
          $this->t('Used in Responsive Image Styles'),
          $this->t('Used in CKEditor'),
          $this->t('Result'),
        ];
    }

    return [];
  }

  /**
   * Gets table rows for a specific operation.
   *
   * @param string $operation
   *   The operation ID.
   *
   * @return array
   *   Array of row arrays.
   */
  public function getRows(string $operation = ''): array {
    switch ($operation) {
      case 'image_styles_display_mode_usage':
        return $this->getImageStylesDisplayModeUsageRows();

      case 'image_styles_responsive_usage':
        return $this->getImageStylesResponsiveUsageRows();

      case 'responsive_styles_display_mode_usage':
        return $this->getResponsiveStylesDisplayModeUsageRows();

      case 'image_styles_ckeditor_usage':
        return $this->getImageStylesCkeditorUsageRows();

      case 'image_styles_summary':
        return $this->getImageStylesSummaryRows();
    }

    return [];
  }

  /**
   * Gets the rows for the 'image_styles_display_mode_usage' operation.
   *
   * This operation shows where image styles are directly used in entity
   * display configurations. It excludes responsive image style settings.
   *
   * Performance optimizations:
   * - Batch loads all image styles once instead of individual loads
   * - Caches field definitions per entity_type/bundle
   * - Uses single query to load all displays
   * - Tracks unique image style IDs for batch loading
   *
   * @return array
   *   The table rows with display mode usage data. Each row contains:
   *   - Image Style ID
   *   - Image Style Name
   *   - Entity Type
   *   - Bundle
   *   - Field Type
   *   - Field Name
   *   - View Mode
   *   - Formatter
   */
  private function getImageStylesDisplayModeUsageRows(): array {
    $result_table = [];
    $all_image_style_ids = [];

    // Load all entity view displays - necessary to inspect configurations.
    $view_displays = $this->entityTypeManager
      ->getStorage('entity_view_display')
      ->loadMultiple();

    // Regex pattern to match image style settings (excludes responsive_*_style).
    $pattern = '^(?!responsive_([^_]+_)?style($|_.+)).+_style($|_.+)';

    // Cache for field definitions to avoid repeated lookups.
    $field_definitions_cache = [];

    // First pass: collect all usage data and track all image style IDs.
    foreach ($view_displays as $view_display) {
      $entity_type_id = $view_display->getTargetEntityTypeId();
      $bundle = $view_display->getTargetBundle();
      $view_mode = $view_display->getMode();
      $components = $view_display->getComponents();

      foreach ($components as $field_name => $component) {
        $formatter_type = $component['type'] ?? '';
        $settings = $component['settings'] ?? [];

        // Find image style settings in component.
        $image_style_ids = $this->extractImageStyleIdsFromSettings($settings, $pattern);

        if (!empty($image_style_ids)) {
          // Get field definition (cached per entity_type/bundle).
          $cache_key = "$entity_type_id:$bundle";
          if (!isset($field_definitions_cache[$cache_key])) {
            $field_definitions_cache[$cache_key] = $this->entityFieldManager
              ->getFieldDefinitions($entity_type_id, $bundle);
          }

          $field_type = $field_definitions_cache[$cache_key][$field_name]->getType() ?? 'unknown';

          foreach ($image_style_ids as $image_style_id) {
            // Track for batch loading.
            $all_image_style_ids[$image_style_id] = $image_style_id;

            // Store row data (we'll add labels after batch load).
            $result_table[] = [
              'style_id' => $image_style_id,
              'entity_type' => $entity_type_id,
              'bundle' => $bundle,
              'field_type' => $field_type,
              'field_name' => $field_name,
              'view_mode' => $view_mode,
              'formatter' => $formatter_type,
            ];
          }
        }
      }
    }

    // Batch load all image styles at once (performance optimization).
    $image_styles = $this->imageStyleStorage->loadMultiple($all_image_style_ids);

    // Second pass: add image style labels to rows.
    $final_result = [];
    foreach ($result_table as $row_data) {
      $style_id = $row_data['style_id'];
      if (isset($image_styles[$style_id])) {
        $final_result[] = [
          $style_id,
          $image_styles[$style_id]->label(),
          $row_data['entity_type'],
          $row_data['bundle'],
          $row_data['field_type'],
          $row_data['field_name'],
          $row_data['view_mode'],
          $row_data['formatter'],
        ];
      }
    }

    // Sort by image style ID (column 0).
    usort($final_result, function ($a, $b) {
      return strcmp($a[0], $b[0]);
    });

    return $final_result;
  }

  /**
   * Extracts image style IDs from component settings.
   *
   * Uses regex pattern matching to identify settings keys that contain
   * image style references, then extracts the image style ID values.
   *
   * @param array $settings
   *   The component settings array from an entity view display component.
   * @param string $pattern
   *   The regex pattern to match setting keys (without delimiters).
   *   Example: '^(?!responsive_([^_]+_)?style($|_.+)).+_style($|_.+)'.
   *
   * @return array
   *   Array of unique image style IDs found in settings.
   */
  private function extractImageStyleIdsFromSettings(array $settings, string $pattern): array {
    $image_style_ids = [];

    foreach ($settings as $key => $value) {
      // Match setting key against pattern.
      if (preg_match('/' . $pattern . '/', $key) && !empty($value)) {
        // The value is typically the image style ID.
        if (is_string($value)) {
          $image_style_ids[] = $value;
        }
        elseif (is_array($value)) {
          // Some formatters may have nested arrays of styles.
          foreach ($value as $nested_value) {
            if (is_string($nested_value)) {
              $image_style_ids[] = $nested_value;
            }
          }
        }
      }
    }

    // Return unique IDs only.
    return array_unique($image_style_ids);
  }

  /**
   * Gets the rows for the 'image_styles_responsive_usage' operation.
   *
   * This operation shows which image styles are configured within responsive
   * image style definitions. It parses the image style mappings to identify
   * all breakpoints, multipliers, and mapping types.
   *
   * @return array
   *   The table rows with responsive image style usage data. Each row contains:
   *   - Image Style ID
   *   - Image Style Name
   *   - Responsive Image Style ID
   *   - Responsive Image Style Label
   *   - Image Mapping Type
   *   - Breakpoint
   *   - Multiplier
   */
  private function getImageStylesResponsiveUsageRows(): array {
    $result_table = [];

    // Return empty if responsive_image module is not enabled.
    if ($this->responsiveImageStyleStorage === NULL) {
      return $result_table;
    }

    // Load all responsive image styles.
    /** @var \Drupal\responsive_image\ResponsiveImageStyleInterface[] $responsive_styles */
    $responsive_styles = $this->responsiveImageStyleStorage->loadMultiple();

    foreach ($responsive_styles as $responsive_style) {
      $responsive_id = $responsive_style->id();
      $responsive_label = $responsive_style->label();
      $mappings = $responsive_style->getImageStyleMappings();

      foreach ($mappings as $mapping) {
        $mapping_type = $mapping['image_mapping_type'] ?? '';
        $breakpoint_id = $mapping['breakpoint_id'] ?? '';
        $multiplier = $mapping['multiplier'] ?? '';

        // Extract image styles from mapping.
        // The structure can be either 'sizes_image_styles' array or direct mapping.
        $image_styles = $mapping['image_mapping']['sizes_image_styles']
          ?? (array) ($mapping['image_mapping'] ?? []);

        foreach ($image_styles as $image_style_id) {
          // Skip if not a string (could be empty or other type).
          if (!is_string($image_style_id) || empty($image_style_id)) {
            continue;
          }

          // Load image style to get label.
          $image_style = $this->imageStyleStorage->load($image_style_id);
          if ($image_style) {
            $result_table[] = [
              $image_style_id,
              $image_style->label(),
              $responsive_id,
              $responsive_label,
              $mapping_type,
              $breakpoint_id,
              $multiplier,
            ];
          }
        }
      }
    }

    return $result_table;
  }

  /**
   * Gets the rows for the 'responsive_styles_display_mode_usage' operation.
   *
   * This operation shows where responsive image styles are used in entity
   * display configurations. It uses the same pattern-matching approach as
   * Operation 2.1 but with a different regex pattern to match responsive
   * image style settings.
   *
   * Performance optimizations:
   * - Reuses field definition caching strategy
   * - Batch loads all responsive image styles once
   * - Single pass through all displays
   *
   * @return array
   *   The table rows with responsive image style display mode usage. Each row:
   *   - Responsive Image Style ID
   *   - Responsive Image Style Name
   *   - Entity Type
   *   - Bundle
   *   - Field Type
   *   - Field Name
   *   - View Mode
   *   - Formatter
   */
  private function getResponsiveStylesDisplayModeUsageRows(): array {
    $result_table = [];
    $all_responsive_style_ids = [];

    // Load all entity view displays.
    $view_displays = $this->entityTypeManager
      ->getStorage('entity_view_display')
      ->loadMultiple();

    // Regex pattern to match responsive image style settings.
    $pattern = '^responsive_([^_]+_)?style($|_.+)';

    // Cache for field definitions to avoid repeated lookups.
    $field_definitions_cache = [];

    // First pass: collect all usage data and track all responsive style IDs.
    foreach ($view_displays as $view_display) {
      $entity_type_id = $view_display->getTargetEntityTypeId();
      $bundle = $view_display->getTargetBundle();
      $view_mode = $view_display->getMode();
      $components = $view_display->getComponents();

      foreach ($components as $field_name => $component) {
        $formatter_type = $component['type'] ?? '';
        $settings = $component['settings'] ?? [];

        // Find responsive image style settings in component.
        $responsive_style_ids = $this->extractImageStyleIdsFromSettings($settings, $pattern);

        if (!empty($responsive_style_ids)) {
          // Get field definition (cached per entity_type/bundle).
          $cache_key = "$entity_type_id:$bundle";
          if (!isset($field_definitions_cache[$cache_key])) {
            $field_definitions_cache[$cache_key] = $this->entityFieldManager
              ->getFieldDefinitions($entity_type_id, $bundle);
          }

          $field_type = $field_definitions_cache[$cache_key][$field_name]->getType() ?? 'unknown';

          foreach ($responsive_style_ids as $responsive_style_id) {
            // Track for batch loading.
            $all_responsive_style_ids[$responsive_style_id] = $responsive_style_id;

            // Store row data (we'll add labels after batch load).
            $result_table[] = [
              'style_id' => $responsive_style_id,
              'entity_type' => $entity_type_id,
              'bundle' => $bundle,
              'field_type' => $field_type,
              'field_name' => $field_name,
              'view_mode' => $view_mode,
              'formatter' => $formatter_type,
            ];
          }
        }
      }
    }

    // Batch load all responsive image styles at once.
    // Return empty if responsive_image module is not enabled.
    if ($this->responsiveImageStyleStorage === NULL) {
      return [];
    }
    $responsive_styles = $this->responsiveImageStyleStorage->loadMultiple($all_responsive_style_ids);

    // Second pass: add responsive image style labels to rows.
    $final_result = [];
    foreach ($result_table as $row_data) {
      $style_id = $row_data['style_id'];
      if (isset($responsive_styles[$style_id])) {
        $final_result[] = [
          $style_id,
          $responsive_styles[$style_id]->label(),
          $row_data['entity_type'],
          $row_data['bundle'],
          $row_data['field_type'],
          $row_data['field_name'],
          $row_data['view_mode'],
          $row_data['formatter'],
        ];
      }
    }

    return $final_result;
  }

  /**
   * Gets the rows for the 'image_styles_ckeditor_usage' operation.
   *
   * This operation identifies which image styles are available through
   * CKEditor media embed configurations by examining filter formats and
   * their allowed view modes for media entities.
   *
   * Performance optimizations:
   * - Uses filtered queries for view displays (media entity type only)
   * - Processes only relevant view modes found in CKEditor configs
   * - Batch loads image styles after collection
   *
   * @return array
   *   The table rows with CKEditor usage data. Each row contains:
   *   - Image Style ID
   *   - Image Style Name
   *   - CKEditor Config ID (filter format ID)
   *   - CKEditor Config Name (filter format label)
   *   - View Mode
   */
  private function getImageStylesCkeditorUsageRows(): array {
    $result_table = [];
    $all_image_style_ids = [];

    // Load all filter formats.
    $formats = $this->entityTypeManager
      ->getStorage('filter_format')
      ->loadMultiple();

    // Track view modes enabled in CKEditor by format.
    $view_modes_by_format = [];

    // Extract allowed view modes for CKEditor media embedding.
    foreach ($formats as $format) {
      $format_id = $format->id();
      $format_label = $format->label();
      $filters = $format->get('filters');

      if (isset($filters['media_embed'])
          && !empty($filters['media_embed']['settings']['allowed_view_modes'])) {
        $allowed_view_modes = $filters['media_embed']['settings']['allowed_view_modes'];

        foreach ($allowed_view_modes as $view_mode) {
          $view_modes_by_format[$view_mode][] = [
            'format_id' => $format_id,
            'format_label' => $format_label,
          ];
        }
      }
    }

    // Now find image styles used in these view modes.
    foreach ($view_modes_by_format as $view_mode => $formats_info) {
      // Query for view displays: media entity type with this view mode.
      $query = $this->entityTypeManager
        ->getStorage('entity_view_display')
        ->getQuery();
      $query->condition('targetEntityType', 'media');
      $query->condition('mode', $view_mode);
      $query->accessCheck(FALSE);
      $view_display_ids = $query->execute();

      $view_displays = $this->entityTypeManager
        ->getStorage('entity_view_display')
        ->loadMultiple($view_display_ids);

      // Regex pattern to match image style settings (not responsive).
      $pattern = '^(?!responsive_([^_]+_)?style($|_.+)).+_style($|_.+)';

      foreach ($view_displays as $view_display) {
        $components = $view_display->getComponents();

        foreach ($components as $field_name => $component) {
          $settings = $component['settings'] ?? [];

          // Find image styles in this component.
          $image_style_ids = $this->extractImageStyleIdsFromSettings($settings, $pattern);

          foreach ($image_style_ids as $image_style_id) {
            // Track for batch loading.
            $all_image_style_ids[$image_style_id] = $image_style_id;

            // Add a row for each format that uses this view mode.
            foreach ($formats_info as $format_info) {
              $result_table[] = [
                'style_id' => $image_style_id,
                'format_id' => $format_info['format_id'],
                'format_label' => $format_info['format_label'],
                'view_mode' => $view_mode,
              ];
            }
          }
        }
      }
    }

    // Batch load all image styles at once.
    $image_styles = $this->imageStyleStorage->loadMultiple($all_image_style_ids);

    // Add image style labels to rows.
    $final_result = [];
    foreach ($result_table as $row_data) {
      $style_id = $row_data['style_id'];
      if (isset($image_styles[$style_id])) {
        $final_result[] = [
          $style_id,
          $image_styles[$style_id]->label(),
          $row_data['format_id'],
          $row_data['format_label'],
          $row_data['view_mode'],
        ];
      }
    }

    return $final_result;
  }

  /**
   * Gets the rows for the 'image_styles_summary' operation.
   *
   * This operation provides a high-level overview of all image styles and where
   * they are used. It aggregates data from other operations by checking
   * cached data when available, or falls back to running the operations.
   *
   * @return array
   *   The table rows with summary data. Each row contains:
   *   - Image Style ID
   *   - Image Style Name
   *   - Used in Display Modes (Yes/No)
   *   - Used in Responsive Image Styles (Yes/No)
   *   - Used in CKEditor (Yes/No)
   *   - Result (Yes/No - Yes if used anywhere)
   */
  private function getImageStylesSummaryRows(): array {
    $result_table = [];

    // Load all image styles.
    $image_style_ids = $this->getImageStylesIds();
    $image_styles = $this->imageStyleStorage->loadMultiple($image_style_ids);

    // Try to get cached data from dependent operations.
    $display_mode_usage_cid = $this->getPluginId() . ':image_styles_display_mode_usage';
    $responsive_usage_cid = $this->getPluginId() . ':image_styles_responsive_usage';
    $ckeditor_usage_cid = $this->getPluginId() . ':image_styles_ckeditor_usage';

    $display_mode_data = $this->pluginRepository->getCachedData($display_mode_usage_cid);
    $responsive_data = $this->pluginRepository->getCachedData($responsive_usage_cid);
    $ckeditor_data = $this->pluginRepository->getCachedData($ckeditor_usage_cid);

    // If cache is empty, generate the data.
    if (empty($display_mode_data) || !is_array($display_mode_data)) {
      $display_mode_data = [
        'header_table' => $this->getHeaders('image_styles_display_mode_usage'),
        'results_table' => $this->getImageStylesDisplayModeUsageRows(),
      ];
    }

    if (empty($responsive_data) || !is_array($responsive_data)) {
      $responsive_data = [
        'header_table' => $this->getHeaders('image_styles_responsive_usage'),
        'results_table' => $this->getImageStylesResponsiveUsageRows(),
      ];
    }

    if (empty($ckeditor_data) || !is_array($ckeditor_data)) {
      $ckeditor_data = [
        'header_table' => $this->getHeaders('image_styles_ckeditor_usage'),
        'results_table' => $this->getImageStylesCkeditorUsageRows(),
      ];
    }

    // Build usage maps from the rows.
    $used_in_display_modes = [];
    $used_in_responsive = [];
    $used_in_ckeditor = [];

    // Extract style IDs from display mode usage (column 0 is style ID).
    if (!empty($display_mode_data['results_table'])) {
      foreach ($display_mode_data['results_table'] as $row) {
        $used_in_display_modes[$row[0]] = TRUE;
      }
    }

    // Extract style IDs from responsive usage (column 0 is style ID).
    if (!empty($responsive_data['results_table'])) {
      foreach ($responsive_data['results_table'] as $row) {
        $used_in_responsive[$row[0]] = TRUE;
      }
    }

    // Extract style IDs from CKEditor usage (column 0 is style ID).
    if (!empty($ckeditor_data['results_table'])) {
      foreach ($ckeditor_data['results_table'] as $row) {
        $used_in_ckeditor[$row[0]] = TRUE;
      }
    }

    // Check if responsive_image and ckeditor modules are enabled.
    $responsive_module_enabled = $this->moduleHandler->moduleExists('responsive_image');
    $ckeditor_module_enabled = $this->moduleHandler->moduleExists('ckeditor5') || $this->moduleHandler->moduleExists('ckeditor');

    // Build summary rows.
    foreach ($image_styles as $image_style_id => $image_style) {
      $in_display_modes = isset($used_in_display_modes[$image_style_id]);
      $in_responsive = isset($used_in_responsive[$image_style_id]);
      $in_ckeditor = isset($used_in_ckeditor[$image_style_id]);

      // Calculate aggregate result: "Yes" if used anywhere, "No" if not used at all.
      // Only include usage in calculation if respective modules are enabled.
      $is_used_anywhere = $in_display_modes
        || ($responsive_module_enabled && $in_responsive)
        || ($ckeditor_module_enabled && $in_ckeditor);

      $result_table[] = [
        $image_style_id,
        $image_style->label(),
        $in_display_modes ? $this->t('Yes') : $this->t('No'),
        // Show "N/A" if responsive_image module is not enabled.
        $responsive_module_enabled
          ? ($in_responsive ? $this->t('Yes') : $this->t('No'))
          : $this->t('N/A'),
        // Show "N/A" if ckeditor module is not enabled.
        $ckeditor_module_enabled
          ? ($in_ckeditor ? $this->t('Yes') : $this->t('No'))
          : $this->t('N/A'),
        $is_used_anywhere ? $this->t('Yes') : $this->t('No'),
      ];
    }

    return $result_table;
  }

  /**
   * {@inheritdoc}
   */
  public function buildDataRenderArray(array $data, string $operation = '') {
    // Add filter for summary operation.
    if ($operation === 'image_styles_summary') {
      $build = parent::buildDataRenderArray($data, $operation);

      // Add filter before the table.
      $build['filter'] = [
        '#type' => 'container',
        '#weight' => -10,
        '#attributes' => ['class' => ['image-styles-summary-filter']],
        'filter_select' => [
          '#type' => 'select',
          '#title' => $this->t('Filter by Result'),
          '#options' => [
            'all' => $this->t('All'),
            'yes' => $this->t('Used (Yes)'),
            'no' => $this->t('Not Used (No)'),
          ],
          '#default_value' => 'all',
          '#attributes' => [
            'class' => ['image-styles-result-filter'],
          ],
        ],
      ];

      // Add JavaScript for filtering.
      $build['#attached']['library'][] = 'xray_audit/image_styles_summary_filter';

      return $build;
    }

    return parent::buildDataRenderArray($data, $operation);
  }

}
