<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric;

use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "queries_data_vocabulary",
 *   label = @Translation("Taxonomy Terms reports"),
 *   description = @Translation("Metrics about Vocabularies."),
 *   group = "content_metric",
 *   sort = 3,
 *   local_task = 1,
 *   operations = {
 *       "taxonomy_term_count_type" = {
 *          "label" = "Terms grouped by type",
 *          "description" = "Number of Terms grouped by type.",
 *          "download" = TRUE
 *       },
 *      "taxonomy_term_count_type_langcode" = {
 *          "label" = "Terms grouped by type and language",
 *          "description" = "Number of Terms grouped by type and language.",
 *          "download" = TRUE
 *       }
 *    },
 *   dependencies = {"taxonomy"}
 * )
 */
class XrayAuditQueryTaskVocabularyPlugin extends XrayAuditTaskPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    switch ($operation) {
      case 'taxonomy_term_count_type':
        return [
          $this->t('VID'),
          $this->t('Count'),
        ];

      case 'taxonomy_term_count_type_langcode':
        return [
          $this->t('VID'),
          $this->t('Langcode'),
          $this->t('Count'),
        ];
    }
    return [];
  }

  /**
   * Gets all vocabulary IDs.
   *
   * @return array
   *   Array of vocabulary IDs.
   */
  private function getAllVocabularyIds(): array {
    $vocabulary_storage = $this->entityTypeManager->getStorage('taxonomy_vocabulary');
    return $vocabulary_storage->getQuery()
      ->accessCheck(FALSE)
      ->execute();
  }

  /**
   * Gets the rows for the 'taxonomy_term_count_type' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getTaxonomyTermCountTypeRows(): array {
    $resultTable = [];
    $alias = 'count';

    $vids = $this->getAllVocabularyIds();

    $storage = $this->entityTypeManager->getStorage('taxonomy_term');
    $result = $storage
      ->getAggregateQuery()
      ->accessCheck(FALSE)
      ->currentRevision()
      ->groupBy('vid')
      ->aggregate('tid', 'COUNT', NULL, $alias)
      ->sort('vid')
      ->execute();

    $term_counts = [];
    foreach ($result as $row) {
      if (!empty($row['vid'])) {
        $term_counts[$row['vid']] = $row['count'];
      }
    }

    $total = 0;
    // Include all vids in the result.
    foreach ($vids as $vid) {
      $count = $term_counts[$vid] ?? 0;
      $resultTable[] = ['vid' => $vid, 'count' => $count];
      $total += $count;
    }

    // Sort by count (descending).
    usort($resultTable, function ($a, $b) {
      return $b['count'] <=> $a['count'];
    });

    // Convert to simple arrays for display.
    $sorted_result = [];
    foreach ($resultTable as $row) {
      $sorted_result[] = [$row['vid'], $row['count']];
    }

    // Add total row.
    $sorted_result[] = [
      $this->t('TOTAL'),
      $total,
    ];

    return $sorted_result;
  }

  /**
   * Gets the rows for the 'taxonomy_term_count_type_langcode' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getTaxonomyTermCountTypeLangcodeRows(): array {
    $resultTable = [];
    $alias = 'count';

    $vids = $this->getAllVocabularyIds();

    $storage = $this->entityTypeManager->getStorage('taxonomy_term');
    $result = $storage
      ->getAggregateQuery()
      ->accessCheck(FALSE)
      ->currentRevision()
      ->groupBy('vid')
      ->groupBy('langcode')
      ->aggregate('tid', 'COUNT', NULL, $alias)
      ->sort('vid')
      ->sort('langcode')
      ->execute();

    $term_counts = [];
    foreach ($result as $row) {
      if (!empty($row['vid'])) {
        $term_counts[$row['vid']][$row['langcode']] = $row['count'];
      }
    }

    foreach ($vids as $vid) {
      $languages = !empty($term_counts[$vid]) ? array_keys($term_counts[$vid]) : ['und'];
      foreach ($languages as $langcode) {
        $count = $term_counts[$vid][$langcode] ?? 0;
        $resultTable[] = ['vid' => $vid, 'langcode' => $langcode, 'count' => $count];
      }
    }

    // Sort by count (descending).
    usort($resultTable, function ($a, $b) {
      return $b['count'] <=> $a['count'];
    });

    // Convert to simple arrays for display.
    $sorted_result = [];
    foreach ($resultTable as $row) {
      $sorted_result[] = [$row['vid'], $row['langcode'], $row['count']];
    }

    return $sorted_result;
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    switch ($operation) {
      case 'taxonomy_term_count_type':
        return $this->getTaxonomyTermCountTypeRows();

      case 'taxonomy_term_count_type_langcode':
        return $this->getTaxonomyTermCountTypeLangcodeRows();
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    $cid = $this->getPluginId() . ':' . $operation;

    $cached_data = $this->pluginRepository->getCachedData($cid);
    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    $data = [
      'header_table' => $this->getHeaders($operation),
      'results_table' => $this->getRows($operation),
    ];

    $this->pluginRepository->setCacheTagsInv($cid, $data, ['taxonomy_term_list', 'taxonomy_vocabulary_list']);
    return $data;
  }

}
