<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentModel;

use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "content_entities",
 *   label = @Translation("Content Entities reports"),
 *   description = @Translation("Reports on all the content entities of this site."),
 *   group = "content_model",
 *   sort = 1,
 *   operations = {
 *      "content_entity_types" = {
 *          "label" = "Content entity types",
 *          "description" = "List of the content entity types of this site (core, contrib and custom).",
 *          "download" = TRUE
 *       }
 *   }
 * )
 */
class XrayAuditQueryTaskContentEntitiesPlugin extends XrayAuditTaskPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    switch ($operation) {
      case 'content_entity_types':
        return [
          $this->t('id'),
          $this->t('Group'),
          $this->t('Provider'),
          $this->t('Class'),
        ];
    }
    return [];
  }

  /**
   * Gets the rows for the 'content_entity_types' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getContentEntityTypesRows(): array {
    $resultTable = [];
    $definitions = $this->entityTypeManager->getDefinitions();
    foreach ($definitions as $key => $definition) {
      $group = $definition->getGroup();
      if ($group !== 'content') {
        continue;
      }
      $resultTable[$key] = [
        'key' => $key,
        'group' => $group,
        'provider' => $definition->getProvider(),
        'class' => $definition->getClass(),
      ];
    }
    ksort($resultTable);
    return $resultTable;
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    switch ($operation) {
      case 'content_entity_types':
        return $this->getContentEntityTypesRows();
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    $cid = $this->getPluginId() . ':' . $operation;

    $cached_data = $this->pluginRepository->getCachedData($cid);
    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    $data = [];
    switch ($operation) {
      case 'content_entity_types':
        $data = [
          'header_table' => $this->getHeaders($operation),
          'results_table' => $this->getRows($operation),
        ];
        break;
    }

    if (!empty($data)) {
      $this->pluginRepository->setCacheTagsInv($cid, $data, ['entity_types']);
    }
    return $data;
  }

}
