<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\Webform;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Link;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;

/**
 * Plugin implementation for Webform report.
 *
 * @XrayAuditTaskPlugin (
 *   id = "webform_report",
 *   label = @Translation("Webform information"),
 *   description = @Translation("Report about webforms"),
 *   group = "forms",
 *   sort = 1,
 *   local_task = 1,
 *   operations = {
 *      "webforms" = {
 *           "label" = "Webforms",
 *           "description" = "Report about webforms.",
 *           "dependencies" = {"webform"},
 *           "download" = TRUE
 *        },
 *    },
 * )
 */
class XrayAuditWebformGeneralTaskPlugin extends XrayAuditTaskPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    if ($operation === 'webforms') {
      return [
        $this->t('Title'),
        $this->t('ID'),
        $this->t('Status'),
        $this->t('Number of Fields'),
        $this->t('Handlers'),
        $this->t('Webform'),
      ];
    }
    return [];
  }

  /**
   * Gets the rows and cache tags for the 'webforms' operation.
   *
   * @return array
   *   An associative array containing 'rows' and 'cache_tags'.
   */
  private function getWebformRowsAndCacheTags(): array {
    $webform_rows = [];
    $collected_cache_tags = [];

    $webform_storage = $this->entityTypeManager->getStorage('webform');
    /** @var \Drupal\webform\Entity\Webform[] $webforms */
    $webforms = $webform_storage->loadMultiple();

    if (empty($webforms)) {
      return ['rows' => [], 'cache_tags' => []];
    }

    foreach ($webforms as $webform) {
      $collected_cache_tags = Cache::mergeTags($collected_cache_tags, $webform->getCacheTags());

      $handlers = $webform->getHandlers();
      $handler_ids = array_keys($handlers->getInstanceIds());

      $url = $webform->toUrl('edit-form');
      $url->setOption('attributes', ['target' => '_blank']);
      $edit_link = $url->access() ? Link::fromTextAndUrl($this->t('Edit'), $url) : $this->t('No access to edit');

      $webform_rows[] = [
        'title' => $webform->label(),
        'id' => $webform->id(),
        'status' => $webform->isOpen() ? $this->t('Open') : $this->t('Closed'),
        'num_fields' => count($webform->getElementsDecoded()),
        'handlers' => implode(', ', $handler_ids),
        'edit_link' => $edit_link,
      ];
    }

    return [
      'rows' => $webform_rows,
      'cache_tags' => $collected_cache_tags,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    if ($operation === 'webforms') {
      $data = $this->getWebformRowsAndCacheTags();
      return $data['rows'];
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    if ($operation !== 'webforms') {
      return ['#markup' => $this->t('No data available for this operation.')];
    }

    $cid = $this->getPluginId() . ':' . $operation;
    $cached_data = $this->pluginRepository->getCachedData($cid);

    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    $headers = $this->getHeaders($operation);
    $rowDataAndTags = $this->getWebformRowsAndCacheTags();
    $results_table = $rowDataAndTags['rows'];
    $collected_cache_tags = $rowDataAndTags['cache_tags'];

    $data = [
      'header_table' => $headers,
      'results_table' => $results_table,
    // For potential use or consistency.
      'num_webforms' => count($results_table),
    ];

    // Add webform list tag for broader invalidation if needed.
    $collected_cache_tags = Cache::mergeTags($collected_cache_tags, ['webform_list']);
    $this->pluginRepository->setCacheTagsInv($cid, $data, $collected_cache_tags);

    return $data;
  }

}
