<?php

namespace Drupal\Tests\xray_audit\Functional\Controller;

use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Drupal\xray_audit\Controller\XrayAuditTaskController;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the XrayAuditTaskController.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 * @group critical
 */
class XrayAuditTaskControllerTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'block',
    'block_content',
    'views',
    'menu_link_content',
    'link',
    'file',
    'image',
    'taxonomy',
    'xray_audit',
  ];

  /**
   * A user with permission to access xray audit.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $privilegedUser;

  /**
   * A user without permission to access xray audit.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $unprivilegedUser;

  /**
   * The plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create users with different permissions.
    $this->privilegedUser = $this->drupalCreateUser([
      'xray_audit access',
      'access content',
    ]);

    $this->unprivilegedUser = $this->drupalCreateUser([
      'access content',
    ]);

    // Get plugin repository service.
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');
  }

  /**
   * Tests that unprivileged users cannot access task routes.
   *
   */
  public function testAccessDeniedForUnprivilegedUsers() {
    // Arrange: Login as unprivileged user.
    $this->drupalLogin($this->unprivilegedUser);

    // Get a task operation to test with.
    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $this->assertNotEmpty($task_definitions, 'Should have task definitions');

    $first_task = reset($task_definitions);
    $operations = $first_task['operations'] ?? [];

    if (empty($operations)) {
      $this->markTestSkipped('No operations available for testing');
    }

    $operation_id = array_key_first($operations);
    $task_operation = str_replace('_', '-', $operation_id);

    // Act: Try to access a task operation route.
    $route_info = $this->pluginRepository->getOnePluginTaskRouteInfo($first_task['id'], $operation_id);

    if (empty($route_info['route_name'])) {
      $this->markTestSkipped('No route available for testing');
    }

    $this->drupalGet(Url::fromRoute($route_info['route_name']));

    // Assert: Access should be denied.
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that privileged users can access task routes.
   *
   */
  public function testAccessGrantedForPrivilegedUsers() {
    // Arrange: Login as privileged user.
    $this->drupalLogin($this->privilegedUser);

    // Get a task operation to test with.
    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $this->assertNotEmpty($task_definitions, 'Should have task definitions');

    $first_task = reset($task_definitions);
    $operations = $first_task['operations'] ?? [];

    if (empty($operations)) {
      $this->markTestSkipped('No operations available for testing');
    }

    $operation_id = array_key_first($operations);

    // Act: Access a task operation route.
    $route_info = $this->pluginRepository->getOnePluginTaskRouteInfo($first_task['id'], $operation_id);

    if (empty($route_info['route_name'])) {
      $this->markTestSkipped('No route available for testing');
    }

    $this->drupalGet(Url::fromRoute($route_info['route_name']));

    // Assert: Should be accessible (200 or render properly).
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests build() method returns proper render array.
   *
   */
  public function testBuildReturnsRenderArray() {
    // Arrange: Get controller and a valid operation.
    $controller = XrayAuditTaskController::create($this->container);

    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $this->assertNotEmpty($task_definitions, 'Should have task definitions');

    $first_task = reset($task_definitions);
    $operations = $first_task['operations'] ?? [];

    if (empty($operations)) {
      $this->markTestSkipped('No operations available for testing');
    }

    $operation_id = array_key_first($operations);

    // Act: Call build method.
    $render_array = $controller->build($operation_id);

    // Assert: Should return an array (render array).
    $this->assertIsArray($render_array, 'build() should return an array');
  }

  /**
   * Tests build() throws exception for invalid operation.
   *
   */
  public function testBuildThrowsExceptionForInvalidOperation() {
    // Arrange: Get controller.
    $controller = XrayAuditTaskController::create($this->container);

    // Expect exception.
    $this->expectException(NotFoundHttpException::class);

    // Act: Call build with invalid operation.
    $controller->build('invalid_operation_that_does_not_exist');
  }

  /**
   * Tests getTitle() returns correct title for operation.
   *
   */
  public function testGetTitleReturnsOperationLabel() {
    // Arrange: Get controller and find an operation with a label.
    $controller = XrayAuditTaskController::create($this->container);

    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $test_operation = NULL;
    $expected_label = NULL;

    foreach ($task_definitions as $task) {
      if (!empty($task['local_task'])) {
        // Skip local task cases.
        continue;
      }
      foreach ($task['operations'] ?? [] as $op_id => $op) {
        if (!empty($op['label'])) {
          $test_operation = $op_id;
          $expected_label = $op['label'];
          break 2;
        }
      }
    }

    if ($test_operation === NULL) {
      $this->markTestSkipped('No operation with label found for testing');
    }

    // Act: Get title for operation.
    $title = $controller->getTitle($test_operation);

    // Assert: Should return the operation label.
    $this->assertIsString($title, 'getTitle() should return a string');
    $this->assertNotEmpty($title, 'Title should not be empty');
    $this->assertEquals($expected_label, $title, 'Title should match operation label');
  }

  /**
   * Tests getTitle() returns plugin label for local task case.
   *
   */
  public function testGetTitleReturnsPluginLabelForLocalTask() {
    // Arrange: Get controller and find a local task.
    $controller = XrayAuditTaskController::create($this->container);

    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $test_operation = NULL;
    $expected_label = NULL;

    foreach ($task_definitions as $task) {
      if (!empty($task['local_task'])) {
        $operations = $task['operations'] ?? [];
        if (!empty($operations)) {
          $test_operation = array_key_first($operations);
          $expected_label = $task['label'];
          break;
        }
      }
    }

    if ($test_operation === NULL) {
      $this->markTestSkipped('No local task found for testing');
    }

    // Act: Get title for local task operation.
    $title = $controller->getTitle($test_operation);

    // Assert: Should return the plugin label.
    $this->assertIsString($title, 'getTitle() should return a string');
    $this->assertEquals($expected_label, $title, 'Title should match plugin label for local task');
  }

  /**
   * Tests getTaskPluginFromTaskParameter() returns correct plugin.
   *
   */
  public function testGetTaskPluginFromTaskParameter() {
    // Arrange: Get controller and a task plugin ID.
    $controller = XrayAuditTaskController::create($this->container);

    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $this->assertNotEmpty($task_definitions, 'Should have task definitions');

    $first_task = reset($task_definitions);
    $task_id = $first_task['id'];

    // Act: Get plugin instance.
    $plugin = $controller->getTaskPluginFromTaskParameter($task_id);

    // Assert: Should return a plugin instance.
    $this->assertInstanceOf(
      XrayAuditTaskPluginInterface::class,
      $plugin,
      'Should return task plugin instance'
    );
    $this->assertEquals($task_id, $plugin->getPluginId(), 'Plugin ID should match');
  }

  /**
   * Tests getTaskPluginFromOperationParameter() returns correct plugin.
   *
   */
  public function testGetTaskPluginFromOperationParameter() {
    // Arrange: Get controller and an operation.
    $controller = XrayAuditTaskController::create($this->container);

    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $test_operation = NULL;
    $expected_plugin_id = NULL;

    foreach ($task_definitions as $task) {
      $operations = $task['operations'] ?? [];
      if (!empty($operations)) {
        $test_operation = array_key_first($operations);
        $expected_plugin_id = $task['id'];
        break;
      }
    }

    if ($test_operation === NULL) {
      $this->markTestSkipped('No operation found for testing');
    }

    // Act: Get plugin instance from operation.
    $plugin = $controller->getTaskPluginFromOperationParameter($test_operation);

    // Assert: Should return correct plugin instance.
    $this->assertInstanceOf(
      XrayAuditTaskPluginInterface::class,
      $plugin,
      'Should return task plugin instance'
    );
    $this->assertEquals($expected_plugin_id, $plugin->getPluginId(), 'Plugin ID should match');
  }

  /**
   * Tests getTaskPluginFromOperationParameter() returns null for invalid op.
   *
   */
  public function testGetTaskPluginFromOperationParameterReturnsNullForInvalid() {
    // Arrange: Get controller.
    $controller = XrayAuditTaskController::create($this->container);

    // Act: Get plugin for invalid operation.
    $plugin = $controller->getTaskPluginFromOperationParameter('invalid_operation_does_not_exist');

    // Assert: Should return null.
    $this->assertNull($plugin, 'Should return null for invalid operation');
  }

  /**
   * Tests getTaskPluginFromOperation() throws exception for invalid operation.
   *
   */
  public function testGetTaskPluginFromOperationThrowsExceptionForInvalid() {
    // Arrange: Get controller.
    $controller = XrayAuditTaskController::create($this->container);

    // Expect exception.
    $this->expectException(NotFoundHttpException::class);

    // Act: Get plugin for invalid operation.
    $controller->getTaskPluginFromOperation('invalid_operation_does_not_exist');
  }

  /**
   * Tests buildBatchProcess() returns redirect response.
   *
   */
  public function testBuildBatchProcessReturnsRedirectResponse() {
    // Arrange: Get controller.
    $controller = XrayAuditTaskController::create($this->container);

    // Find a task with batch process.
    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $test_task_id = NULL;
    $test_batch_process = NULL;

    foreach ($task_definitions as $task) {
      if (empty($task['id'])) {
        continue;
      }

      $task_id = $task['id'];
      $plugin = $this->pluginRepository->getInstancePluginTask($task_id);

      // Check if plugin has batch processes.
      $batch_processes = $task['batch'] ?? [];
      if (!empty($batch_processes)) {
        $batch_id = array_key_first($batch_processes);
        $batch_method = $plugin->getBatchClass($batch_id);

        if ($batch_method && method_exists($plugin, $batch_method)) {
          $test_task_id = $task_id;
          $test_batch_process = $batch_id;
          break;
        }
      }
    }

    if ($test_task_id === NULL) {
      $this->markTestSkipped('No batch process found for testing');
    }

    // Act: Build batch process.
    $response = $controller->buildBatchProcess($test_task_id, $test_batch_process);

    // Assert: Should return redirect response.
    $this->assertInstanceOf(
      RedirectResponse::class,
      $response,
      'buildBatchProcess() should return RedirectResponse'
    );
  }

  /**
   * Tests buildBatchProcess() throws exception for invalid batch process.
   *
   */
  public function testBuildBatchProcessThrowsExceptionForInvalidBatch() {
    // Arrange: Get controller and a valid task.
    $controller = XrayAuditTaskController::create($this->container);

    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $this->assertNotEmpty($task_definitions, 'Should have task definitions');

    $first_task = reset($task_definitions);
    $task_id = $first_task['id'];

    // Expect exception.
    $this->expectException(NotFoundHttpException::class);

    // Act: Try to build with invalid batch process.
    $controller->buildBatchProcess($task_id, 'invalid_batch_process');
  }

  /**
   * Tests buildBatchProcess() throws exception for non-existent batch method.
   *
   */
  public function testBuildBatchProcessThrowsExceptionForNonExistentMethod() {
    // Arrange: Get controller and a task.
    $controller = XrayAuditTaskController::create($this->container);

    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $this->assertNotEmpty($task_definitions, 'Should have task definitions');

    $first_task = reset($task_definitions);
    $task_id = $first_task['id'];

    // Expect exception - this will throw because the batch process doesn't exist.
    $this->expectException(NotFoundHttpException::class);

    // Act: Try to build with a batch process name that won't have a method.
    $controller->buildBatchProcess($task_id, 'non_existent_batch_method_xyz');
  }

  /**
   * Tests that all report pages return HTTP 200 status.
   *
   * This test visits all available task operation pages and verifies
   * they load successfully without errors.
   *
   */
  public function testAllReportPagesReturnHttp200() {
    // Arrange: Login as privileged user.
    $this->drupalLogin($this->privilegedUser);

    // Get all task definitions.
    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $this->assertNotEmpty($task_definitions, 'Should have task definitions');

    $total_operations = 0;
    $failed_operations = [];
    $skipped_operations = [];

    // Operations that require additional setup in functional tests.
    $skip_in_tests = [
      'themes' => 'Requires additional themes to be installed',
    ];

    // Act & Assert: Visit each task operation page.
    foreach ($task_definitions as $task_id => $task_definition) {
      $operations = $task_definition['operations'] ?? [];

      foreach ($operations as $operation_id => $operation_info) {
        // Skip operations known to have issues in functional test environment.
        if (isset($skip_in_tests[$operation_id])) {
          $skipped_operations[] = [
            'task' => $task_id,
            'operation' => $operation_id,
            'reason' => $skip_in_tests[$operation_id],
          ];
          continue;
        }

        $total_operations++;

        // Get route information for this operation.
        $route_info = $this->pluginRepository->getOnePluginTaskRouteInfo($task_id, $operation_id);

        if (empty($route_info['route_name'])) {
          $this->fail("No route found for task '{$task_id}', operation '{$operation_id}'");
          continue;
        }

        // Visit the page.
        $url = Url::fromRoute($route_info['route_name']);
        $this->drupalGet($url);

        // Check status code.
        $status_code = $this->getSession()->getStatusCode();

        if ($status_code !== 200) {
          // Get error message from page if available.
          $page_content = $this->getSession()->getPage()->getText();
          $error_excerpt = substr($page_content, 0, 200);

          $failed_operations[] = [
            'task' => $task_id,
            'operation' => $operation_id,
            'route' => $route_info['route_name'],
            'status' => $status_code,
            'error' => $error_excerpt,
          ];

          // Don't fail immediately, collect all failures.
          continue;
        }
      }
    }

    // Final assertion with summary.
    if (!empty($failed_operations)) {
      $failure_message = sprintf("\n%d operations failed:\n\n", count($failed_operations));
      foreach ($failed_operations as $failure) {
        $failure_message .= sprintf(
          "Task: %s\nOperation: %s\nRoute: %s\nHTTP Status: %d\nError: %s\n\n",
          $failure['task'],
          $failure['operation'],
          $failure['route'],
          $failure['status'],
          $failure['error'] ?? 'No error details'
        );
      }
      $this->fail($failure_message);
    }

    // Success message with summary.
    $success_message = sprintf(
      'Successfully tested %d operation pages (HTTP 200)',
      $total_operations
    );

    if (!empty($skipped_operations)) {
      $success_message .= sprintf(
        "\nSkipped %d operations in test environment:\n",
        count($skipped_operations)
      );
      foreach ($skipped_operations as $skip) {
        $success_message .= sprintf(
          "  - %s/%s: %s\n",
          $skip['task'],
          $skip['operation'],
          $skip['reason']
        );
      }
    }

    $this->assertTrue(TRUE, $success_message);
  }

}
