<?php

declare(strict_types=1);

namespace Drupal\Tests\xray_audit\Kernel\Form;

use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormState;
use Drupal\KernelTests\ConfigFormTestBase;
use Drupal\xray_audit\Form\SettingsForm;

/**
 * Tests the SettingsForm.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class SettingsFormTest extends ConfigFormTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'xray_audit',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\xray_audit\Form\SettingsForm
   */
  protected $form;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // KernelTestBase::bootKernel() sets a global override for system.file
    // which can cause schema validation errors in strict modes (e.g., CI).
    // Remove the global override before installing config.
    unset($GLOBALS['config']['system.file']);

    $this->installConfig(['xray_audit']);

    // Get services.
    $this->configFactory = $this->container->get('config.factory');

    // Create form instance.
    $this->form = SettingsForm::create($this->container);
  }

  /**
   * Tests form instantiation through dependency injection.
   */
  public function testFormInstantiation() {
    // Arrange & Act: Get form from container.
    $form = SettingsForm::create($this->container);

    // Assert: Form is instantiated correctly.
    $this->assertInstanceOf(
      SettingsForm::class,
      $form,
      'Form should be properly instantiated'
    );
  }

  /**
   * Tests getFormId returns the correct form ID.
   */
  public function testGetFormId() {
    // Act: Get form ID.
    $form_id = $this->form->getFormId();

    // Assert: Form ID is correct.
    $this->assertEquals(
      'xray_audit.settings',
      $form_id,
      'Form ID should be xray_audit.settings'
    );
  }

  /**
   * Tests getEditableConfigNames returns correct config names.
   */
  public function testGetEditableConfigNames() {
    // Act: Get editable config names using reflection.
    $reflection = new \ReflectionClass($this->form);
    $method = $reflection->getMethod('getEditableConfigNames');
    $method->setAccessible(TRUE);
    $config_names = $method->invoke($this->form);

    // Assert: Config names are correct.
    $this->assertIsArray($config_names, 'Config names should be an array');
    $this->assertContains('xray_audit.settings', $config_names, 'Should include xray_audit.settings');
  }

  /**
   * Tests that buildForm returns proper form array structure.
   */
  public function testBuildFormStructure() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Form has required elements.
    $this->assertArrayHasKey('revisions_thresholds', $form_array, 'Form should have revisions_thresholds fieldset');
    $this->assertArrayHasKey('size_thresholds', $form_array, 'Form should have size_thresholds fieldset');
    $this->assertArrayHasKey('actions', $form_array, 'Form should have actions');
  }

  /**
   * Tests that revisions_thresholds contains node and paragraph fields.
   */
  public function testRevisionsThresholdsContainsRequiredFields() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Revisions thresholds contains required fields.
    $this->assertArrayHasKey('node', $form_array['revisions_thresholds'], 'Should have node threshold field');
    $this->assertArrayHasKey('paragraph', $form_array['revisions_thresholds'], 'Should have paragraph threshold field');
  }

  /**
   * Tests that size_thresholds contains tables field.
   */
  public function testSizeThresholdsContainsTablesField() {
    // Arrange: Create form state.
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Size thresholds contains tables field.
    $this->assertArrayHasKey('tables', $form_array['size_thresholds'], 'Should have tables threshold field');
  }

  /**
   * Tests that form loads default values from configuration.
   */
  public function testFormLoadsDefaultValuesFromConfig() {
    // Arrange: Set config values.
    $config = $this->configFactory->getEditable('xray_audit.settings');
    $config->set('revisions_thresholds.node', 100);
    $config->set('revisions_thresholds.paragraph', 50);
    $config->set('size_thresholds.tables', 200);
    $config->save();

    // Recreate form to load new config.
    $this->form = SettingsForm::create($this->container);
    $form_state = new FormState();

    // Act: Build form.
    $form_array = $this->form->buildForm([], $form_state);

    // Assert: Default values match config.
    $this->assertEquals(100, $form_array['revisions_thresholds']['node']['#default_value'], 'Node threshold should load from config');
    $this->assertEquals(50, $form_array['revisions_thresholds']['paragraph']['#default_value'], 'Paragraph threshold should load from config');
    $this->assertEquals(200, $form_array['size_thresholds']['tables']['#default_value'], 'Tables threshold should load from config');
  }

  /**
   * Tests form submission saves configuration.
   */
  public function testFormSubmissionSavesConfiguration() {
    // Arrange: Create form state with values.
    $form_state = new FormState();
    $form_state->setValue('revisions_thresholds', [
      'node' => 150,
      'paragraph' => 75,
    ]);
    $form_state->setValue('size_thresholds', [
      'tables' => 250,
    ]);

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: Configuration is saved.
    $config = $this->configFactory->get('xray_audit.settings');
    $this->assertEquals(150, $config->get('revisions_thresholds.node'), 'Node threshold should be saved');
    $this->assertEquals(75, $config->get('revisions_thresholds.paragraph'), 'Paragraph threshold should be saved');
    $this->assertEquals(250, $config->get('size_thresholds.tables'), 'Tables threshold should be saved');
  }

  /**
   * Tests form submission with nested configuration structure.
   */
  public function testFormSubmissionHandlesNestedConfig() {
    // Arrange: Create form state with nested values.
    $form_state = new FormState();
    $form_state->setValue('revisions_thresholds', [
      'node' => 200,
      'paragraph' => 100,
    ]);

    // Act: Submit form.
    $form_array = [];
    $this->form->submitForm($form_array, $form_state);

    // Assert: Nested config is saved correctly.
    $config = $this->configFactory->get('xray_audit.settings');
    $revisions = $config->get('revisions_thresholds');
    $this->assertIsArray($revisions, 'Revisions thresholds should be saved as array');
    $this->assertEquals(200, $revisions['node'], 'Node threshold should be in nested structure');
    $this->assertEquals(100, $revisions['paragraph'], 'Paragraph threshold should be in nested structure');
  }

  /**
   * {@inheritdoc}
   *
   * Override parent test to provide required field values.
   */
  public function testConfigForm(): void {
    // Arrange: Build form.
    $form_builder = $this->container->get('form_builder');
    $form_id = $this->form->getFormId();

    // Act: Get form.
    $form = $form_builder->getForm($this->form);

    // Assert: Form is built.
    $this->assertNotEmpty($form);

    // Arrange: Set required values for submission.
    $values = [
      'revisions_thresholds' => [
        'node' => 100,
        'paragraph' => 50,
      ],
      'size_thresholds' => [
        'tables' => 200,
      ],
      'op' => 'Save configuration',
    ];

    // Act: Submit form with valid values.
    $form_state = (new FormState())->setValues($values);
    $form_builder->submitForm($this->form, $form_state);

    // Assert: No validation errors.
    $errors = $form_state->getErrors();
    $this->assertEmpty($errors, 'No validation errors should occur with valid values');
  }

}
