<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentAccessControl;

use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;

/**
 * Comprehensive tests for XrayAuditQueryTaskUserPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditQueryTaskUserPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The user plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentAccessControl\XrayAuditQueryTaskUserPlugin
   */
  protected $plugin;

  /**
   * Test users created for testing.
   *
   * @var \Drupal\user\Entity\User[]
   */
  protected $testUsers = [];

  /**
   * Test roles created for testing.
   *
   * @var \Drupal\user\Entity\Role[]
   */
  protected $testRoles = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install user_data schema (required for user deletion).
    $this->installSchema('user', ['users_data']);

    // Get task plugin manager.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('queries_data_user');

    // Create test users with various states.
    $this->createTestUsers();

    // Create test roles.
    $this->createTestRoles();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clean up test data.
    foreach ($this->testUsers as $user) {
      if ($user) {
        $user->delete();
      }
    }
    foreach ($this->testRoles as $role) {
      if ($role) {
        $role->delete();
      }
    }
    parent::tearDown();
  }

  /**
   * Creates test users with various states.
   */
  protected function createTestUsers() {
    $current_time = time();
    $one_month_seconds = 2592000;

    // Active user - accessed within last 6 months (2 months ago).
    $this->testUsers['active_recent'] = User::create([
      'name' => 'active_user_recent',
      'mail' => 'active_recent@example.com',
      'status' => 1,
      'access' => $current_time - ($one_month_seconds * 2),
    ]);
    $this->testUsers['active_recent']->save();

    // Active user - accessed 8 months ago (6-12 months range).
    $this->testUsers['active_medium'] = User::create([
      'name' => 'active_user_medium',
      'mail' => 'active_medium@example.com',
      'status' => 1,
      'access' => $current_time - ($one_month_seconds * 8),
    ]);
    $this->testUsers['active_medium']->save();

    // Active user - accessed 14 months ago (>12 months).
    $this->testUsers['active_old'] = User::create([
      'name' => 'active_user_old',
      'mail' => 'active_old@example.com',
      'status' => 1,
      'access' => $current_time - ($one_month_seconds * 14),
    ]);
    $this->testUsers['active_old']->save();

    // Blocked user - should appear in status count but not activity.
    $this->testUsers['blocked_user'] = User::create([
      'name' => 'blocked_user',
      'mail' => 'blocked@example.com',
      'status' => 0,
      'access' => $current_time - ($one_month_seconds * 3),
    ]);
    $this->testUsers['blocked_user']->save();

    // Active user with no access (access = 0).
    $this->testUsers['never_accessed'] = User::create([
      'name' => 'never_accessed',
      'mail' => 'never@example.com',
      'status' => 1,
      'access' => 0,
    ]);
    $this->testUsers['never_accessed']->save();

    // Additional active user for counting.
    $this->testUsers['active_extra'] = User::create([
      'name' => 'active_extra',
      'mail' => 'extra@example.com',
      'status' => 1,
      'access' => $current_time - ($one_month_seconds * 1),
    ]);
    $this->testUsers['active_extra']->save();
  }

  /**
   * Creates test roles.
   */
  protected function createTestRoles() {
    // Create editor role.
    $this->testRoles['editor'] = Role::create([
      'id' => 'editor',
      'label' => 'Editor',
    ]);
    $this->testRoles['editor']->save();

    // Create content_manager role.
    $this->testRoles['content_manager'] = Role::create([
      'id' => 'content_manager',
      'label' => 'Content Manager',
    ]);
    $this->testRoles['content_manager']->save();

    // Assign roles to some users.
    $this->testUsers['active_recent']->addRole('editor');
    $this->testUsers['active_recent']->save();

    $this->testUsers['active_medium']->addRole('content_manager');
    $this->testUsers['active_medium']->save();

    $this->testUsers['active_old']->addRole('editor');
    $this->testUsers['active_old']->addRole('content_manager');
    $this->testUsers['active_old']->save();
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  // -------------------------------------------------------------------------
  // getDataOperationResult()
  // -------------------------------------------------------------------------

  /**
   * Tests getDataOperationResult() with 'user_count_status' operation.
   */
  public function testGetDataOperationResultUserCountStatus() {
    // Act.
    $result = $this->plugin->getDataOperationResult('user_count_status');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure.
    $this->assertCount(2, $result['header_table']);

    // Verify results contain status data.
    $this->assertNotEmpty($result['results_table']);

    // Should have at least Active, Blocked, and Total rows.
    $this->assertGreaterThanOrEqual(2, count($result['results_table']));
  }

  /**
   * Tests getDataOperationResult() with 'user_count_activity' operation.
   *
   */
  public function testGetDataOperationResultUserCountActivity() {
    // Act.
    $result = $this->plugin->getDataOperationResult('user_count_activity');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure.
    $this->assertCount(2, $result['header_table']);

    // Verify results contain activity periods.
    $this->assertCount(3, $result['results_table']);

    // Verify structure of each row.
    foreach ($result['results_table'] as $row) {
      $this->assertCount(2, $row);
      $this->assertIsNumeric($row[1]);
    }
  }

  /**
   * Tests getDataOperationResult() with 'user_count_role' operation.
   *
   */
  public function testGetDataOperationResultUserCountRole() {
    // Act.
    $result = $this->plugin->getDataOperationResult('user_count_role');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure.
    $this->assertCount(3, $result['header_table']);

    // Verify results contain role data.
    $this->assertNotEmpty($result['results_table']);

    // Verify results_table is an array.
    $this->assertIsArray($result['results_table']);

    // Verify each role has proper structure.
    // results_table is now an indexed array (array_values was called).
    // Each row is [role_id, role_label, count].
    foreach ($result['results_table'] as $role_data) {
      $this->assertIsArray($role_data);
      $this->assertCount(3, $role_data);
      // First element is role_id.
      $this->assertIsString($role_data[0]);
      // Second element is role_label.
      $this->assertIsString($role_data[1]);
      // Third element is count.
      $this->assertIsNumeric($role_data[2]);
    }
  }

  // -------------------------------------------------------------------------
  // DATA STRUCTURE & INTEGRATION TESTS
  // -------------------------------------------------------------------------

}
