<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentMetric;

use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Request;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;
use Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric\XrayAuditQueryTaskParagraphsPlugin;

/**
 * Comprehensive tests for XrayAuditQueryTaskParagraphsPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditQueryTaskParagraphsPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'filter',
    'file',
    'paragraphs',
    'entity_reference_revisions',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The paragraphs plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric\XrayAuditQueryTaskParagraphsPlugin
   */
  protected $plugin;

  /**
   * Test paragraph types created for testing.
   *
   * @var \Drupal\paragraphs\Entity\ParagraphsType[]
   */
  protected $testParagraphTypes = [];

  /**
   * Test paragraphs created for testing.
   *
   * @var \Drupal\paragraphs\Entity\Paragraph[]
   */
  protected $testParagraphs = [];

  /**
   * Test nodes created for testing.
   *
   * @var \Drupal\node\Entity\Node[]
   */
  protected $testNodes = [];

  /**
   * Test content types created for testing.
   *
   * @var \Drupal\node\Entity\NodeType[]
   */
  protected $testContentTypes = [];

  /**
   * Plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * State service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install required entity schemas.
    $this->installEntitySchema('paragraph');
    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['node', 'filter', 'system', 'paragraphs', 'field']);

    // Get services.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');
    $this->database = $this->container->get('database');
    $this->state = $this->container->get('state');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('queries_data_paragraphs');

    // Create test data.
    $this->createTestParagraphTypes();
    $this->createTestContentTypes();
    $this->createTestFieldsForParagraphs();
    $this->createTestParagraphs();
    $this->createTestNodesWithParagraphs();

    // Run install actions to create temporary table.
    $this->plugin->paragraphsInstallActions();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clean up test nodes.
    foreach ($this->testNodes as $node) {
      if ($node) {
        $node->delete();
      }
    }

    // Clean up test paragraphs.
    foreach ($this->testParagraphs as $paragraph) {
      if ($paragraph) {
        $paragraph->delete();
      }
    }

    // Clean up test content types.
    foreach ($this->testContentTypes as $type) {
      if ($type) {
        $type->delete();
      }
    }

    // Clean up test paragraph types.
    foreach ($this->testParagraphTypes as $type) {
      if ($type) {
        $type->delete();
      }
    }

    // Run uninstall actions.
    $this->plugin->paragraphsUninstallActions();

    // Clear all cache.
    $this->pluginRepository->clearAllCache();

    parent::tearDown();
  }

  /**
   * Creates test paragraph types.
   */
  protected function createTestParagraphTypes(): void {
    // Create text paragraph type.
    $text_type = ParagraphsType::create([
      'id' => 'text_paragraph',
      'label' => 'Text Paragraph',
    ]);
    $text_type->save();
    $this->testParagraphTypes['text_paragraph'] = $text_type;

    // Create image paragraph type.
    $image_type = ParagraphsType::create([
      'id' => 'image_paragraph',
      'label' => 'Image Paragraph',
    ]);
    $image_type->save();
    $this->testParagraphTypes['image_paragraph'] = $image_type;

    // Create quote paragraph type.
    $quote_type = ParagraphsType::create([
      'id' => 'quote_paragraph',
      'label' => 'Quote Paragraph',
    ]);
    $quote_type->save();
    $this->testParagraphTypes['quote_paragraph'] = $quote_type;

    // Create nested paragraph type (for hierarchy testing).
    $nested_type = ParagraphsType::create([
      'id' => 'nested_paragraph',
      'label' => 'Nested Paragraph',
    ]);
    $nested_type->save();
    $this->testParagraphTypes['nested_paragraph'] = $nested_type;
  }

  /**
   * Creates test content types.
   */
  protected function createTestContentTypes(): void {
    // Create article content type.
    $article = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $article->save();
    $this->testContentTypes['article'] = $article;

    // Create page content type.
    $page = NodeType::create([
      'type' => 'page',
      'name' => 'Page',
    ]);
    $page->save();
    $this->testContentTypes['page'] = $page;
  }

  /**
   * Creates field configurations for paragraphs.
   */
  protected function createTestFieldsForParagraphs(): void {
    // Create paragraph field for article nodes.
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_paragraphs',
      'entity_type' => 'node',
      'type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'paragraph',
      ],
      'cardinality' => -1,
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'article',
      'label' => 'Paragraphs',
      'settings' => [
        'handler' => 'default:paragraph',
        'handler_settings' => [
          'target_bundles' => [
            'text_paragraph' => 'text_paragraph',
            'image_paragraph' => 'image_paragraph',
            'quote_paragraph' => 'quote_paragraph',
            'nested_paragraph' => 'nested_paragraph',
          ],
        ],
      ],
    ]);
    $field->save();

    // Create paragraph field for page nodes.
    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'page',
      'label' => 'Paragraphs',
      'settings' => [
        'handler' => 'default:paragraph',
        'handler_settings' => [
          'target_bundles' => [
            'text_paragraph' => 'text_paragraph',
            'image_paragraph' => 'image_paragraph',
          ],
        ],
      ],
    ]);
    $field->save();

    // Create nested paragraph field for nested_paragraph type.
    $nested_field_storage = FieldStorageConfig::create([
      'field_name' => 'field_nested_paragraphs',
      'entity_type' => 'paragraph',
      'type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'paragraph',
      ],
      'cardinality' => -1,
    ]);
    $nested_field_storage->save();

    $nested_field = FieldConfig::create([
      'field_storage' => $nested_field_storage,
      'bundle' => 'nested_paragraph',
      'label' => 'Nested Paragraphs',
      'settings' => [
        'handler' => 'default:paragraph',
        'handler_settings' => [
          'target_bundles' => [
            'text_paragraph' => 'text_paragraph',
            'image_paragraph' => 'image_paragraph',
          ],
        ],
      ],
    ]);
    $nested_field->save();
  }

  /**
   * Creates test paragraphs with various configurations.
   */
  protected function createTestParagraphs(): void {
    // Create text paragraphs (5 in English).
    for ($i = 1; $i <= 5; $i++) {
      $paragraph = Paragraph::create([
        'type' => 'text_paragraph',
        'langcode' => 'en',
      ]);
      $paragraph->save();
      $this->testParagraphs[] = $paragraph;
    }

    // Create image paragraphs (3 in English).
    for ($i = 1; $i <= 3; $i++) {
      $paragraph = Paragraph::create([
        'type' => 'image_paragraph',
        'langcode' => 'en',
      ]);
      $paragraph->save();
      $this->testParagraphs[] = $paragraph;
    }

    // Create quote paragraphs (2 in English).
    for ($i = 1; $i <= 2; $i++) {
      $paragraph = Paragraph::create([
        'type' => 'quote_paragraph',
        'langcode' => 'en',
      ]);
      $paragraph->save();
      $this->testParagraphs[] = $paragraph;
    }

    // Create nested paragraphs for hierarchy testing.
    for ($i = 1; $i <= 2; $i++) {
      $paragraph = Paragraph::create([
        'type' => 'nested_paragraph',
        'langcode' => 'en',
      ]);
      $paragraph->save();
      $this->testParagraphs[] = $paragraph;
    }

    // Create paragraphs with multiple revisions for revision testing.
    $revision_test_paragraph = Paragraph::create([
      'type' => 'text_paragraph',
      'langcode' => 'en',
    ]);
    $revision_test_paragraph->save();

    // Create multiple revisions.
    for ($i = 1; $i <= 5; $i++) {
      $revision_test_paragraph->setNewRevision(TRUE);
      $revision_test_paragraph->save();
    }
    $this->testParagraphs[] = $revision_test_paragraph;
  }

  /**
   * Creates test nodes with attached paragraphs.
   */
  protected function createTestNodesWithParagraphs(): void {
    // Create article with text paragraphs.
    $node1 = Node::create([
      'type' => 'article',
      'title' => 'Article with Text Paragraphs',
      'langcode' => 'en',
      'status' => 1,
      'field_paragraphs' => [
        ['target_id' => $this->testParagraphs[0]->id(), 'target_revision_id' => $this->testParagraphs[0]->getRevisionId()],
        ['target_id' => $this->testParagraphs[1]->id(), 'target_revision_id' => $this->testParagraphs[1]->getRevisionId()],
      ],
    ]);
    $node1->save();
    $this->testNodes[] = $node1;

    // Create article with image paragraphs.
    $node2 = Node::create([
      'type' => 'article',
      'title' => 'Article with Image Paragraphs',
      'langcode' => 'en',
      'status' => 1,
      'field_paragraphs' => [
        ['target_id' => $this->testParagraphs[5]->id(), 'target_revision_id' => $this->testParagraphs[5]->getRevisionId()],
      ],
    ]);
    $node2->save();
    $this->testNodes[] = $node2;

    // Create page with text paragraphs.
    $node3 = Node::create([
      'type' => 'page',
      'title' => 'Page with Paragraphs',
      'langcode' => 'en',
      'status' => 1,
      'field_paragraphs' => [
        ['target_id' => $this->testParagraphs[2]->id(), 'target_revision_id' => $this->testParagraphs[2]->getRevisionId()],
      ],
    ]);
    $node3->save();
    $this->testNodes[] = $node3;

    // Create nested paragraph structure for hierarchy testing.
    $child_paragraph = Paragraph::create([
      'type' => 'text_paragraph',
      'langcode' => 'en',
    ]);
    $child_paragraph->save();

    $parent_paragraph = Paragraph::create([
      'type' => 'nested_paragraph',
      'langcode' => 'en',
      'field_nested_paragraphs' => [
        ['target_id' => $child_paragraph->id(), 'target_revision_id' => $child_paragraph->getRevisionId()],
      ],
    ]);
    $parent_paragraph->save();

    $node4 = Node::create([
      'type' => 'article',
      'title' => 'Article with Nested Paragraphs',
      'langcode' => 'en',
      'status' => 1,
      'field_paragraphs' => [
        ['target_id' => $parent_paragraph->id(), 'target_revision_id' => $parent_paragraph->getRevisionId()],
      ],
    ]);
    $node4->save();
    $this->testNodes[] = $node4;
  }

  /**
   * Invokes a protected method using reflection.
   *
   * Helper method for testing protected/private methods.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   Method parameters.
   *
   * @return mixed
   *   Method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Tests paragraphsTypes() counts paragraphs correctly.
   */
  public function testParagraphsTypesCountsCorrectly(): void {
    $result = $this->plugin->getDataOperationResult('paragraphs_count_type');
    $results_table = $result['results_table'];

    // Find text_paragraph row (5 base + 1 from revision test + 1 from nested = 7).
    $text_row = array_filter($results_table, function ($row) {
      return isset($row[0]) && $row[0] === 'text_paragraph';
    });
    $text_row = reset($text_row);
    $this->assertNotFalse($text_row);
    $this->assertGreaterThanOrEqual(6, $text_row[2], 'Text paragraph count should be at least 6');

    // Find image_paragraph row (should have 3).
    $image_row = array_filter($results_table, function ($row) {
      return isset($row[0]) && $row[0] === 'image_paragraph';
    });
    $image_row = reset($image_row);
    $this->assertNotFalse($image_row);
    $this->assertEquals(3, $image_row[2], 'Image paragraph count should be 3');

    // Find quote_paragraph row (should have 2).
    $quote_row = array_filter($results_table, function ($row) {
      return isset($row[0]) && $row[0] === 'quote_paragraph';
    });
    $quote_row = reset($quote_row);
    $this->assertNotFalse($quote_row);
    $this->assertEquals(2, $quote_row[2], 'Quote paragraph count should be 2');
  }

  /**
   * Tests paragraphsTypes() includes total row.
   */
  public function testParagraphsTypesIncludesTotalRow(): void {
    $result = $this->plugin->getDataOperationResult('paragraphs_count_type');
    $results_table = $result['results_table'];

    $last_row = end($results_table);

    // Verify total row structure.
    $this->assertIsArray($last_row);
    $this->assertStringContainsString('TOTAL', (string) $last_row[0]);
    $this->assertEquals('', $last_row[1]);

    // Verify total is greater than 0.
    $this->assertGreaterThan(0, $last_row[2]);
  }

  /**
   * Tests paragraphsItemsCountByLanguage() returns correct structure.
   */
  public function testParagraphsItemsCountByLanguageReturnsCorrectStructure(): void {
    $result = $this->plugin->getDataOperationResult('paragraphs_count_langcode');

    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header has correct columns.
    $this->assertCount(4, $result['header_table']);
  }

  /**
   * Tests paragraphsItemsCountByLanguage() groups by language correctly.
   *
   */
  public function testParagraphsItemsCountByLanguageGroupsCorrectly(): void {
    $result = $this->plugin->getDataOperationResult('paragraphs_count_langcode');
    $results_table = $result['results_table'];

    $this->assertNotEmpty($results_table);

    // All test paragraphs are English, verify langcode is 'en'.
    foreach ($results_table as $row) {
      $this->assertEquals('en', $row['langcode']);
      $this->assertArrayHasKey('label', $row);
      $this->assertArrayHasKey('id', $row);
      $this->assertArrayHasKey('total', $row);
    }
  }

  /**
   * Tests paragraphsTypesHierarchy() includes hierarchy levels.
   *
   */
  public function testParagraphsTypesHierarchyIncludesLevels(): void {
    $result = $this->plugin->getDataOperationResult('paragraphs_count_hierarchy');
    $results_table = $result['results_table'];

    $this->assertNotEmpty($results_table);

    // Find level 1 rows (excluding TOTAL row).
    $level_1_rows = array_filter($results_table, function ($row) {
      // Skip TOTAL row.
      if (is_object($row[0]) || strpos((string) $row[0], 'Total') !== FALSE) {
        return FALSE;
      }
      return isset($row[0]) && $row[0] == 1;
    });

    $this->assertNotEmpty($level_1_rows, 'Should have level 1 paragraphs');
  }

  /**
   * Tests paragraphsTypesHierarchy() shows parent bundles correctly.
   *
   */
  public function testParagraphsTypesHierarchyShowsParentBundles(): void {
    $result = $this->plugin->getDataOperationResult('paragraphs_count_hierarchy');
    $results_table = $result['results_table'];

    // Find level 1 rows (should have parent_bundle = node/article/page), excluding TOTAL row.
    $level_1_rows = array_filter($results_table, function ($row) {
      // Skip TOTAL row.
      if (is_object($row[0]) || strpos((string) $row[0], 'Total') !== FALSE) {
        return FALSE;
      }
      return isset($row[0]) && $row[0] == 1 && !empty($row[1]);
    });

    $this->assertNotEmpty($level_1_rows, 'Level 1 rows should have parent bundle');

    // Verify parent bundles are entity types (node, etc.).
    foreach ($level_1_rows as $row) {
      $this->assertNotEmpty($row[1], 'Parent bundle should not be empty');
    }
  }

  /**
   * Tests paragraphsTypesHierarchy() includes total row.
   *
   */
  public function testParagraphsTypesHierarchyIncludesTotalRow(): void {
    $result = $this->plugin->getDataOperationResult('paragraphs_count_hierarchy');
    $results_table = $result['results_table'];

    $last_row = end($results_table);

    // Verify total row.
    $this->assertStringContainsString('Total', (string) $last_row[0]);
    $this->assertEquals('', $last_row[1]);
    $this->assertEquals('', $last_row[2]);
    $this->assertGreaterThan(0, $last_row[3]);
  }

  /**
   * Tests paragraphsUsePlace() returns correct structure.
   *
   */
  public function testParagraphsUsePlaceReturnsCorrectStructure(): void {
    // Create a new request stack to avoid session issues.
    $request = Request::create(
      '/admin/reports/xray-audit',
      'GET',
      ['parent' => 'node', 'bundle' => 'text_paragraph']
    );

    $request_stack = new RequestStack();
    $request_stack->push($request);

    // Use reflection to replace the request stack in the plugin.
    $reflection = new \ReflectionClass($this->plugin);
    $property = $reflection->getProperty('requestStack');
    $property->setAccessible(TRUE);
    $property->setValue($this->plugin, $request_stack);

    $result = $this->invokeProtectedMethod($this->plugin, 'paragraphsUsePlace');

    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header has 7 columns.
    $this->assertCount(7, $result['header_table']);
  }

  /**
   * Tests getParagraphsGroupedByRevision() returns correct structure.
   *
   */
  public function testGetParagraphsGroupedByRevisionReturnsCorrectStructure(): void {
    $result = $this->plugin->getDataOperationResult('paragraphs_revisions_count');

    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);
    $this->assertArrayHasKey('extended_data_render', $result);

    // Verify header has 4 columns.
    $this->assertCount(4, $result['header_table']);

    // Verify extended render includes library.
    $this->assertArrayHasKey('#attached', $result['extended_data_render']);
    $this->assertArrayHasKey('library', $result['extended_data_render']['#attached']);
  }

  /**
   * Tests getParagraphsGroupedByRevision() shows paragraphs with most revisions.
   *
   */
  public function testGetParagraphsGroupedByRevisionShowsHighestCounts(): void {
    $result = $this->plugin->getDataOperationResult('paragraphs_revisions_count');
    $results_table = $result['results_table'];

    $this->assertNotEmpty($results_table);

    // Verify results are ordered by revision count (descending).
    $previous_count = PHP_INT_MAX;
    foreach ($results_table as $row) {
      if (isset($row['revisions']['data'])) {
        $current_count = $row['revisions']['data'];
        $this->assertLessThanOrEqual($previous_count, $current_count);
        $previous_count = $current_count;
      }
    }
  }

  /**
   * Tests getParagraphsGroupedByRevision() includes paragraph labels.
   *
   */
  public function testGetParagraphsGroupedByRevisionIncludesLabels(): void {
    $result = $this->plugin->getDataOperationResult('paragraphs_revisions_count');
    $results_table = $result['results_table'];

    foreach ($results_table as $row) {
      $this->assertArrayHasKey('label', $row);
      $this->assertNotEmpty($row['label']);
    }
  }

}
