<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentModel;

use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;
use Drupal\Core\Entity\Entity\EntityViewDisplay;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Comprehensive tests for XrayAuditEntityArchitecturePlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditEntityArchitecturePluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'link',
    'image',
    'file',
    'datetime',
    'options',
    'taxonomy',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The entity architecture plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentModel\XrayAuditEntityArchitecturePlugin
   */
  protected $plugin;

  /**
   * The entity architecture service.
   *
   * @var \Drupal\xray_audit\Services\EntityArchitectureInterface
   */
  protected $entityArchitectureService;

  /**
   * The plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install additional schemas for complex field testing.
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);
    $this->installConfig(['field', 'node', 'taxonomy', 'file', 'image']);

    // Get services.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');
    $this->entityArchitectureService = $this->container->get('xray_audit.entity_architecture');
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('entity_architecture');

    // Set up test content types and fields.
    $this->setUpTestContentTypes();
    $this->setUpTestFields();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    parent::tearDown();
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Sets up test content types.
   *
   * Creates multiple content types to test entity architecture analysis
   * across different bundles.
   */
  protected function setUpTestContentTypes(): void {
    // Create article content type.
    $this->createContentType('article', 'Article');

    // Create page content type.
    $this->createContentType('page', 'Page');

    // Create blog content type.
    $this->createContentType('blog', 'Blog');

    // Create taxonomy vocabulary for entity reference testing.
    $vocabulary = Vocabulary::create([
      'vid' => 'tags',
      'name' => 'Tags',
    ]);
    $vocabulary->save();
  }

  /**
   * Sets up test fields on content types.
   *
   * Creates various field types to comprehensively test entity architecture
   * analysis including text, link, entity reference, datetime, etc.
   */
  protected function setUpTestFields(): void {
    // TEXT FIELD: body field (simulating standard body field).
    $this->createTestField(
      'field_body_text',
      'node',
      'article',
      'text_long',
      [],
      ['required' => FALSE]
    );

    // LINK FIELD: external link field.
    $this->createTestField(
      'field_external_link',
      'node',
      'article',
      'link',
      [],
      ['required' => FALSE, 'title' => 1]
    );

    // ENTITY REFERENCE FIELD: taxonomy term reference.
    $this->createTestField(
      'field_tags',
      'node',
      'article',
      'entity_reference',
      [
        'target_type' => 'taxonomy_term',
        'cardinality' => -1,
      ],
      [
        'required' => FALSE,
        'settings' => [
          'handler' => 'default:taxonomy_term',
          'handler_settings' => [
            'target_bundles' => ['tags' => 'tags'],
          ],
        ],
      ]
    );

    // REQUIRED TEXT FIELD: to test mandatory flag.
    $this->createTestField(
      'field_required_text',
      'node',
      'article',
      'string',
      [],
      ['required' => TRUE]
    );

    // INTEGER FIELD: to test numeric field types.
    $this->createTestField(
      'field_integer',
      'node',
      'page',
      'integer',
      [],
      ['required' => FALSE]
    );

    // BOOLEAN FIELD: to test boolean field types.
    $this->createTestField(
      'field_published',
      'node',
      'page',
      'boolean',
      [],
      ['required' => FALSE]
    );

    // EMAIL FIELD: to test email field type.
    $this->createTestField(
      'field_email',
      'node',
      'blog',
      'email',
      [],
      ['required' => FALSE]
    );

    // DATETIME FIELD: to test datetime field type.
    $this->createTestField(
      'field_publish_date',
      'node',
      'blog',
      'datetime',
      ['datetime_type' => 'datetime'],
      ['required' => FALSE]
    );

    // Set up entity form displays with widgets.
    $this->setUpFormDisplays();

    // Set up entity view displays with formatters.
    $this->setUpViewDisplays();
  }

  /**
   * Sets up entity form displays with widgets.
   */
  protected function setUpFormDisplays(): void {
    // Article form display.
    $form_display = EntityFormDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'article',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $form_display->setComponent('field_body_text', [
      'type' => 'text_textarea',
      'weight' => 1,
    ]);
    $form_display->setComponent('field_external_link', [
      'type' => 'link_default',
      'weight' => 2,
    ]);
    $form_display->setComponent('field_tags', [
      'type' => 'entity_reference_autocomplete',
      'weight' => 3,
    ]);
    $form_display->setComponent('field_required_text', [
      'type' => 'string_textfield',
      'weight' => 4,
    ]);
    $form_display->save();

    // Page form display.
    $form_display_page = EntityFormDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'page',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $form_display_page->setComponent('field_integer', [
      'type' => 'number',
      'weight' => 1,
    ]);
    $form_display_page->setComponent('field_published', [
      'type' => 'boolean_checkbox',
      'weight' => 2,
    ]);
    $form_display_page->save();

    // Blog form display.
    $form_display_blog = EntityFormDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'blog',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $form_display_blog->setComponent('field_email', [
      'type' => 'email_default',
      'weight' => 1,
    ]);
    $form_display_blog->setComponent('field_publish_date', [
      'type' => 'datetime_default',
      'weight' => 2,
    ]);
    $form_display_blog->save();
  }

  /**
   * Sets up entity view displays with formatters.
   */
  protected function setUpViewDisplays(): void {
    // Article view display (default).
    $view_display = EntityViewDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'article',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $view_display->setComponent('field_body_text', [
      'type' => 'text_default',
      'weight' => 1,
      'label' => 'above',
    ]);
    $view_display->setComponent('field_external_link', [
      'type' => 'link',
      'weight' => 2,
      'label' => 'above',
    ]);
    $view_display->setComponent('field_tags', [
      'type' => 'entity_reference_label',
      'weight' => 3,
      'label' => 'above',
    ]);
    $view_display->setComponent('field_required_text', [
      'type' => 'string',
      'weight' => 4,
      'label' => 'above',
    ]);
    $view_display->save();

    // Article view display (teaser).
    $view_display_teaser = EntityViewDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'article',
      'mode' => 'teaser',
      'status' => TRUE,
    ]);
    $view_display_teaser->setComponent('field_body_text', [
      'type' => 'text_summary_or_trimmed',
      'weight' => 1,
      'label' => 'hidden',
    ]);
    $view_display_teaser->save();

    // Page view display (default).
    $view_display_page = EntityViewDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'page',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $view_display_page->setComponent('field_integer', [
      'type' => 'number_integer',
      'weight' => 1,
      'label' => 'above',
    ]);
    $view_display_page->setComponent('field_published', [
      'type' => 'boolean',
      'weight' => 2,
      'label' => 'above',
    ]);
    $view_display_page->save();

    // Blog view display (default).
    $view_display_blog = EntityViewDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'blog',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $view_display_blog->setComponent('field_email', [
      'type' => 'basic_string',
      'weight' => 1,
      'label' => 'above',
    ]);
    $view_display_blog->setComponent('field_publish_date', [
      'type' => 'datetime_default',
      'weight' => 2,
      'label' => 'above',
    ]);
    $view_display_blog->save();
  }

  /**
   * Test getDataOperationResult() with content_entity_definition operation.
   *
   */
  public function testGetDataOperationResultContentEntityDefinition() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Result should be an array.
    $this->assertIsArray($result);
    $this->assertNotEmpty($result, 'Should return entity architecture data');

    // Verify data structure contains expected elements.
    $found_article = FALSE;
    $found_page = FALSE;
    $found_blog = FALSE;

    foreach ($result as $item) {
      $this->assertArrayHasKey('content', $item);
      $content = $item['content'];

      // Check for our test content types.
      if (isset($content['entity']) && $content['entity'] === 'node') {
        if (isset($content['bundle']) && $content['bundle'] === 'article') {
          $found_article = TRUE;
        }
        if (isset($content['bundle']) && $content['bundle'] === 'page') {
          $found_page = TRUE;
        }
        if (isset($content['bundle']) && $content['bundle'] === 'blog') {
          $found_blog = TRUE;
        }
      }
    }

    $this->assertTrue($found_article, 'Should contain article content type');
    $this->assertTrue($found_page, 'Should contain page content type');
    $this->assertTrue($found_blog, 'Should contain blog content type');
  }

  /**
   * Test getDataOperationResult() includes entity rows.
   *
   * Verifies that entity definition rows (type='entity') are included
   * before field rows.
   *
   */
  public function testGetDataOperationResultIncludesEntityRows() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Should contain entity rows with type='entity'.
    $found_entity_row = FALSE;

    foreach ($result as $item) {
      if (isset($item['content']['type']) && $item['content']['type'] === 'entity') {
        $found_entity_row = TRUE;
        // Verify entity row structure.
        $this->assertArrayHasKey('entity', $item['content']);
        $this->assertArrayHasKey('bundle', $item['content']);
        $this->assertArrayHasKey('machine_name', $item['content']);
        $this->assertArrayHasKey('label', $item['content']);
        break;
      }
    }

    $this->assertTrue($found_entity_row, 'Should contain entity definition rows');
  }

  /**
   * Test getDataOperationResult() includes field rows.
   *
   * Verifies that field definition rows (type='base_field', 'field_config', etc.)
   * are included with complete field metadata.
   *
   */
  public function testGetDataOperationResultIncludesFieldRows() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Should contain field rows.
    $found_field_row = FALSE;

    foreach ($result as $item) {
      $type = $item['content']['type'] ?? '';
      if (in_array($type, ['base_field', 'base_field_override', 'field_config'])) {
        $found_field_row = TRUE;
        // Verify field row has expected keys.
        $this->assertArrayHasKey('machine_name', $item['content']);
        $this->assertArrayHasKey('label', $item['content']);
        $this->assertArrayHasKey('data_type', $item['content']);
        break;
      }
    }

    $this->assertTrue($found_field_row, 'Should contain field definition rows');
  }

  /**
   * Test getDataOperationResult() includes custom fields.
   *
   * Verifies that custom fields created in setUp() are properly included
   * in the entity architecture data.
   *
   */
  public function testGetDataOperationResultIncludesCustomFields() {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_definition');

    // Assert: Should contain our custom test fields.
    $found_fields = [
      'field_body_text' => FALSE,
      'field_external_link' => FALSE,
      'field_tags' => FALSE,
      'field_required_text' => FALSE,
      'field_integer' => FALSE,
      'field_published' => FALSE,
      'field_email' => FALSE,
      'field_publish_date' => FALSE,
    ];

    foreach ($result as $item) {
      $machine_name = $item['content']['machine_name'] ?? '';
      if (isset($found_fields[$machine_name])) {
        $found_fields[$machine_name] = TRUE;
      }
    }

    foreach ($found_fields as $field_name => $found) {
      $this->assertTrue($found, "Should contain custom field: $field_name");
    }
  }

}
