<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\Database;

use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Comprehensive tests for XrayAuditDatabaseGeneralTaskPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditDatabaseGeneralTaskPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'filter',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The database general plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\Database\XrayAuditDatabaseGeneralTaskPlugin
   */
  protected $plugin;

  /**
   * Plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Database connection service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Get services.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');
    $this->database = $this->container->get('database');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('database_general');

    // Create test tables with data to ensure meaningful database metrics.
    $this->createTestTablesWithData();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clear all cache.
    $this->pluginRepository->clearAllCache();

    parent::tearDown();
  }

  /**
   * Creates test tables with data for database metrics testing.
   */
  protected function createTestTablesWithData() {
    // Test table 1: Small table with a few rows.
    $schema = [
      'description' => 'Test table 1 for database metrics',
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
          'description' => 'Primary key',
        ],
        'name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
          'default' => '',
          'description' => 'Name field',
        ],
        'data' => [
          'type' => 'text',
          'size' => 'normal',
          'not null' => FALSE,
          'description' => 'Data field',
        ],
      ],
      'primary key' => ['id'],
      'indexes' => [
        'name_idx' => ['name'],
      ],
    ];

    try {
      $this->database->schema()->createTable('test_xray_table1', $schema);

      // Insert some data.
      for ($i = 1; $i <= 5; $i++) {
        $this->database->insert('test_xray_table1')
          ->fields([
            'name' => "Test entry $i",
            'data' => str_repeat("Sample data $i ", 100),
          ])
          ->execute();
      }
    }
    catch (\Exception $e) {
      // Table might already exist in some test scenarios.
    }

    // Test table 2: Medium table with more rows.
    $schema2 = [
      'description' => 'Test table 2 for database metrics',
      'fields' => [
        'id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'value' => [
          'type' => 'int',
          'not null' => TRUE,
          'default' => 0,
        ],
        'description' => [
          'type' => 'text',
          'size' => 'big',
          'not null' => FALSE,
        ],
      ],
      'primary key' => ['id'],
      'indexes' => [
        'value_idx' => ['value'],
      ],
    ];

    try {
      $this->database->schema()->createTable('test_xray_table2', $schema2);

      // Insert more data.
      for ($i = 1; $i <= 20; $i++) {
        $this->database->insert('test_xray_table2')
          ->fields([
            'value' => $i * 100,
            'description' => str_repeat("Larger data set entry $i ", 200),
          ])
          ->execute();
      }
    }
    catch (\Exception $e) {
      // Table might already exist.
    }
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Tests tableSize() returns correct data structure.
   *
   */
  public function testTableSizeReturnsProperStructure() {
    // Act.
    $result = $this->plugin->getDataOperationResult('database');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);
    $this->assertArrayHasKey('extended_data_render', $result);
  }

  /**
   * Tests tableSize() returns correct header structure.
   *
   */
  public function testTableSizeReturnsCorrectHeader() {
    // Act.
    $result = $this->plugin->getDataOperationResult('database');

    // Assert: Should have 5 header columns.
    $this->assertCount(5, $result['header_table'], 'Should have 5 header columns');

    // Verify headers are TranslatableMarkup objects or strings.
    foreach ($result['header_table'] as $header) {
      $this->assertTrue(
        is_string($header) || is_object($header),
        'Header should be string or TranslatableMarkup'
      );
    }
  }

  /**
   * Tests tableSize() includes all database tables.
   *
   */
  public function testTableSizeIncludesAllTables() {
    // Act.
    $result = $this->plugin->getDataOperationResult('database');

    // Assert: Should have multiple tables.
    $this->assertIsArray($result['results_table']);
    $this->assertNotEmpty($result['results_table'], 'Should have at least some tables');

    // Verify each row has required fields.
    foreach ($result['results_table'] as $row) {
      $this->assertArrayHasKey('name', $row);
      $this->assertArrayHasKey('rows', $row);
      $this->assertArrayHasKey('size', $row);
      $this->assertArrayHasKey('index_size', $row);
      $this->assertArrayHasKey('total_size', $row);
    }
  }

}
