<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\Webform;

use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Comprehensive tests for XrayAuditWebformGeneralTaskPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditWebformGeneralTaskPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'file',
    'webform',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The webform plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\Webform\XrayAuditWebformGeneralTaskPlugin
   */
  protected $plugin;

  /**
   * Plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('webform');
    $this->installEntitySchema('webform_submission');
    $this->installSchema('webform', ['webform']);
    $this->installSchema('user', ['users_data']);
    $this->installConfig(['webform']);

    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');
    $this->plugin = $this->taskPluginManager->createInstance('webform_report');
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clear cache.
    $this->pluginRepository->clearAllCache();

    parent::tearDown();
  }

  /**
   * Helper to invoke protected methods.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Tests getDataOperationResult() returns proper structure.
   */
  public function testGetDataOperationResultReturnsProperStructure() {
    $result = $this->plugin->getDataOperationResult('webforms');

    $this->assertIsArray($result);
    $this->assertArrayHasKey('num_webforms', $result);
    $this->assertArrayHasKey('results_table', $result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertIsArray($result['results_table']);
    $this->assertIsArray($result['header_table']);
  }

  /**
   * Tests getRows() includes required webform data fields.
   */
  public function testGetRowsIncludesRequiredDataFields() {
    $rows = $this->plugin->getRows('webforms');

    if (!empty($rows)) {
      $first_row = reset($rows);

      $this->assertArrayHasKey('title', $first_row);
      $this->assertArrayHasKey('id', $first_row);
      $this->assertArrayHasKey('status', $first_row);
      $this->assertArrayHasKey('num_fields', $first_row);
      $this->assertArrayHasKey('handlers', $first_row);
      $this->assertArrayHasKey('edit_link', $first_row);
    }
    else {
      $this->markTestSkipped('No webforms available for data field testing');
    }
  }

  /**
   * Tests buildDataRenderArray() creates table for webforms operation.
   */
  public function testBuildDataRenderArrayCreatesTableForWebformsOperation() {
    $data = [
      'header_table' => [
        'Title',
        'ID',
        'Status',
        'Number of Fields',
        'Handlers',
        'Webform',
      ],
      'results_table' => [
        [
          'title' => 'Test Webform',
          'id' => 'test_webform',
          'status' => 'Open',
          'num_fields' => 5,
          'handlers' => 'email',
          'edit_link' => 'Edit link',
        ],
      ],
      'num_webforms' => 1,
    ];

    $build = $this->plugin->buildDataRenderArray($data, 'webforms');

    $this->assertIsArray($build);
    $this->assertArrayHasKey('table', $build);
    $this->assertEquals('table', $build['table']['#type']);
    $this->assertArrayHasKey('#header', $build['table']);
    $this->assertArrayHasKey('#rows', $build['table']);
  }

  /**
   * Tests buildDataRenderArray() has correct header columns.
   */
  public function testBuildDataRenderArrayHasCorrectHeaderColumns() {
    $data = [
      'header_table' => [
        'Title',
        'ID',
        'Status',
        'Number of Fields',
        'Handlers',
        'Webform',
      ],
      'results_table' => [
        [
          'title' => 'Test',
          'id' => 'test',
          'status' => 'Open',
          'num_fields' => 1,
          'handlers' => '',
          'edit_link' => '',
        ],
      ],
      'num_webforms' => 1,
    ];

    $build = $this->plugin->buildDataRenderArray($data, 'webforms');

    // Should have 6 columns: Title, ID, Status, Number of Fields, Handlers, Webform.
    $this->assertCount(6, $build['table']['#header']);
  }

  /**
   * Tests buildDataRenderArray() includes download button.
   */
  public function testBuildDataRenderArrayIncludesDownloadButton() {
    $data = [
      'header_table' => [
        'Title',
        'ID',
        'Status',
        'Number of Fields',
        'Handlers',
        'Webform',
      ],
      'results_table' => [
        [
          'title' => 'Test',
          'id' => 'test',
          'status' => 'Open',
          'num_fields' => 1,
          'handlers' => '',
          'edit_link' => '',
        ],
      ],
      'num_webforms' => 1,
    ];

    $build = $this->plugin->buildDataRenderArray($data, 'webforms');

    $this->assertIsArray($build);
    $this->assertArrayHasKey('download_button', $build);
  }

  /**
   * Tests prepareCsvHeaders() returns proper headers for webforms operation.
   */
  public function testPrepareCsvHeadersWebforms() {
    // Arrange.
    $operation = 'webforms';

    // Act.
    $headers = $this->plugin->prepareCsvHeaders($operation);

    // Assert.
    $this->assertIsArray($headers);
    $this->assertNotEmpty($headers);
    $this->assertCount(6, $headers);

    // Verify all headers are strings (TranslatableMarkup converted).
    foreach ($headers as $header) {
      $this->assertIsString($header);
    }
  }

  /**
   * Tests prepareCsvData() returns flattened data for CSV export.
   */
  public function testPrepareCsvDataWebforms() {
    // Arrange.
    $operation = 'webforms';

    // Act.
    $csv_data = $this->plugin->prepareCsvData($operation, []);

    // Assert.
    $this->assertIsArray($csv_data);

    // If there are rows, verify they are properly flattened.
    if (!empty($csv_data)) {
      $first_row = reset($csv_data);
      $this->assertIsArray($first_row);

      // Verify expected keys exist in the flattened data.
      $this->assertArrayHasKey('title', $first_row);
      $this->assertArrayHasKey('id', $first_row);
      $this->assertArrayHasKey('status', $first_row);
      $this->assertArrayHasKey('num_fields', $first_row);
      $this->assertArrayHasKey('handlers', $first_row);
    }
  }

}
