<?php

namespace Drupal\Tests\xray_audit\Kernel\Services;

use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\system\Entity\Menu;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Tests the NavigationArchitecture service.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class NavigationArchitectureTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'link',
    'menu_link_content',
    'xray_audit',
  ];

  /**
   * The navigation architecture service under test.
   *
   * @var \Drupal\xray_audit\Services\NavigationArchitecture
   */
  protected $navigationArchitecture;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install menu link content schema.
    $this->installEntitySchema('menu_link_content');

    // Get the service under test.
    $this->navigationArchitecture = $this->container->get('xray_audit.navigation_architecture');
  }

  /**
   * Tests getMenuList() returns available menus.
   *
   */
  public function testGetMenuList() {
    // Act.
    $menus = $this->navigationArchitecture->getMenuList();

    // Assert.
    $this->assertIsArray($menus);
    $this->assertNotEmpty($menus, 'Should have at least one menu');

    // System menus like 'main' should be present.
    // Admin menu should be excluded.
    $this->assertArrayNotHasKey('admin', $menus, 'Admin menu should be excluded');
  }

  /**
   * Tests getMenuList() excludes admin menu.
   *
   */
  public function testGetMenuListExcludesAdmin() {
    // Act.
    $menus = $this->navigationArchitecture->getMenuList();

    // Assert: Admin menu should not be in the list.
    $this->assertArrayNotHasKey('admin', $menus);
  }

  /**
   * Tests getMenuArchitecture() with menu items.
   *
   */
  public function testGetMenuArchitectureWithItems() {
    // Arrange: Create menu with items.
    $menu = Menu::create([
      'id' => 'test_menu_items',
      'label' => 'Test Menu Items',
    ]);
    $menu->save();

    // Create a menu link.
    MenuLinkContent::create([
      'title' => 'Test Link 1',
      'link' => ['uri' => 'https://example.com'],
      'menu_name' => 'test_menu_items',
      'expanded' => TRUE,
    ])->save();

    // Act.
    $result = $this->navigationArchitecture->getMenuArchitecture('test_menu_items');

    // Assert.
    $this->assertIsArray($result);
    $this->assertNotEmpty($result['items']);
    $this->assertGreaterThan(0, $result['item_number']);

    // Check first item structure.
    $first_item = $result['items'][0];
    $this->assertArrayHasKey('level', $first_item);
    $this->assertArrayHasKey('title', $first_item);
    $this->assertArrayHasKey('link', $first_item);
    $this->assertArrayHasKey('enabled', $first_item);
    $this->assertArrayHasKey('parent', $first_item);
    $this->assertArrayHasKey('route_name', $first_item);
    $this->assertArrayHasKey('target', $first_item);
  }

  /**
   * Tests getMenuArchitecture() with nested menu items.
   *
   */
  public function testGetMenuArchitectureWithNestedItems() {
    // Arrange: Create menu with nested structure.
    $menu = Menu::create([
      'id' => 'test_nested_menu',
      'label' => 'Test Nested Menu',
    ]);
    $menu->save();

    // Create parent link.
    $parent = MenuLinkContent::create([
      'title' => 'Parent Link',
      'link' => ['uri' => 'route:<nolink>'],
      'menu_name' => 'test_nested_menu',
      'expanded' => TRUE,
    ]);
    $parent->save();

    // Create child link.
    MenuLinkContent::create([
      'title' => 'Child Link',
      'link' => ['uri' => 'https://example.com/child'],
      'menu_name' => 'test_nested_menu',
      'parent' => $parent->getPluginId(),
      'expanded' => TRUE,
    ])->save();

    // Act.
    $result = $this->navigationArchitecture->getMenuArchitecture('test_nested_menu');

    // Assert.
    $this->assertGreaterThanOrEqual(2, $result['item_number']);
    $this->assertGreaterThanOrEqual(2, $result['level_max']);

    // Check levels.
    $has_level_1 = FALSE;
    $has_level_2 = FALSE;
    foreach ($result['items'] as $item) {
      if ($item['level'] === 1) {
        $has_level_1 = TRUE;
      }
      if ($item['level'] === 2) {
        $has_level_2 = TRUE;
      }
    }

    $this->assertTrue($has_level_1);
    $this->assertTrue($has_level_2);
  }

  /**
   * Tests getMenuArchitecture() counts items correctly.
   *
   */
  public function testGetMenuArchitectureCountsItems() {
    // Arrange: Create menu with multiple items.
    $menu = Menu::create([
      'id' => 'test_count_menu',
      'label' => 'Test Count Menu',
    ]);
    $menu->save();

    // Create 3 menu links.
    for ($i = 1; $i <= 3; $i++) {
      MenuLinkContent::create([
        'title' => "Link $i",
        'link' => ['uri' => "https://example.com/$i"],
        'menu_name' => 'test_count_menu',
      ])->save();
    }

    // Act.
    $result = $this->navigationArchitecture->getMenuArchitecture('test_count_menu');

    // Assert.
    $this->assertEquals(3, $result['item_number']);
    $this->assertCount(3, $result['items']);
  }

  /**
   * Tests getMenuArchitecture() handles disabled links.
   *
   */
  public function testGetMenuArchitectureWithDisabledLink() {
    // Arrange: Create menu with disabled link.
    $menu = Menu::create([
      'id' => 'test_disabled_menu',
      'label' => 'Test Disabled Menu',
    ]);
    $menu->save();

    MenuLinkContent::create([
      'title' => 'Disabled Link',
      'link' => ['uri' => 'https://example.com/disabled'],
      'menu_name' => 'test_disabled_menu',
      'enabled' => FALSE,
    ])->save();

    // Act.
    $result = $this->navigationArchitecture->getMenuArchitecture('test_disabled_menu');

    // Assert: Method should still process disabled links.
    $this->assertNotEmpty($result['items']);

    // Check if we have a disabled link.
    $has_disabled = FALSE;
    foreach ($result['items'] as $item) {
      if ($item['enabled'] === FALSE) {
        $has_disabled = TRUE;
        break;
      }
    }

    // Disabled links are included in the architecture.
    $this->assertTrue($has_disabled, 'Should include disabled links in architecture');
  }

  /**
   * Tests getMenuArchitecture() with external link.
   *
   */
  public function testGetMenuArchitectureExternalLink() {
    // Arrange.
    $menu = Menu::create(['id' => 'test_external', 'label' => 'Test External']);
    $menu->save();

    MenuLinkContent::create([
      'title' => 'External Link',
      'link' => ['uri' => 'https://drupal.org'],
      'menu_name' => 'test_external',
    ])->save();

    // Act.
    $result = $this->navigationArchitecture->getMenuArchitecture('test_external');

    // Assert.
    $this->assertNotEmpty($result['items']);
    $item = $result['items'][0];

    // External links have empty route_name.
    $this->assertEquals('', $item['route_name']);
  }

  /**
   * Tests getMenuList() with custom menu.
   *
   */
  public function testGetMenuListWithCustomMenu() {
    // Arrange: Create custom menu.
    Menu::create([
      'id' => 'custom_test_menu',
      'label' => 'Custom Test Menu',
    ])->save();

    // Act.
    $menus = $this->navigationArchitecture->getMenuList();

    // Assert.
    $this->assertArrayHasKey('custom_test_menu', $menus);
    $this->assertEquals('Custom Test Menu', $menus['custom_test_menu']);
  }

  /**
   * Tests menu architecture max level tracking.
   *
   */
  public function testMenuArchitectureMaxLevel() {
    // Arrange: Create menu with 3 levels.
    $menu = Menu::create(['id' => 'test_max_level', 'label' => 'Test Max Level']);
    $menu->save();

    $level1 = MenuLinkContent::create([
      'title' => 'Level 1',
      'link' => ['uri' => 'route:<nolink>'],
      'menu_name' => 'test_max_level',
      'expanded' => TRUE,
    ]);
    $level1->save();

    $level2 = MenuLinkContent::create([
      'title' => 'Level 2',
      'link' => ['uri' => 'route:<nolink>'],
      'menu_name' => 'test_max_level',
      'parent' => $level1->getPluginId(),
      'expanded' => TRUE,
    ]);
    $level2->save();

    MenuLinkContent::create([
      'title' => 'Level 3',
      'link' => ['uri' => 'https://example.com'],
      'menu_name' => 'test_max_level',
      'parent' => $level2->getPluginId(),
    ])->save();

    // Act.
    $result = $this->navigationArchitecture->getMenuArchitecture('test_max_level');

    // Assert.
    $this->assertEquals(3, $result['level_max']);
  }

}
