# Xray Audit Module - Architecture Documentation

## 1. Module Overview

### Purpose
Xray Audit is a comprehensive Drupal analysis and reporting module that provides deep insights into site architecture, content metrics, and configuration. It generates detailed reports on various aspects of a Drupal site, helping developers, site builders, and administrators understand how their site is constructed and identify optimization opportunities.

### Core Functionality
- **Content Architecture Analysis**: Entity types, bundles, fields, and their relationships
- **Content Metrics**: Usage statistics for nodes, paragraphs, media, vocabularies, and blocks
- **Display Configuration**: View modes, field formatters, and display settings analysis
- **Site Structure**: Navigation menus, views configuration, and information architecture
- **Access Control**: User roles, permissions, and access control configuration
- **Package Management**: Module and theme analysis with dependency tracking
- **Database Analysis**: Database metrics, table sizes, and performance indicators
- **Layout System**: Block layout and region configuration
- **Form System**: Webform configuration and structure analysis (when available)
- **CSV Export**: Bulk export of all reports to CSV format with batch processing
- **Caching System**: Sophisticated caching with tag-based invalidation

### Key Features
- **Plugin-based Architecture**: Extensible system using Group and Task plugins
- **Dynamic Routing**: Routes automatically generated from plugin definitions
- **Batch Processing**: Support for large-scale data operations
- **CSV Export**: Individual and bulk CSV downloads with ZIP archive generation
- **Performance Optimized**: Single-pass processing, field definition caching, lazy loading
- **Test Coverage**: 90+ tests ensuring reliability and maintainability

### Version Compatibility
- **Core Requirement**: Drupal 10.x, 11.x
- **PHP Requirement**: 8.0+

## 2. Directory Structure

```
xray_audit/
├── assets/
│   ├── css/               # Stylesheets for reports and UI
│   │   ├── xray-audit.css
│   │   ├── xray-audit-color.css
│   │   └── xray-audit-examples-preview.css
│   └── js/                # JavaScript for interactive features
│       ├── xray-audit.js
│       ├── table-filter.js
│       └── visibility-toggler.js
├── config/
│   ├── install/           # Default configuration
│   │   └── xray_audit.views_report.yml
│   └── schema/           # Configuration schemas
│       └── xray_audit.schema.yml
├── modules/
│   └── xray_audit_insight/  # Submodule for insights and warnings
│       ├── config/schema/
│       ├── src/
│       │   ├── Annotation/
│       │   ├── Form/
│       │   └── Plugin/insights/
│       └── [module files]
├── src/
│   ├── Annotation/        # Plugin annotation classes
│   │   ├── XrayAuditGroupPlugin.php
│   │   └── XrayAuditTaskPlugin.php
│   ├── Controller/        # Route controllers
│   │   ├── XrayAuditHomeController.php
│   │   ├── XrayAuditGroupsController.php
│   │   ├── XrayAuditTaskController.php
│   │   ├── XrayAuditDisplayModeExampleController.php
│   │   └── XrayAuditBuildTaskItemsInControllerTrait.php
│   ├── Drush/
│   │   └── Commands/     # Drush command implementations
│   │       └── XrayAuditCommands.php
│   ├── Form/             # Form classes
│   │   ├── FlushCacheForm.php
│   │   ├── MenuSelectorForm.php
│   │   └── SettingsForm.php
│   ├── Plugin/           # Plugin system base classes and managers
│   │   ├── XrayAuditGroupPluginBase.php
│   │   ├── XrayAuditGroupPluginInterface.php
│   │   ├── XrayAuditGroupPluginManager.php
│   │   ├── XrayAuditTaskPluginBase.php
│   │   ├── XrayAuditTaskPluginInterface.php
│   │   ├── XrayAuditTaskPluginManager.php
│   │   └── xray_audit/
│   │       ├── groups/   # Group plugin implementations
│   │       └── tasks/    # Task plugin implementations
│   ├── Routing/          # Route subscriber
│   │   └── RouteSubscriber.php
│   ├── Services/         # Service classes
│   │   ├── CacheManager.php & Interface
│   │   ├── CsvDownloadManager.php & Interface
│   │   ├── EntityArchitecture.php & Interface
│   │   ├── EntityDisplayArchitecture.php & Interface
│   │   ├── EntityUseBase.php & Interface
│   │   ├── EntityUseNode.php
│   │   ├── EntityUseParagraph.php
│   │   ├── NavigationArchitecture.php & Interface
│   │   ├── ParagraphUsageMap.php
│   │   ├── PluginRepository.php & Interface
│   │   ├── StorageNode.php
│   │   └── StorageTree.php
│   └── Utils/            # Utility classes
│       ├── XrayAuditBatchHelper.php
│       ├── XrayAuditDeprecationHelper.php
│       └── XrayAuditTableFilter.php
├── templates/            # Twig templates
│   ├── page--xray-audit.html.twig
│   └── xray-audit-popup.html.twig
├── tests/                # Test suites
│   ├── modules/          # Test modules
│   └── src/
│       ├── Functional/   # Functional tests
│       ├── Kernel/       # Kernel tests
│       └── Unit/         # Unit tests
├── xray_audit.info.yml
├── xray_audit.install
├── xray_audit.libraries.yml
├── xray_audit.links.menu.yml
├── xray_audit.module
├── xray_audit.permissions.yml
├── xray_audit.routing.yml
├── xray_audit.services.yml
└── XrayAuditTaskCsvDownloadTrait.php
```

## 3. Service Architecture

### Service Dependency Tree

```
PluginRepository (Central Hub)
├── XrayAuditGroupPluginManager
├── XrayAuditTaskPluginManager
├── CacheManager
│   └── cache.xray_audit (cache bin)
└── Logger

CsvDownloadManager
├── RequestStack
└── PluginRepository

EntityArchitecture
├── EntityTypeManager
├── EntityFieldManager
├── EntityDisplayRepository
└── EntityBundleInfo

EntityDisplayArchitecture
├── EntityTypeManager
├── EntityArchitecture
└── Renderer

NavigationArchitecture
├── EntityTypeManager
├── Renderer
├── MenuLinkTree
└── MenuLinkManager

EntityUseParagraph
├── EntityArchitecture
├── EntityTypeManager
├── EntityBundleInfo
├── ConfigFactory
└── ParagraphUsageMap
    ├── EntityArchitecture
    ├── EntityTypeManager
    └── Database

EntityUseNode
├── EntityArchitecture
├── EntityTypeManager
├── EntityBundleInfo
└── ConfigFactory
```

### Core Services

#### cache.xray_audit
- **Purpose**: Custom cache bin for module data
- **Factory**: Core cache factory
- **Usage**: Stores report data with tag-based invalidation

#### xray_audit.cache_manager
- **Purpose**: Manages caching operations for report data
- **Key Methods**:
  - `setCacheTagsInv()`: Cache with tag-based invalidation
  - `setCacheTempInv()`: Cache with time-based expiration
  - `getCachedData()`: Retrieve cached data
  - `clearAllCache()`: Clear all module cache
  - `invalidateCache()`: Invalidate specific cache entries
- **Cache Pattern**: `{plugin_id}:{operation}` keys

#### xray_audit.plugin_repository
- **Purpose**: Central repository for plugin operations and route building
- **Key Responsibilities**:
  - Plugin instantiation and management
  - Dynamic route generation
  - Cache coordination
  - URL generation for reports
- **Constants**:
  - `PARAMETER_GROUP_PLUGIN`
  - `PARAMETER_TASK_PLUGIN`
  - `PARAMETER_OPERATION`
  - `PARAMETER_BATCH_OPERATION`

#### xray_audit.csv_download_manager
- **Purpose**: Handles CSV file generation and downloads
- **Features**:
  - Stream-based CSV generation for memory efficiency
  - Automatic type conversion (TranslatableMarkup, Link, Url)
  - Download link generation with preserved query parameters
  - Integration with batch processing for bulk exports

### Entity Analysis Services

#### xray_audit.entity_architecture
- **Purpose**: Analyzes entity structure, fields, and configurations
- **Key Features**:
  - Field definition extraction (BaseFieldDefinition, FieldConfig, BaseFieldOverride)
  - Bundle and entity type analysis
  - Display configuration processing
  - Field type and widget mapping
- **Constants**:
  - `TYPE_BASE_FIELD_DEFINITION`
  - `TYPE_BASE_FIELD_OVERRIDE`
  - `TYPE_BASE_FIELD_CONFIG`

#### xray_audit.entity_display_architecture
- **Purpose**: Analyzes display mode configurations and rendering
- **Features**:
  - View mode analysis
  - Field formatter configuration extraction
  - Entity rendering examples
  - Display component settings

#### xray_audit.navigation_architecture
- **Purpose**: Analyzes navigation structure and menu hierarchies
- **Features**:
  - Menu tree traversal
  - Hierarchical breadcrumb generation
  - Menu link analysis
  - Parent-child relationship mapping

### Entity Usage Services

#### xray_audit.entity_use_node
- **Purpose**: Tracks node entity usage and statistics
- **Features**:
  - Published/unpublished counts
  - Bundle-based statistics
  - Content revision tracking

#### xray_audit.entity_use_paragraph
- **Purpose**: Tracks paragraph entity usage and relationships
- **Features**:
  - Hierarchical paragraph mapping
  - Reference field tracking
  - Usage location identification

#### xray_audit.paragraph_usage_map
- **Purpose**: Creates usage maps for paragraph entities
- **Features**:
  - Temporary table creation for analysis
  - Parent-child relationship tracking
  - Cross-entity reference mapping

## 4. Plugin System Architecture

### Plugin Discovery and Management

```
Plugin Discovery Flow:
1. Annotation Parser → Plugin Definition
2. Plugin Manager → Plugin Discovery
3. Plugin Repository → Plugin Instantiation
4. Route Subscriber → Dynamic Route Generation
5. Controller → Plugin Execution
6. CSV Manager → Data Export
```

### XrayAuditGroupPlugin

**Purpose**: Organizes reports into logical categories

**Annotation Properties**:
```php
@XrayAuditGroupPlugin(
  id = "content_model",
  label = @Translation("Content Model"),
  description = @Translation("Entity architecture and field configuration"),
  sort = 10
)
```

**Implemented Groups**:
- `content_access_control`: User roles and permissions
- `content_display`: Display mode configurations
- `content_metric`: Content usage statistics
- `content_model`: Entity architecture
- `database`: Database metrics
- `form`: Form and webform reports
- `layout`: Block and layout configuration
- `package`: Module and theme information
- `site_structure`: Navigation and views

### XrayAuditTaskPlugin

**Purpose**: Generates specific reports within groups

**Annotation Properties**:
```php
@XrayAuditTaskPlugin(
  id = "entity_architecture",
  label = @Translation("Entity architecture"),
  description = @Translation("Analyze entity field architecture"),
  group = "content_model",
  sort = 5,
  operations = {
    "content_entity_definition" = {
      "label" = "Content entity definitions",
      "description" = "All content entity definitions",
      "download" = TRUE  // Enables CSV export
    }
  },
  dependencies = {"field"},
  batches = {
    "process_entities" = "processBatchEntities"
  },
  install = "installMethod",
  uninstall = "uninstallMethod",
  local_task = TRUE
)
```

**Key Methods**:
- `getDataOperationResult(string $operation)`: Fetches and caches report data
- `buildDataRenderArray(array $data, string $operation)`: Builds render array
- `getHeaders(string $operation)`: Returns CSV headers
- `getRows(string $operation)`: Returns CSV data rows
- `prepareCsvHeaders()`: Prepares headers for CSV export
- `prepareCsvData()`: Prepares data for CSV export
- `getBatchClass(string $batch_id)`: Returns batch processor method

**Task Plugin Categories**:

```
ContentAccessControl/
├── XrayAuditQueryTaskRolesPlugin
└── XrayAuditQueryTaskUserPlugin

ContentDisplay/
└── XrayAuditEntityDisplaysPlugin

ContentMetric/
├── XrayAuditQueryTaskPluginBase (base)
├── XrayAuditQueryTaskBlockPlugin
├── XrayAuditQueryTaskImageStylesPlugin
├── XrayAuditQueryTaskMediaPlugin
├── XrayAuditQueryTaskNodePlugin
├── XrayAuditQueryTaskParagraphsPlugin
└── XrayAuditQueryTaskVocabularyPlugin

ContentModel/
├── XrayAuditEntityArchitecturePlugin
├── XrayAuditQueryTaskContentEntitiesPlugin
└── XrayAuditVocabularyPlugin

Database/
├── XrayAuditDatabaseTaskPluginBase (base)
└── XrayAuditDatabaseGeneralTaskPlugin

Layout/
└── XrayAuditBlockLayoutPlugin

Packages/
├── XrayAuditModulesPlugin
└── XrayAuditThemesPlugin

SiteStructure/
├── XrayAuditNavigationArchitecturePlugin
└── XrayAuditViewsPlugin

Webform/
└── XrayAuditWebformGeneralTaskPlugin
```

### CSV Export Integration

The module uses `XrayAuditTaskCsvDownloadTrait` to provide standardized CSV export:

```php
trait XrayAuditTaskCsvDownloadTrait {
  // Prepares CSV headers from getHeaders()
  public function prepareCsvHeaders(string $operation): array;

  // Prepares CSV data from getRows()
  public function prepareCsvData(string $operation, array $data): array;

  // Handles CSV generation and download
  protected function handleCsv(string $operation, array $data): void;

  // Abstract methods plugins must implement
  abstract public function getHeaders(string $operation): array;
  abstract public function getRows(string $operation): array;
}
```

## 5. Routing and Controllers

### Static Routes

```yaml
xray_audit.home: /admin/reports/xray-audit
xray_audit.download_all_csv_batch: /admin/reports/xray-audit/download-all-csv-batch
xray_audit.group: /admin/reports/xray-audit/{group_id}
xray_audit.batch_process: /admin/reports/xray-audit/batch/{task_plugin}/{batch_process}
xray_audit.display_mode_example: /xray-audit/{entity_type}/{entity_id}/{view_mode}/example
xray_audit.example_popup: /xray-audit/{entity_type}/{entity_bundle}/{view_mode}/popup
xray_audit.development: /admin/config/development/xray_audit
xray_audit.settings: /admin/config/development/xray_audit/settings
xray_audit.download_generated_zip_file: /admin/reports/xray-audit/download-zip
```

### Dynamic Routes

Generated by `RouteSubscriber` based on plugin definitions:
- Group pages: `xray_audit.group_page.{group_id}`
- Task operations: `xray_audit.task_page.{operation_id}`

### Controllers

#### XrayAuditHomeController
- **Purpose**: Main landing page with all report groups
- **Features**:
  - Group listing with descriptions
  - Cache flush form integration
  - Bulk CSV download initiation
  - Batch process management

#### XrayAuditGroupsController
- **Purpose**: Displays tasks within a group
- **Features**:
  - Dynamic title from plugin
  - Task listing with operations
  - Uses `XrayAuditBuildTaskItemsInControllerTrait`

#### XrayAuditTaskController
- **Purpose**: Renders task operation results
- **Features**:
  - Batch process support
  - CSV download handling
  - Cache management
  - Operation result rendering

#### XrayAuditDisplayModeExampleController
- **Purpose**: Provides entity rendering examples
- **Features**:
  - Popup preview generation
  - View mode examples
  - Entity rendering isolation

## 6. Batch Processing System

### XrayAuditBatchHelper

Provides batch operation support for large-scale data processing:

```php
class XrayAuditBatchHelper {
  // Process single task operation for CSV
  public static function processSingleTaskOperationCsv($plugin_id, $operation_id, &$context);

  // Finish callback for bulk CSV download
  public static function downloadAllCsvFinished($success, $results, $operations);

  // Creates ZIP archive of all CSV files
  private static function createZipArchive($csv_files);
}
```

### Batch Processing Flow

```
1. User clicks "Download all reports as CSV"
2. HomeController::startDownloadAllCsvBatch()
3. Iterate through all task plugins with download=TRUE
4. Create batch operations array
5. batch_set() with operations
6. Process each operation:
   - Generate CSV data
   - Write to temporary file
   - Track in context['results']
7. Finish callback:
   - Create ZIP archive
   - Store path in session
   - Redirect to download
8. Download ZIP and cleanup
```

## 7. Caching Strategy

### Cache Architecture

```
Cache Layers:
1. Plugin Repository Cache (Coordination)
   └── Cache Manager (Implementation)
       └── cache.xray_audit (Storage)

Cache Key Pattern: {plugin_id}:{operation}
Cache Tags Pattern: [
  'config:entity_type_list',
  'node_list',
  'paragraph_list',
  'entity_view_display_list',
  ...
]
```

### Cache Invalidation

- **Tag-based**: Configuration changes invalidate related caches
- **Time-based**: Optional expiration for volatile data
- **Manual**: Cache flush form for administrators
- **Cascade**: Summary operations invalidate when dependencies change

## 8. Common Development Tasks

### Adding a New Report Group

1. Create group plugin in `src/Plugin/xray_audit/groups/`:
```php
/**
 * @XrayAuditGroupPlugin(
 *   id = "my_group",
 *   label = @Translation("My Group"),
 *   description = @Translation("Description"),
 *   sort = 100
 * )
 */
class MyGroupPlugin extends XrayAuditGroupPluginBase {
  // Implementation if needed
}
```

2. Routes are automatically generated
3. Group appears on main page

### Adding a New Report Task

1. Create task plugin in `src/Plugin/xray_audit/tasks/{GroupName}/`:
```php
/**
 * @XrayAuditTaskPlugin(
 *   id = "my_task",
 *   label = @Translation("My Task"),
 *   group = "my_group",
 *   operations = {
 *     "my_operation" = {
 *       "label" = "My Operation",
 *       "download" = TRUE
 *     }
 *   }
 * )
 */
class MyTaskPlugin extends XrayAuditTaskPluginBase {

  public function getHeaders(string $operation): array {
    return ['Column 1', 'Column 2'];
  }

  public function getRows(string $operation): array {
    // Return data rows
  }

  public function getDataOperationResult(string $operation): array {
    $cache_id = $this->pluginId . ':' . $operation;
    $cached = $this->pluginRepository->getCachedData($cache_id);

    if ($cached) {
      $this->handleCsv($operation, $cached['results_table']['rows']);
      return $cached;
    }

    $rows = $this->getRows($operation);
    $data = [
      'header_table' => ['header' => $this->getHeaders($operation)],
      'results_table' => ['rows' => $rows]
    ];

    $cache_tags = ['node_list']; // Add appropriate tags
    $this->pluginRepository->setCacheTagsInv($cache_id, $data, $cache_tags);
    $this->handleCsv($operation, $rows);

    return $data;
  }
}
```

2. Implement CSV export methods
3. Add tests in `tests/src/Kernel/Plugin/Tasks/{GroupName}/`
4. Routes and menu items are generated automatically

### Enabling CSV Export

1. Add `"download" = TRUE` to operation annotation
2. Implement `getHeaders()` and `getRows()` methods
3. Use `XrayAuditTaskCsvDownloadTrait` in plugin
4. Call `handleCsv()` in `getDataOperationResult()`

## 9. Key Design Patterns

### Plugin Pattern
- Extensible architecture through annotation-based discovery
- Automatic route generation from plugin definitions
- Standardized interfaces for consistent behavior

### Repository Pattern
- `PluginRepository` as central access point
- Encapsulates plugin management complexity
- Provides unified caching interface

### Strategy Pattern
- Different report strategies per task plugin
- Swappable data extraction methods
- Operation-specific processing

### Template Method Pattern
- `XrayAuditTaskPluginBase` defines algorithm structure
- Subclasses implement specific steps
- Common functionality in base class

### Trait Pattern
- `XrayAuditTaskCsvDownloadTrait` for CSV functionality
- `XrayAuditBuildTaskItemsInControllerTrait` for UI building
- Composition over inheritance

## 10. Quick Reference

### Service Injection in Plugins
```php
public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
  $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
  $instance->myService = $container->get('my.service');
  return $instance;
}
```

### Cache Usage
```php
// Set cache with tags
$this->pluginRepository->setCacheTagsInv($cache_id, $data, ['node_list']);

// Get cached data
$cached = $this->pluginRepository->getCachedData($cache_id);

// Invalidate cache
$this->pluginRepository->invalidateCache($cache_id);
```

### CSV Export
```php
// In getDataOperationResult()
$this->handleCsv($operation, $rows);

// Required methods
public function getHeaders(string $operation): array { }
public function getRows(string $operation): array { }
```

### Batch Processing
```php
// In annotation
batches = {
  "my_batch" = "myBatchMethod"
}

// Method implementation
public function myBatchMethod(&$context) {
  // Process batch
}
```

---

*Date: 31/10/2025.*