/**
 * @file
 * Xray Audit visibility toggler.
 */
((Drupal, once) => {
  'use strict';

  Drupal.behaviors.xrayAuditVisibilityToggler = {
    attach: (context) => {

      /**
       * Checks if a given table is empty.
       *
       * A table is considered empty if it has no rows or if it has only one row
       * and that row has the class 'empty'.
       *
       * @param {HTMLTableElement} table - The table element to check.
       * @returns {boolean} - Returns true if table is empty, false otherwise.
       */
      function tableIsEmpty(table) {
        const rows = table.querySelectorAll('tbody tr');
        const rowsLength = rows.length;
        let isEmpty = true;
        const emptyRowFound = rowsLength === 1 && rows[0].querySelector('td')?.classList.contains('empty');

        if (!emptyRowFound && rowsLength > 0) {
          isEmpty = false;
        }

        return isEmpty;
      }

      once('xrayAuditVisibilityToggler', 'input[xa-visibility-toggler]', context).forEach((input) => {
        input.addEventListener('change', () => {
            const wrappers = context.querySelectorAll('[xa-results-wrapper]');
            wrappers.forEach((wrapper) => {
              const tables = Array.from(wrapper.querySelectorAll('table'));
              const notEmptyTables = tables.filter((table) => {
                // Flagged table so we can know this table belongs to a
                // wrapper
                table.setAttribute('xa-wrapped-table', '');
                return !tableIsEmpty(table);
              });
              wrapper.style.display = notEmptyTables.length <= 0 && input.checked ? 'none' : '';
            });

            // Try to hide the rest of tables that are not inside a
            // xa-results-wrapper table.
            const unwrappedTables = context.querySelectorAll('table:not([xa-wrapped-table])');
            unwrappedTables.forEach((table) => {
              table.style.display = tableIsEmpty(table) && input.checked ? 'none' : '';
            });
        });
      });
    }
  };

})(Drupal, once);
