<?php

namespace Drupal\xray_audit\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\xray_audit\Services\PluginRepositoryInterface;
use Drupal\xray_audit\Plugin\XrayAuditGroupPluginManager;

/**
 * Controller to handle actions related to Xray audit groups.
 */
class XrayAuditGroupsController extends ControllerBase {

  use XrayAuditBuildTaskItemsInControllerTrait;

  /**
   * Plugin repository.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Xray audit group plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditGroupPluginManager
   */
  protected $xrayAuditGroupPluginManager;

  /**
   * Constructs the controller.
   *
   * @param \Drupal\xray_audit\Services\PluginRepositoryInterface $plugin_repository
   *   The plugin repository service.
   * @param \Drupal\xray_audit\Plugin\XrayAuditGroupPluginManager $xrayAuditGroupPluginManager
   *   The Xray audit group plugin manager service.
   */
  public function __construct(PluginRepositoryInterface $plugin_repository, XrayAuditGroupPluginManager $xrayAuditGroupPluginManager) {
    $this->pluginRepository = $plugin_repository;
    $this->xrayAuditGroupPluginManager = $xrayAuditGroupPluginManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('xray_audit.plugin_repository'),
      $container->get('plugin_manager.xray_audit_group')
    );
  }

  /**
   * Returns the title of the group by its ID.
   *
   * @param string $group_id
   *   The group ID.
   *
   * @return string
   *   The translated group title.
   */
  public function getGroupTitle(string $group_id): string {
    $group_id = str_replace('-', '_', $group_id);
    $plugin_definitions = $this->xrayAuditGroupPluginManager->getDefinitions();

    foreach ($plugin_definitions as $plugin_definition) {
      if ($plugin_definition['id'] === $group_id) {
        $label = $plugin_definition['label'];
        return $this->t('@label', ['@label' => $label]);
      }
    }

    // Return a default message if the group ID is not found.
    return $this->t('@group_id group not found.', ['@group_id' => $group_id]);
  }

  /**
   * Builds the page content for the specified group ID.
   *
   * @param string $group_id
   *   The group ID.
   *
   * @return array
   *   A render array containing the tasks for the group.
   */
  public function build(string $group_id): array {
    $build = [];
    $group_id = str_replace('-', '_', $group_id);

    $build[] = [
      '#theme' => 'admin_block_content',
      '#content' => $this->buildTaskItems($group_id),
    ];

    return $build;
  }

}
