<?php

namespace Drupal\xray_audit\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\xray_audit\Form\FlushCacheForm;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Url;
use Drupal\xray_audit\Utils\XrayAuditBatchHelper;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

/**
 * Returns responses for the list of groups.
 */
final class XrayAuditHomeController extends ControllerBase {

  use XrayAuditBuildTaskItemsInControllerTrait;

  /**
   * Plugin repository.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The URL generator.
   *
   * @var \Drupal\Core\Routing\UrlGeneratorInterface
   */
  protected $urlGenerator;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->pluginRepository = $container->get('xray_audit.plugin_repository');
    $instance->renderer = $container->get('renderer');
    $instance->urlGenerator = $container->get('url_generator');
    $instance->fileSystem = $container->get('file_system');
    $instance->requestStack = $container->get('request_stack');
    return $instance;
  }

  /**
   * Builds the home response.
   *
   * @return array
   *   Render array.
   */
  public function build(): array {
    $build = [];
    $groupDefinitions = $this->pluginRepository->getGroupPluginDefinitions();
    if (empty($groupDefinitions)) {
      return [];
    }
    usort($groupDefinitions, function ($a, $b) {
      return ($a['sort'] ?? 0) <=> ($b['sort'] ?? 0);
    });

    foreach ($groupDefinitions as $group_definition) {
      // The different tasks.
      $build[] = [
        '#theme' => 'admin_block',
        '#block' => [
          'title' => (string) $group_definition['label'],
          'content' => [
            '#theme' => 'admin_block_content',
            '#content' => $this->buildTaskItems($group_definition['id']),
          ],
          'description' => (string) $group_definition['description'],
        ],
      ];
    }

    $build['form'] = $this->formBuilder()->getForm(FlushCacheForm::class);

    // Add the "Download all reports as CSV" button.
    $build['download_all_csv'] = [
      '#type' => 'link',
      '#title' => $this->t('Download all reports as CSV'),
      '#url' => Url::fromRoute('xray_audit.download_all_csv_batch'),
      '#attributes' => [
        'class' => ['button', 'button--primary', 'xray-audit-download-all-button'],
      ],
      // Place it before other elements.
      '#weight' => -10,
    ];
    // Wrap existing build items if needed or adjust structure.
    // For simplicity, adding the button at the top level of $build.
    // It might be better to group $build items if $build was a more complex render array.
    return $build;
  }

  /**
   * Starts the batch process for downloading all CSV reports.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the batch execution page.
   */
  public function startDownloadAllCsvBatch() {
    $task_definitions = $this->pluginRepository->getTaskPluginDefinitions();
    $operations = [];

    foreach ($task_definitions as $plugin_id => $definition) {
      if (!empty($definition['operations']) && is_array($definition['operations'])) {
        foreach ($definition['operations'] as $operation_id => $operation_details) {
          // Check if the operation is marked for download.
          if (!empty($operation_details['download']) && $operation_details['download'] === TRUE) {
            $operations[] = [
              // The callable for the batch operation.
              [XrayAuditBatchHelper::class, 'processSingleTaskOperationCsv'],
              // Arguments for the callable: plugin_id and operation_id.
              [$plugin_id, $operation_id],
            ];
          }
        }
      }
    }

    if (empty($operations)) {
      $this->messenger()->addWarning($this->t('No X-Ray audit task operations configured for CSV download were found.'));
      return new RedirectResponse($this->urlGenerator->generate('xray_audit.home'));
    }

    // Construct the batch definition array directly.
    $batch = [
      'title' => $this->t('Generating X-Ray Audit Report CSVs...'),
      'operations' => [],
      'finished' => [XrayAuditBatchHelper::class, 'downloadAllCsvFinished'],
      'init_message' => $this->t('Initializing CSV generation process...'),
      'progress_message' => $this->t('Processed @current out of @total operations.'),
      'error_message' => $this->t('An error occurred during CSV generation.'),
    ];

    foreach ($operations as $operation_item) {
      // Each operation is an array: [callable, [args]].
      $batch['operations'][] = $operation_item;
    }

    batch_set($batch);
    // Redirect to the standard batch processing page.
    // This will pick up the batch definition we just set.
    return batch_process(Url::fromRoute('xray_audit.home'));
  }

  /**
   * Handles the download of the generated ZIP file after batch processing.
   *
   * @return \Symfony\Component\HttpFoundation\BinaryFileResponse|\Symfony\Component\HttpFoundation\RedirectResponse
   *   The file download response or a redirect if an error occurs.
   */
  public function downloadGeneratedZipFile() {
    $session = $this->requestStack->getCurrentRequest()->getSession();
    $zip_filepath = $session->get('xray_audit_batch_zip_filepath');

    if (empty($zip_filepath)) {
      $this->messenger()->addError($this->t('No ZIP file path found in session. Please try generating the reports again.'));
      return new RedirectResponse($this->urlGenerator->generate('xray_audit.home'));
    }

    // Clear the session variable immediately.
    $session->remove('xray_audit_batch_zip_filepath');

    $real_zip_filepath = $this->fileSystem->realpath($zip_filepath);

    if (!$real_zip_filepath || !file_exists($real_zip_filepath)) {
      $this->messenger()->addError($this->t('Generated ZIP file not found at @path. Please try generating the reports again.', ['@path' => $zip_filepath]));
      return new RedirectResponse($this->urlGenerator->generate('xray_audit.home'));
    }

    $response = new BinaryFileResponse($real_zip_filepath);
    $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT, basename($real_zip_filepath));
    $response->headers->set('Content-Type', 'application/zip');

    // Delete the file after sending the response.
    $response->deleteFileAfterSend();

    return $response;
  }

}
