<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentDisplay;

use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of entities display.
 *
 * @XrayAuditTaskPlugin (
 *   id = "entity_displays",
 *   label = @Translation("Entities Displays"),
 *   description = @Translation("Display modes of entities."),
 *   group = "content_display",
 *   sort = 1,
 *   local_task = 1,
 *   operations = {
 *      "node_display_mode" = {
 *          "label" = "Node display configurations",
 *          "description" = "Generates a report on the display modes configured for node entities.",
 *          "dependencies" = {"node"},
 *          "download" = TRUE
 *       },
 *     "paragraph_display_mode" = {
 *          "label" = "Paragraphs display configurations",
 *          "description" = "Generates a report on the display modes configured for paragraph entities.",
 *          "dependencies" = {"paragraphs"},
 *          "download" = TRUE
 *      },
 *     "media_display_mode" = {
 *          "label" = "Media display configurations",
 *          "description" = "Generates a report on the display modes configured for media entities.",
 *          "dependencies" = {"media"},
 *          "download" = TRUE
 *     },
 *     "taxonomy_display_mode" = {
 *          "label" = "Taxonomy display configurations",
 *          "description" = "Generates a report on the display modes configured for taxonomy entities.",
 *          "dependencies" = {"taxonomy"},
 *          "download" = TRUE
 *     },
 *     "block_display_mode" = {
 *          "label" = "Block display configurations",
 *          "description" = "Generates a report on the display modes configured for block entities.",
 *          "dependencies" = {"block"},
 *          "download" = TRUE
 *     }
 *   },
 *   dependencies = {"field"}
 * )
 */
class XrayAuditEntityDisplaysPlugin extends XrayAuditTaskPluginBase {

  /**
   * Service "entity_display_architecture".
   *
   * @var \Drupal\xray_audit\Services\EntityDisplayArchitectureInterface
   */
  protected $entityDisplayArchitecture;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityDisplayArchitecture = $container->get('xray_audit.entity_display_architecture');
    return $instance;
  }

  /**
   * Helper to get entity type and bundle entity type from operation.
   */
  private function getEntityTypeInfoFromOperation(string $operation): ?array {
    switch ($operation) {
      case 'node_display_mode':
        return ['bundle_entity_type' => 'node_type', 'entity_type_id' => 'node'];

      case 'paragraph_display_mode':
        return ['bundle_entity_type' => 'paragraphs_type', 'entity_type_id' => 'paragraph'];

      case 'media_display_mode':
        return ['bundle_entity_type' => 'media_type', 'entity_type_id' => 'media'];

      case 'taxonomy_display_mode':
        return ['bundle_entity_type' => 'taxonomy_vocabulary', 'entity_type_id' => 'taxonomy_term'];

      case 'block_display_mode':
        return ['bundle_entity_type' => 'block_content_type', 'entity_type_id' => 'block_content'];
    }
    return NULL;
  }

  /**
   * Fetches plain text data from the entityDisplayArchitecture service.
   *
   * This calls getPlainDataStructure() on the service, which returns rows
   * with plain text cell values.
   *
   * @param string $operation
   *   The operation ID.
   *
   * @return array
   *   An array containing 'headers', 'rows' (with plain text cells),
   *   'computed_text' (raw string), and 'cache_tags'.
   */
  private function getDisplayData(string $operation): array {
    $entity_type_info = $this->getEntityTypeInfoFromOperation($operation);
    if (!$entity_type_info) {
      return [
        'headers' => [],
        'rows' => [],
        'computed_text' => NULL,
        'cache_tags' => [],
      ];
    }

    // Call the new service method that returns plain text data.
    $service_data = $this->entityDisplayArchitecture->getPlainDataStructure(
      $entity_type_info['bundle_entity_type'],
      $entity_type_info['entity_type_id']
    );

    $headers = $service_data['main_table']['headers'] ?? [];
    // Rows from getPlainDataStructure are arrays of plain strings/values.
    $plain_rows = $service_data['main_table']['rows'] ?? [];
    $computed_text = $service_data['main_table']['computed'] ?? NULL;

    // Define cache tags.
    $cache_tags = ['config:core.entity_view_display.' . $entity_type_info['entity_type_id'] . '.*'];
    if ($this->entityTypeManager->hasDefinition($entity_type_info['bundle_entity_type'])) {
      $cache_tags = array_merge($this->entityTypeManager->getDefinition($entity_type_info['bundle_entity_type'])->getListCacheTags(), $cache_tags);
    }

    return [
      'headers' => $headers,
    // These are now rows with plain text cells.
      'rows' => $plain_rows,
      'computed_text' => $computed_text,
      'cache_tags' => $cache_tags,
    ];
  }

  /**
   * Formats the plain display data for table rendering.
   *
   * Takes the plain text data from getDisplayData() and prepares it for the
   * table theme by structuring rows and formatting the computed fields text.
   *
   * @param string $operation
   *   The operation ID.
   *
   * @return array
   *   An array containing 'headers', 'table_ready_rows' (with plain text cells),
   *   'cache_tags', and 'computed_fields_markup'.
   */
  private function getFormattedDisplayData(string $operation): array {
    $plain_data = $this->getDisplayData($operation);

    $headers = $plain_data['headers'];
    $table_ready_rows = [];

    if (!empty($headers) && !empty($plain_data['rows'])) {
      // The rows from getDisplayData are now arrays of plain text/values.
      // Structure them for the table theme.
      foreach ($plain_data['rows'] as $row_key => $row_content) {
        // $row_content is an associative array where keys are header keys
        // and values are plain strings/values.
        $table_ready_rows[$row_key]['data'] = $row_content;
        $table_ready_rows[$row_key]['class'] = ['xray-audit__row'];
      }
    }

    $computed_fields_markup = isset($plain_data['computed_text'])
      ? $this->t("<b>Computed fields:</b> %computed", ['%computed' => $plain_data['computed_text']])
      : NULL;

    return [
      'headers' => $headers,
      'cache_tags' => $plain_data['cache_tags'],
      'table_ready_rows' => array_values($table_ready_rows),
      'computed_fields_markup' => $computed_fields_markup,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    // Get plain headers from getDisplayData for CSV.
    $data = $this->getDisplayData($operation);
    return $data['headers'];
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    // Get plain rows from getDisplayData for CSV.
    $data = $this->getDisplayData($operation);
    // Return plain rows, not table-ready rows.
    return $data['rows'];
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvData(string $operation, array $data): array {
    // $data argument is the result of getDataOperationResult.
    // We need the plain rows for CSV.
    $plain_data = $this->getDisplayData($operation);
    return $plain_data['rows'] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    $entity_type_info = $this->getEntityTypeInfoFromOperation($operation);
    if (!$entity_type_info) {
      return [];
    }

    $cid = $this->getPluginId() . ':' . $operation;
    $cached_data = $this->pluginRepository->getCachedData($cid);
    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    // Call getData() which returns the full structure with summaries.
    $data = $this->entityDisplayArchitecture->getData(
      $entity_type_info['bundle_entity_type'],
      $entity_type_info['entity_type_id']
    );

    // Define cache tags.
    $cache_tags = ['config:core.entity_view_display.' . $entity_type_info['entity_type_id'] . '.*'];
    if ($this->entityTypeManager->hasDefinition($entity_type_info['bundle_entity_type'])) {
      $cache_tags = array_merge($this->entityTypeManager->getDefinition($entity_type_info['bundle_entity_type'])->getListCacheTags(), $cache_tags);
    }

    $this->pluginRepository->setCacheTagsInv($cid, $data, $cache_tags);
    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function buildDataRenderArray(array $data, string $operation = ''): array {
    $build = [];

    // Get operation description.
    $description = '';
    $operation_data = $this->getOperations()[$operation] ?? NULL;
    if ($operation_data) {
      $description = $operation_data['description'] ?? '';
    }

    // Summary table introduction.
    $build['summary_table_introduction'] = [
      '#markup' => '<h2>' . $this->t('Display Mode Summary') . '</h2>',
      '#weight' => 0,
    ];

    // Build summary_bundle_display table.
    $summary_bundle_display = $data['summary_bundle_display'] ?? [];
    $bundle_display_rows = [];
    foreach ($summary_bundle_display as $bundle_data) {
      $bundle_display_rows[] = [
        $bundle_data['label'],
        $bundle_data['displays'],
        $bundle_data['count'],
      ];
    }

    $build['summary_bundle_display'] = [
      '#theme' => 'table',
      '#header' => [
        $this->t('Bundle'),
        $this->t('Display Modes'),
        $this->t('Count'),
      ],
      '#rows' => $bundle_display_rows,
      '#weight' => 1,
      '#attributes' => [
        'class' => ['xray-audit__table'],
      ],
    ];

    // Build summary_display_bundles table.
    $summary_display_bundles = $data['summary_display_bundles'] ?? [];
    $display_bundles_rows = [];
    foreach ($summary_display_bundles as $display_data) {
      $display_bundles_rows[] = [
        $display_data['display'],
        $display_data['bundles'],
        $display_data['count'],
      ];
    }

    $build['summary_display_bundles'] = [
      '#theme' => 'table',
      '#header' => [
        $this->t('Display Mode'),
        $this->t('Bundles'),
        $this->t('Count'),
      ],
      '#rows' => $display_bundles_rows,
      '#weight' => 2,
      '#attributes' => [
        'class' => ['xray-audit__table'],
      ],
    ];

    // Main table introduction.
    $build['main_table_introduction'] = [
      '#markup' => '<h2>' . $this->t('Field Display Configuration') . '</h2>' .
      ($description ? '<p>' . $description . '</p>' : ''),
      '#weight' => 10,
    ];

    // Build main table.
    $main_table_data = $data['main_table'] ?? [];
    $headers = $main_table_data['headers'] ?? [];
    $rows = $main_table_data['rows'] ?? [];

    // Convert headers to array values for render array.
    // Headers are TranslatableMarkup objects, convert them to strings.
    $header_values = [];
    foreach ($headers as $header) {
      $header_values[] = (string) $header;
    }

    // Format rows for table rendering.
    $formatted_rows = [];
    foreach ($rows as $row) {
      $formatted_rows[] = [
        'data' => array_values($row),
        'class' => ['xray-audit__row'],
      ];
    }

    $build['main_table'] = [
      '#theme' => 'table',
      '#header' => $header_values,
      '#rows' => $formatted_rows,
      '#weight' => 20,
      '#attributes' => [
        'class' => ['xray-audit__table', 'xray-audit__sticky-header'],
      ],
      '#attached' => [
        'library' => ['xray_audit/xray_audit'],
      ],
    ];

    // Add computed fields if present.
    $computed = $main_table_data['computed'] ?? '';
    if (!empty($computed)) {
      $build['computed'] = [
        '#markup' => $this->t('<b>Computed fields:</b> <em class="placeholder">@computed</em>', ['@computed' => $computed]),
        '#prefix' => '<p></p><p>',
        '#suffix' => '</p>',
        '#weight' => 30,
      ];
    }

    // Process CSV download - button will be added after the main table.
    $this->processCsvDownload($operation, $data, $build);

    // Position the download button after the main table and computed fields.
    if (isset($build['download_button'])) {
      $build['download_button']['#weight'] = 40;
      $build['download_button']['#prefix'] = '<div class="xray-audit__download-section">';
      $build['download_button']['#suffix'] = '</div>';
    }

    return $build;
  }

}
