<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentDisplay;

use Drupal\Component\Utility\Html;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;
use Drupal\xray_audit\Utils\XrayAuditTableFilter;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of external resources.
 *
 * @XrayAuditTaskPlugin (
 *   id = "external_resources",
 *   label = @Translation("External Resources"),
 *   description = @Translation("External resources referenced in content."),
 *   group = "content_display",
 *   sort = 2,
 *   operations = {
 *     "external_resources" = {
 *       "label" = "External Resources",
 *       "description" = "Lists external resources referenced in content"
 *     }
 *   }
 * )
 */
class XrayAuditExternalResourcesPlugin extends XrayAuditTaskPluginBase {

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|null
   */
  protected $moduleHandler = NULL;

  /**
   * The URL resolver service.
   *
   * @var \Drupal\media\OEmbed\UrlResolverInterface|null
   */
  protected $urlResolver = NULL;

  /**
   * The link generator service.
   *
   * @var \Drupal\Core\Utility\LinkGeneratorInterface
   */
  protected $linkGenerator;

  /**
   * The oEmbed resource fetcher.
   *
   * @var \Drupal\media\OEmbed\ResourceFetcherInterface
   */
  protected $resourceFetcher;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->entityFieldManager = $container->get('entity_field.manager');
    $instance->linkGenerator = $container->get('link_generator');

    if ($instance->moduleHandler->moduleExists('media')) {
      $instance->urlResolver = $container->get('media.oembed.url_resolver');
      $instance->resourceFetcher = $container->get('media.oembed.resource_fetcher');
    }

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    if (!$this->moduleHandler->moduleExists('media')) {
      return [];
    }

    $oembed_fields = $this->getOembedFields();
    $results = $this->processMediaEntities($oembed_fields);

    return $results;
  }

  /**
   * Gets all OEmbed fields from media bundles.
   */
  private function getOembedFields(): array {
    $oembed_fields = [];
    $bundles = $this->entityTypeManager->getStorage('media_type')->loadMultiple();
    $display_storage = $this->entityTypeManager->getStorage('entity_view_display');

    foreach ($bundles as $bundle_id => $bundle) {
      $displays = $display_storage->loadByProperties([
        'targetEntityType' => 'media',
        'bundle' => $bundle_id,
      ]);

      foreach ($displays as $display) {
        foreach ($display->getComponents() as $field_name => $component) {
          if (isset($component['type']) && str_contains($component['type'], 'oembed')) {
            $oembed_fields[$bundle_id][$field_name] = $field_name;
          }
        }
      }
    }

    return $oembed_fields;
  }

  /**
   * Process media entities and collect results.
   */
  private function processMediaEntities(array $oembed_fields): array {
    $results = [];
    $media_storage = $this->entityTypeManager->getStorage('media');

    foreach ($oembed_fields as $bundle_id => $field_names) {
      $media_ids = $media_storage->getQuery()
        ->condition('bundle', $bundle_id)
        ->accessCheck(FALSE)
        ->execute();

      if (empty($media_ids)) {
        continue;
      }

      $media_entities = $media_storage->loadMultiple($media_ids);
      foreach ($media_entities as $media_id => $media) {
        $this->processMediaEntity($media, $media_id, $bundle_id, $field_names, $results);
      }
    }

    return $results;
  }

  /**
   * Process a single media entity.
   */
  private function processMediaEntity($media, $media_id, $bundle_id, array $field_names, array &$results): void {
    foreach ($field_names as $field_name) {
      if (!$media->hasField($field_name) || $media->get($field_name)->isEmpty()) {
        continue;
      }

      foreach ($media->get($field_name) as $field_item) {
        $value = $field_item->getValue();
        if (!isset($value['value'])) {
          continue;
        }

        $result = $this->createBaseResult($media_id, $bundle_id, $value['value'], $media);
        $this->processInsightData($result, $value['value']);
        $results[] = (object) $result;
      }
    }
  }

  /**
   * Creates base result array for a media item.
   */
  private function createBaseResult($media_id, $bundle_id, $resource_url, $media): array {
    return [
      'entity_id' => $media_id,
      'entity_type' => 'media',
      'bundle' => $bundle_id,
      'resource' => $resource_url,
      'edit_url' => $media->toUrl('edit-form'),
      'needs_review' => FALSE,
      'insight_message' => '',
    ];
  }

  /**
   * Adds insight data to the result.
   */
  private function processInsightData(array &$result, string $resource_url): void {

    try {
      $normalized_url = $this->urlResolver->getResourceUrl($resource_url);
      $this->resourceFetcher->fetchResource($normalized_url);
    }
    catch (\Exception $e) {
      $result['needs_review'] = TRUE;
      $result['insight_message'] = $e->getMessage();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildDataRenderArray(array $data, string $operation = '') {
    $build = [];

    $rows = [];

    $headers = [
      $this->t('Entity ID'),
      $this->t('Entity Type'),
      $this->t('Bundle'),
      $this->t('Resource URL'),
      $this->t('Edit'),
    ];

    foreach ($data as $row) {
      $highlight = FALSE;
      $row_data = [
        $row->entity_id ?? '',
        $row->entity_type ?? '',
        $row->bundle ?? '',
        $row->resource ?? '',
        $this->linkGenerator->generate($this->t('Edit'), $row->edit_url) ?? '',
      ];

      // Add insight columns if xray_audit_insight module is enabled.
      if ($this->moduleHandler->moduleExists('xray_audit_insight')) {
        $row_data[] = isset($row->needs_review) && $row->needs_review ? $this->t('Yes') : $this->t('No');
        $row_data[] = $row->insight_message ?? '';
        $highlight = isset($row->needs_review) && $row->needs_review;
      }

      $rows[] = ['data' => $row_data, 'class' => $highlight ? ['xray-audit--highlighted'] : []];
    }

    // Add insight headers if xray_audit_insight module is enabled.
    if ($this->moduleHandler->moduleExists('xray_audit_insight')) {
      $headers[] = $this->t('Needs Review');
      $headers[] = $this->t('Message');
    }
    $unique_id = Html::getUniqueId('xray-audit-external-resources');
    $build['table'] = [
      '#theme' => 'table',
      '#header' => $headers,
      '#sticky' => TRUE,
      '#weight' => 10,
      '#rows' => $rows,
      '#empty' => $this->t('No external resources found'),
      '#attributes' => [
        'class' => ['xray-audit__table'],
        'id' => $unique_id,
      ],
      '#attached' => [
        'library' => [
          'xray_audit/xray_audit',
        ],
      ],
    ];

    $columns_indexes = [0, 1, 2, 3, 4, 5, 6];
    $build['filter'] = XrayAuditTableFilter::generateRenderableFilterInput($unique_id, $columns_indexes, NULL, $headers);
    $build['filter']['#weight'] = 6;

    return $build;
  }

}
