<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric;

use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "queries_data_media",
 *   label = @Translation("Media reports"),
 *   description = @Translation("Reports on the Medias of this site"),
 *   group = "content_metric",
 *   sort = 3,
 *   operations = {
 *      "media_count_type" = {
 *          "label" = "Medias grouped by type",
 *          "description" = "Number of Medias grouped by type.",
 *           "download" = TRUE
 *       },
 *    },
 *   dependencies = {"media"}
 * )
 */
class XrayAuditQueryTaskMediaPlugin extends XrayAuditTaskPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    switch ($operation) {
      case 'media_count_type':
        return [
          $this->t('ID'),
          $this->t('Label'),
          $this->t('Count'),
        ];
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    switch ($operation) {
      case 'media_count_type':
        return $this->getMediaCountTypeRows();
    }
    return [];
  }

  /**
   * Gets the rows for the 'media_count_type' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getMediaCountTypeRows(): array {
    $resultTable = [];
    $query = $this->entityTypeManager->getStorage('media')->getAggregateQuery();
    $aliasCount = 'count';
    $result = $query->accessCheck(FALSE)
      ->currentRevision()
      ->groupBy('bundle')
      ->aggregate('mid', 'COUNT', NULL, $aliasCount)
      ->sort('bundle')
      ->execute();
    $result_processed = [];
    /** @var array<string> $row */
    foreach ($result as $row) {
      $result_processed[$row['bundle']] = $row['count'];
    }
    $medias = $this->entityTypeManager->getStorage('media_type')->loadMultiple();

    $total = 0;
    foreach ($medias as $key => $media) {
      $count = $result_processed[$key] ?? 0;
      $resultTable[] = [
        'id' => $key,
        'label' => $media->label(),
        'count' => $count,
      ];
      $total += $count;
    }

    // Sort by count (descending).
    usort($resultTable, function ($a, $b) {
      return $b['count'] <=> $a['count'];
    });

    // Convert to simple arrays for display.
    $sorted_result = [];
    foreach ($resultTable as $row) {
      $sorted_result[] = [$row['id'], $row['label'], $row['count']];
    }

    // Add total row.
    $sorted_result[] = [
      $this->t('TOTAL'),
      '',
      $total,
    ];

    return $sorted_result;
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    $cid = $this->getPluginId() . ':' . $operation;

    $cached_data = $this->pluginRepository->getCachedData($cid);
    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    $data = [
      'header_table' => $this->getHeaders($operation),
      'results_table' => $this->getRows($operation),
    ];

    $this->pluginRepository->setCacheTagsInv($cid, $data, ['media_list']);
    return $data;
  }

}
