<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\xray_audit\Form\SettingsForm;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "queries_data_nodes",
 *   label = @Translation("Node reports"),
 *   description = @Translation("Metrics about node entities."),
 *   group = "content_metric",
 *   sort = 1,
 *   local_task = 1,
 *   operations = {
 *      "node_count_type" = {
 *          "label" = "Nodes grouped by type",
 *          "description" = "Number of Nodes grouped by type.",
 *           "download" = TRUE
 *       },
 *       "node_count_type_langcode" = {
 *          "label" = "Nodes grouped by type and language",
 *          "description" = "Number of Nodes grouped by type and language.",
 *           "download" = TRUE
 *       },
 *      "node_revisions_count" = {
 *          "label" = "Nodes grouped by revisions",
 *          "description" = "Nodes with the highest number of revisions.",
 *           "download" = TRUE
 *       },
 *       "node_revisions_count_language" = {
 *          "label" = "Nodes grouped by revisions and language",
 *          "description" = "Nodes with the highest number of revisions grouped by language.",
 *           "download" = TRUE
 *       }
 *    },
 *   dependencies = {"node"}
 * )
 */
final class XrayAuditQueryTaskNodePlugin extends XrayAuditTaskPluginBase {

  const XRA_NODE_TEMPORARY_TABLE_NAME = 'xra_nodes_hierarchy_tmp';

  /**
   * Service "xray_audit.entity_use_node".
   *
   * @var \Drupal\xray_audit\Services\EntityUseInterface
   */
  protected $serviceEntityUsenode;

  /**
   * Service "request_stack".
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Service "state".
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Service "config.factory".
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Service "module_handler".
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Configuration settings for Xray Audit.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $xrayAuditConfig;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->serviceEntityUsenode = $container->get('xray_audit.entity_use_node');
    $instance->requestStack = $container->get('request_stack');
    $instance->state = $container->get('state');
    $configFactory = $container->get('config.factory');
    $instance->xrayAuditConfig = $configFactory->get(SettingsForm::SETTINGS);
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    switch ($operation) {
      case 'node_count_type':
        return [
          $this->t('ID'),
          $this->t('Label'),
          $this->t('Total'),
          $this->t('Published'),
          $this->t('Unpublished'),
        ];

      case 'node_count_type_langcode':
        return [
          $this->t('ID'),
          $this->t('Label'),
          $this->t('Language'),
          $this->t('Total'),
          $this->t('Published'),
          $this->t('Unpublished'),
        ];

      case 'node_revisions_count':
        return [
          $this->t('ID'),
          $this->t('Label'),
          $this->t('Revisions'),
        ];

      case 'node_revisions_count_language':
        return [
          $this->t('ID'),
          $this->t('Label'),
          $this->t('Language'),
          $this->t('Revisions'),
        ];
    }
    return [];
  }

  /**
   * Gets the rows for the 'node_count_type' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getNodeCountTypeRows(): array {
    $resultTable = [];
    $alias_count = 'count';
    // Get label of content types.
    $label_content_types = [];
    $types = $this->entityTypeManager->getStorage("node_type")->loadMultiple();
    foreach ($types as $key => $type) {
      $label_content_types[$key] = $type->label();
      // Initialize each type with default zero counts.
      $resultTable[$key] = [
        'id' => $key,
        'label' => $label_content_types[$key] ?? $key,
        'total' => 0,
        'published' => 0,
        'unpublished' => 0,
      ];
    }

    $query = $this->entityTypeManager
      ->getStorage('node')
      ->getAggregateQuery()
      ->accessCheck(FALSE)
      ->aggregate('nid', 'count', NULL, $alias_count)
      ->groupBy('type')
      ->groupBy('status');

    $query_results = $query->execute();

    $total = 0;
    foreach ($query_results as $result) {
      $bundle = $result['type'] ?? '';
      $status = $result['status'] ?? 0;
      $status = (int) $status;
      $count = $result[$alias_count] ?? 0;

      $resultTable[$bundle]['total'] += $count;

      if ($status === 1) {
        $resultTable[$bundle]['published'] = $count;
      }
      elseif ($status === 0) {
        $resultTable[$bundle]['unpublished'] = $count;
      }

      $total += $count;
    }

    $final_total_published = 0;
    $final_total_unpublished = 0;
    foreach ($resultTable as $key => $data_row) {
      $final_total_published += $data_row['published'];
      $final_total_unpublished += $data_row['unpublished'];
    }

    // Sort by total count (descending).
    uasort($resultTable, function ($a, $b) {
      return $b['total'] <=> $a['total'];
    });

    // Add total row at the end.
    $resultTable['total'] = [
      'id' => $this->t('TOTAL'),
      'label' => '---',
      'total' => $total,
      'published' => $final_total_published,
      'unpublished' => $final_total_unpublished,
    ];

    return $resultTable;
  }

  /**
   * Gets the rows for the 'node_count_type_langcode' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getNodeCountTypeLangcodeRows(): array {
    $result_table = [];
    $alias_count = 'count';
    // Get label of content types.
    $label_content_types = [];
    $types = $this->entityTypeManager->getStorage("node_type")->loadMultiple();
    foreach ($types as $key => $type) {
      $label_content_types[$key] = $type->label();
    }

    $query = $this->entityTypeManager
      ->getStorage("node")
      ->getAggregateQuery()
      ->accessCheck(FALSE)
      ->aggregate('nid', 'COUNT', NULL, $alias_count)
      ->groupBy('langcode')
      ->groupBy('type')
      ->groupBy('status');

    $query_results = $query->execute();
    foreach ($query_results as $result) {
      $bundle = $result['type'] ?? '';
      $status = $result['status'] ?? 0;
      $status = (int) $status;
      $count = $result[$alias_count] ?? 0;
      $langcode = $result['langcode'] ?? '';
      $key = $bundle . '-' . $langcode;

      if (!isset($result_table[$key])) {
        $result_table[$key] = [
          'id' => $bundle,
          'label' => $label_content_types[$bundle] ?? $bundle,
          'language' => $langcode,
          'total' => 0,
          'published' => 0,
          'unpublished' => 0,
        ];
      }

      $result_table[$key]['total'] += $count;

      if ($status === 1) {
        $result_table[$key]['published'] = $count;
      }
      elseif ($status === 0) {
        $result_table[$key]['unpublished'] = $count;
      }
    }

    // Sort by total count (descending).
    uasort($result_table, function ($a, $b) {
      return $b['total'] <=> $a['total'];
    });

    return $result_table;
  }

  /**
   * Gets the rows for the 'node_revisions_count' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getNodeRevisionsCountRows(): array {
    $resultTable = [];
    $alias_count = 'count';
    $query = $this->database->select('node_revision', 'nr');
    $query->addField('nr', 'nid');
    $query->addExpression('COUNT(nr.vid)', $alias_count);
    $query->groupBy('nr.nid');
    $query->orderBy($alias_count, 'DESC');
    $query->range(0, 20);

    $result = $query->execute()->fetchAll();
    $max_revision_count = $this->xrayAuditConfig->get('revisions_thresholds.node');

    foreach ($result as $row) {
      $nid = $row->nid;
      $revisions = $row->{$alias_count};

      // Load the node to get its title.
      $node = $this->entityTypeManager->getStorage('node')->load($nid);
      $label = $node ? $node->getTitle() : $this->t('(No title)');

      $resultTable[$nid] = [
        'id' => $nid,
        'label' => $label,
        'revisions' => [
          'data' => $revisions,
        ],
      ];

      if (!empty($max_revision_count) && $revisions > $max_revision_count) {
        $resultTable[$nid]['revisions']['class'] = 'xray-audit-color-error';
      }
    }
    if (empty($resultTable)) {
      $resultTable[] = [
        'id' => $this->t('No nodes found'),
        'label' => '-',
        'revisions' => 0,
      ];
    }
    return $resultTable;
  }

  /**
   * Gets the rows for the 'node_revisions_count_language' operation.
   *
   * Efficiently retrieves nodes with the highest revision counts grouped by
   * language. Uses a single optimized database query with proper indexing.
   *
   * @return array
   *   The table rows, each containing node ID, title, language, and revision
   *   count, with error highlighting for nodes exceeding the threshold.
   */
  private function getNodeRevisionsCountLanguageRows(): array {
    $resultTable = [];
    $alias_count = 'count';
    $query = $this->database->select('node_revision', 'nr');
    $query->addField('nr', 'nid');
    $query->addField('nr', 'langcode');
    $query->addExpression('COUNT(nr.vid)', $alias_count);
    $query->groupBy('nr.nid');
    $query->groupBy('nr.langcode');
    $query->orderBy($alias_count, 'DESC');
    $query->range(0, 20);

    $result = $query->execute()->fetchAll();
    $max_revision_count = $this->xrayAuditConfig->get('revisions_thresholds.node');

    foreach ($result as $row) {
      $nid = $row->nid;
      $langcode = $row->langcode;
      $revisions = (int) $row->{$alias_count};

      // Load the node translation to get its title.
      $node = $this->entityTypeManager->getStorage('node')->load($nid);
      if ($node && $node->hasTranslation($langcode)) {
        $translation = $node->getTranslation($langcode);
        $label = $translation->getTitle();
      }
      else {
        $label = $this->t('(No title)');
      }

      // Create unique key for each node-language combination.
      $key = $nid . '-' . $langcode;

      $resultTable[$key] = [
        'id' => $nid,
        'label' => $label,
        'language' => $langcode,
        'revisions' => [
          'data' => $revisions,
        ],
      ];

      // Apply error class if revision count exceeds threshold.
      if (!empty($max_revision_count) && $revisions > $max_revision_count) {
        $resultTable[$key]['revisions']['class'] = 'xray-audit-color-error';
      }
    }

    // Handle empty result set.
    if (empty($resultTable)) {
      $resultTable[] = [
        'id' => $this->t('No nodes found'),
        'label' => '-',
        'language' => '-',
        'revisions' => 0,
      ];
    }

    return $resultTable;
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    switch ($operation) {
      case 'node_count_type':
        return $this->getNodeCountTypeRows();

      case 'node_count_type_langcode':
        return $this->getNodeCountTypeLangcodeRows();

      case 'node_revisions_count':
        return $this->getNodeRevisionsCountRows();

      case 'node_revisions_count_language':
        return $this->getNodeRevisionsCountLanguageRows();
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    $cid = $this->getPluginId() . ':' . $operation;

    $cached_data = $this->pluginRepository->getCachedData($cid);
    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    $data = [
      'header_table' => $this->getHeaders($operation),
      'results_table' => $this->getRows($operation),
    ];

    if ($operation === 'node_revisions_count') {
      $data['extended_data_render'] = [
        '#attached' => [
          'library' => [
            'xray_audit/color',
          ],
        ],
      ];
    }

    $this->pluginRepository->setCacheTagsInv($cid, $data, ['node_list']);
    return $data;
  }

}
