<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Database\Query\Insert;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Link;
use Drupal\paragraphs\ParagraphInterface;
use Drupal\xray_audit\Form\SettingsForm;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "queries_data_paragraphs",
 *   label = @Translation("Paragraphs reports"),
 *   description = @Translation("Metrics about paragraphs entities."),
 *   group = "content_metric",
 *   sort = 2,
 *   local_task = 1,
 *   operations = {
 *     "paragraphs_count_type" = {
 *          "label" = "Paragraphs grouped by type",
 *          "description" = "Number of Paragraphs grouped by type.",
 *           "download" = TRUE
 *       },
 *     "paragraphs_count_langcode" = {
 *          "label" = "Paragraphs grouped by language",
 *           "description" = "Number of Paragraphs grouped by language.",
 *           "download" = TRUE
 *       },
 *      "paragraphs_count_hierarchy" = {
 *          "label" = "Paragraphs grouped hierarchically by type",
 *          "description" = "Number of Paragraphs
 *          grouped hierarchically by type.",
 *          "download" = TRUE
 *       },
 *      "paragraphs_usage" = {
 *          "label" = "Paragraphs Usage",
 *          "description" = "Paragraphs Usage.",
 *          "not_show" = true,
 *          "download" = TRUE
 *       },
 *      "paragraphs_revisions_count" = {
 *          "label" = "Paragraphs grouped by revisions",
 *          "description" = "Paragraphs with the highest number of revisions.",
 *          "download" = TRUE
 *       }
 *    },
 *   dependencies = {"paragraphs"},
 *   install = "paragraphsInstallActions",
 *   uninstall = "paragraphsUninstallActions"
 * )
 *
 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
 */
final class XrayAuditQueryTaskParagraphsPlugin extends XrayAuditTaskPluginBase {

  const XRA_PARAGRAPH_TEMPORARY_TABLE_NAME = 'xra_paragraphs_hierarchy_tmp';

  /**
   * Service "xray_audit.entity_use_paragraph".
   *
   * @var \Drupal\xray_audit\Services\EntityUseInterface
   */
  protected $serviceEntityUseParagraph;

  /**
   * Service "request_stack".
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Service "state".
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Service "config.factory".
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Service "module_handler".
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Configuration settings for Xray Audit.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $xrayAuditConfig;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->serviceEntityUseParagraph = $container->get('xray_audit.entity_use_paragraph');
    $instance->requestStack = $container->get('request_stack');
    $instance->state = $container->get('state');
    $instance->xrayAuditConfig = $container->get('config.factory')->get(SettingsForm::SETTINGS);
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    switch ($operation) {
      case 'paragraphs_count_type':
        return [
          $this->t('ID'),
          $this->t('Label'),
          $this->t('Count'),
        ];

      case 'paragraphs_count_langcode':
        return [
          $this->t('Label'),
          $this->t('Type'),
          $this->t('Langcode'),
          $this->t('Total'),
        ];

      case 'paragraphs_count_hierarchy':
        return [
          $this->t('Level'),
          $this->t('Parent'),
          '',
          $this->t('Count'),
          $this->t('Parent list'),
        ];

      case 'paragraphs_usage':
        return [
          $this->t('Num.'),
          $this->t('Parent entity type'),
          $this->t('Entity type'),
          $this->t('Bundle'),
          $this->t('ID parent'),
          $this->t('Link'),
          $this->t('Status'),
        ];

      case 'paragraphs_revisions_count':
        return [
          $this->t('ID'),
          $this->t('Label'),
          $this->t('Revisions'),
          $this->t('Host'),
        ];
    }
    return [];
  }

  /**
   * Gets the rows for the 'paragraphs_count_type' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getParagraphsCountTypeRows(): array {
    $resultTable = [];
    $storage_type = $this->entityTypeManager->getStorage('paragraphs_type');
    $storage = $this->entityTypeManager->getStorage('paragraph');
    $alias_count = 'count';

    $result = $storage
      ->getAggregateQuery()
      ->accessCheck(FALSE)
      ->currentRevision()
      ->aggregate('id', 'count', NULL, $alias_count)
      ->groupBy('type')
      ->sort('type')
      ->execute();

    $result_processed = [];
    /** @var mixed[] $row */
    foreach ($result as $row) {
      $result_processed[$row['type']] = $row['count'];
    }
    $total = 0;

    $paragraph_types = $storage_type->loadMultiple();
    foreach ($paragraph_types as $key => $paragraph_type) {
      $count = $result_processed[$key] ?? 0;
      $resultTable[] = [
        'id' => $key,
        'label' => $paragraph_type->label(),
        'count' => $count,
      ];
      $total = $total + $count;
    }

    // Sort by count (descending).
    usort($resultTable, function ($a, $b) {
      return $b['count'] <=> $a['count'];
    });

    // Convert to simple arrays for display.
    $sorted_result = [];
    foreach ($resultTable as $row) {
      $sorted_result[] = [$row['id'], $row['label'], $row['count']];
    }

    $sorted_result[] = [
      $this->t('TOTAL'),
      '',
      $total,
    ];
    return $sorted_result;
  }

  /**
   * Gets the rows for the 'paragraphs_count_langcode' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getParagraphsCountLangcodeRows(): array {
    $resultTable = [];
    $storage_type = $this->entityTypeManager->getStorage('paragraphs_type');
    $storage = $this->entityTypeManager->getStorage('paragraph');
    $alias_count = 'count';

    $label = [];
    $types = $storage_type->loadMultiple();
    foreach ($types as $key => $type) {
      $label[$key] = $type->label();
    }

    $query = $storage->getAggregateQuery();
    $result = $query->accessCheck(FALSE)
      ->currentRevision()
      ->aggregate('id', 'COUNT', NULL, $alias_count)
      ->groupBy('langcode')
      ->groupBy('type')
      ->sort('type')
      ->sort('langcode')
      ->execute();

    /** @var mixed[] $row */
    foreach ($result as $row) {
      $resultTable[] = [
        'label' => $label[$row['type']] ?? $row['type'],
        'id' => $row['type'],
        'langcode' => $row['langcode'],
        'total' => $row['count'] ?? 0,
      ];
    }

    // Sort by total count (descending).
    usort($resultTable, function ($a, $b) {
      return $b['total'] <=> $a['total'];
    });

    return $resultTable;
  }

  /**
   * Gets the rows for the 'paragraphs_count_hierarchy' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getParagraphsCountHierarchyRows(): array {
    $this->paragraphsHierarchyCreateTemporaryTable();
    $resultTable = [];
    $language_by_default = $this->languageManager->getDefaultLanguage()->getId();

    $query_select = $this->database->select(self::XRA_PARAGRAPH_TEMPORARY_TABLE_NAME, 'tmp')
      ->fields('tmp', ['level', 'parent_bundle', 'type']);
    $query_select->addExpression('COUNT(tmp.id)', 'count');
    $query_select = $query_select->condition('tmp.langcode', $language_by_default)
      ->groupBy('tmp.hierarchy_id')
      ->groupBy('tmp.parent_bundle')
      ->groupBy('tmp.level')
      ->groupBy('tmp.type')
      ->orderBy('tmp.hierarchy_id')
      ->execute();

    $results = [];
    if ($query_select instanceof StatementInterface) {
      $results = $query_select->fetchAll();
    }

    $paragraph_types = $this->entityTypeManager->getStorage('paragraphs_type')->loadMultiple();
    $total = 0;
    $parent_list_link_base = "<a href='@url' target='_blank'>See usage</a>";

    foreach ($results as $result) {
      $parent_list_link = '';
      $result = (array) $result;
      $indent_string = $result['level'] != '1' ? str_repeat(' ', $result['level'] - 1) . '- ' : NULL;
      $parent_label = $result['level'] == '1' ? $result['parent_bundle'] : NULL;
      if ($result['level'] == '1') {
        $parent_list_url = $this->pluginRepository->getTaskPageOperationFromIdOperation(
          'paragraphs_usage',
          [
            'parent' => $result['parent_bundle'],
            'bundle' => $result['type'],
          ]
        )->toString();
        $parent_list_link = new FormattableMarkup($parent_list_link_base, ['@url' => $parent_list_url]);
      }
      $paragraph_type = $paragraph_types[$result['type']] ?? NULL;
      $label = ($paragraph_type) ? $paragraph_type->label() : $result['type'];
      $resultTable[] = [
        $result['level'],
        $parent_label,
        $indent_string . $label . ' (' . $result['type'] . ')',
        $result['count'],
        $parent_list_link,
      ];
      $total = $total + $result['count'];
    }

    $resultTable[] = [
      $this->t('Total'),
      '',
      '',
      $total,
    ];
    return $resultTable;
  }

  /**
   * Gets the rows for the 'paragraphs_usage' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getParagraphsUsageRows(): array {
    $parameters = $this->getQueryParametersParagraphUsagePlace();
    if ($parameters === NULL) {
      return [];
    }
    $parent_entity_type = $parameters['parent'];
    $paragraph_bundle = $parameters['bundle'];
    $this->serviceEntityUseParagraph->initParameters($parent_entity_type, $paragraph_bundle);

    /** @var mixed[] $resultTable */
    $resultTable = $this->serviceEntityUseParagraph->getEntityUsePlaces();
    $count = 1;
    foreach ($resultTable as &$row) {
      $link = "<a href='@url' target='_blank'>Open page</a>";
      $row = [
        'num' => $count,
        'entity_type_parent' => $row['entity_type_parent'],
        'entity_type' => $row['entity_type'],
        'bundle' => $row['bundle'],
        'nid' => $row['nid'],
        'link' => new FormattableMarkup($link, ['@url' => $row['url']]),
        'status' => $row['status'],
      ];
      $count++;
    }
    return $resultTable;
  }

  /**
   * Gets the rows for the 'paragraphs_revisions_count' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getParagraphsRevisionsCountRows(): array {
    $resultTable = [];
    $alias_count = 'count';
    $label_paragraph_types = [];
    $types = $this->entityTypeManager->getStorage('paragraphs_type')->loadMultiple();
    foreach ($types as $key => $type) {
      $label_paragraph_types[$key] = $type->label();
    }

    $query = $this->database->select('paragraphs_item_revision', 'pir');
    $query->addField('pir', 'id');
    $query->addExpression('COUNT(1)', $alias_count);
    $query->join('paragraphs_item', 'pi', 'pir.id = pi.id');
    $query->addExpression('MAX(pi.type)', 'paragraph_type');
    $query->groupBy('pir.id');
    $query->orderBy($alias_count, 'DESC');
    $query->range(0, 20);

    $result = $query->execute()->fetchAll();
    $max_revision_count = $this->xrayAuditConfig->get('revisions_thresholds.paragraph');
    $paragraph_storage = $this->entityTypeManager->getStorage('paragraph');

    $get_host_entity = function (ParagraphInterface $paragraph) use (&$get_host_entity): ?EntityInterface {
      $parent = $paragraph->getParentEntity();
      return $parent instanceof ParagraphInterface ? $get_host_entity($parent) : $parent;
    };

    foreach ($result as $row) {
      $paragraph_id = $row->id;
      $type = $row->paragraph_type;
      $label = $label_paragraph_types[$type] ?? $this->t('Unknown');
      $revisions = $row->{$alias_count};
      $paragraph = $paragraph_storage->load($paragraph_id);
      $host_entity = $paragraph && $paragraph instanceof ParagraphInterface ? $get_host_entity($paragraph) : NULL;

      $resultTable[$paragraph_id] = [
        'id' => $paragraph_id,
        'label' => $label,
        'revisions' => ['data' => $revisions],
      ];

      if ($host_entity instanceof EntityInterface) {
        $resultTable[$paragraph_id]['link'] = Link::fromTextAndUrl($this->t('Link'), $host_entity->toUrl());
      }

      if (!empty($max_revision_count) && $revisions > $max_revision_count) {
        $resultTable[$paragraph_id]['revisions']['class'] = 'xray-audit-color-error';
      }
    }

    if (empty($resultTable)) {
      $resultTable[] = [
        'id' => $this->t('No paragraphs found'),
        'label' => '-',
        'revisions' => 0,
      ];
    }
    return $resultTable;
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    switch ($operation) {
      case 'paragraphs_count_type':
        return $this->getParagraphsCountTypeRows();

      case 'paragraphs_count_langcode':
        return $this->getParagraphsCountLangcodeRows();

      case 'paragraphs_count_hierarchy':
        return $this->getParagraphsCountHierarchyRows();

      case 'paragraphs_usage':
        return $this->getParagraphsUsageRows();

      case 'paragraphs_revisions_count':
        return $this->getParagraphsRevisionsCountRows();
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    $cid = $this->getPluginId() . ':' . $operation;
    $cache_tags = ['paragraph_list'];

    if ($operation === 'paragraphs_usage') {
      $parameters = $this->getQueryParametersParagraphUsagePlace();
      if ($parameters === NULL) {
        return [];
      }
      $cache_tags[] = 'node_list';
      $cid .= ':' . $parameters['parent'] . ':' . $parameters['bundle'];
    }

    $cached_data = $this->pluginRepository->getCachedData($cid);
    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    $data = [
      'header_table' => $this->getHeaders($operation),
      'results_table' => $this->getRows($operation),
    ];

    if ($operation === 'paragraphs_revisions_count') {
      $data['extended_data_render'] = [
        '#attached' => [
          'library' => [
            'xray_audit/color',
          ],
        ],
      ];
    }

    $this->pluginRepository->setCacheTagsInv($cid, $data, $cache_tags);
    return $data;
  }

  /**
   * Create/update the paragraphs Hierarchy temporally table.
   */
  protected function paragraphsHierarchyCreateTemporaryTable(): void {
    $state_service = $this->state;
    $last_execution = $state_service->get('xray_audit.paragraphs_hierarchy_temporary_table_creation', 0);

    if ($last_execution > strtotime("-1 day")) {
      return;
    }
    if ($this->database->schema()->tableExists(self::XRA_PARAGRAPH_TEMPORARY_TABLE_NAME)) {
      $this->database->truncate(self::XRA_PARAGRAPH_TEMPORARY_TABLE_NAME)->execute();
    }
    else {
      // If paragraphs was installed after xray_audit,
      // we need to create the table.
      $this->paragraphsInstallActions();
    }

    $select = $this->database->select('paragraphs_item_field_data')
      ->fields('paragraphs_item_field_data', [
        'id',
        'parent_id',
        'type',
        'langcode',
        'parent_type',
      ])
      ->condition('id', $this->getParagraphsUsedOnCurrentRevisions(), 'IN')
      ->condition('parent_type', 'paragraph', '<>')
      ->execute();

    if (!$select instanceof StatementInterface) {
      return;
    }
    $results = $select->fetchAll();

    $query = $this->database->insert(self::XRA_PARAGRAPH_TEMPORARY_TABLE_NAME)
      ->fields([
        'hierarchy_id',
        'id',
        'parent_id',
        'type',
        'parent_type',
        'parent_bundle',
        'langcode',
        'level',
      ]);

    foreach ($results as $result) {
      $result = (array) $result;
      $result['hierarchy_id'] = $result['parent_type'] . "-" . $result['type'];
      $result['parent_bundle'] = $result['parent_type'];
      $result['level'] = 1;
      $query->values($result);
      $this->paragraphsHierarchyProcessLevel($result, $query);
    }

    $query->execute();

    $state_service->set('xray_audit.paragraphs_hierarchy_temporary_table_creation', time());
  }

  /**
   * Get the list of paragraphs that are being used on current revisions.
   *
   * This is used to not show unused paragraphs in paragraphs hierarchy.
   *
   * @return array
   *   IDs of unused paragraphs.
   */
  protected function getParagraphsUsedOnCurrentRevisions() {
    $paragraph_field_tables_query = $this->database->select('paragraphs_item_field_data', 'pd');
    $paragraph_field_tables_query->addExpression('CONCAT(pd.parent_type, :underscores , pd.parent_field_name)', 'table_name', [':underscores' => '__']);
    $paragraph_field_tables_query->groupBy('pd.parent_type');
    $paragraph_field_tables_query->groupBy('pd.parent_field_name');
    $executed_paragraph_field_tables_query = $paragraph_field_tables_query->execute();
    if (!$executed_paragraph_field_tables_query instanceof StatementInterface) {
      return [];
    }
    $paragraph_field_tables = $executed_paragraph_field_tables_query->fetchAllKeyed(0, 0);
    $paragraphs_ids = [];
    foreach ($paragraph_field_tables as $paragraph_field_table) {
      if ($paragraph_field_table === NULL) {
        continue;
      }
      [, $paragraph_field_table_field_name] = explode('__', $paragraph_field_table);
      if ($this->database->schema()->tableExists($paragraph_field_table)) {
        $executed_paragraphs_ids = $this->database->select($paragraph_field_table)
          ->fields($paragraph_field_table, [$paragraph_field_table_field_name . '_target_id'])
          ->execute();
        if ($executed_paragraphs_ids instanceof StatementInterface) {
          $paragraphs_ids = array_merge($paragraphs_ids, $executed_paragraphs_ids->fetchAllKeyed(0, 0));
        }
      }
    }
    return $paragraphs_ids;
  }

  /**
   * Process a level of the paragraphs Hierarchy temporally table.
   *
   * @param array $parent
   *   Parent result.
   * @param \Drupal\Core\Database\Query\Insert $query
   *   Drupal Query object.
   */
  protected function paragraphsHierarchyProcessLevel(array $parent, Insert $query): void {
    $results = [];
    $executed_results = $this->database->select('paragraphs_item_field_data')
      ->fields('paragraphs_item_field_data', [
        'id',
        'parent_id',
        'type',
        'langcode',
        'parent_type',
      ])
      ->condition('parent_id', $parent['id'])
      ->condition('parent_type', 'paragraph')
      ->execute();
    if ($executed_results) {
      $results = $executed_results->fetchAll();
    }
    foreach ($results as $result) {
      $result = (array) $result;
      $result['hierarchy_id'] = $parent['hierarchy_id'] . "-" . $result['type'];
      $result['parent_bundle'] = $parent['type'];
      $result['level'] = $parent['level'] + 1;
      $query->values($result);

      if ($result['level'] <= 10) {
        $this->paragraphsHierarchyProcessLevel($result, $query);
      }
    }
  }

  /**
   * Get the table configuration for the paragraphs Hierarchy temporally table.
   *
   * @return array
   *   Table configuration array.
   */
  protected function paragraphsHierarchyGetTableConfiguration(): array {
    return [
      'description' => 'Temporary table to get paragraph usage with hierarchy',
      'fields' => [
        'xra_id' => [
          'type' => 'serial',
          'not null' => TRUE,
        ],
        'hierarchy_id' => [
          'type' => 'varchar',
          'length' => 255,
        ],
        'id' => [
          'type' => 'int',
          'not null' => TRUE,
          'default' => '0',
          'unsigned' => TRUE,
        ],
        'parent_id' => [
          'type' => 'int',
          'not null' => TRUE,
          'default' => '0',
          'unsigned' => TRUE,
        ],
        'type' => [
          'type' => 'varchar',
          'length' => 32,
        ],
        'parent_type' => [
          'type' => 'varchar',
          'length' => 32,
        ],
        'parent_bundle' => [
          'type' => 'varchar',
          'length' => 32,
        ],
        'langcode' => [
          'type' => 'varchar',
          'length' => 10,
        ],
        'level' => [
          'type' => 'int',
          'not null' => TRUE,
          'default' => '1',
        ],
      ],
      'primary key' => ['xra_id'],
    ];
  }

  /**
   * Get the parameters from the URL query.
   *
   * @return array
   *   Parameters.
   */
  protected function getQueryParametersParagraphUsagePlace(): ?array {
    $parent_entity_type_key = 'parent';
    $paragraph_bundle_key = 'bundle';

    $parameters_from_url_query = $this->getParametersFromUrl();

    if (!isset($parameters_from_url_query[$parent_entity_type_key]) || !isset($parameters_from_url_query[$paragraph_bundle_key])) {
      return NULL;
    }
    return [
      $parent_entity_type_key => $parameters_from_url_query[$parent_entity_type_key],
      $paragraph_bundle_key => $parameters_from_url_query[$paragraph_bundle_key],
    ];
  }

  /**
   * Calculate the paragraph usage.
   *
   * @return array
   *   Paragraph usage places.
   */
  protected function paragraphsUsePlace(): array {
    $parameters = $this->getQueryParametersParagraphUsagePlace();
    if ($parameters === NULL) {
      return [];
    }

    $parent_entity_type = $parameters['parent'];
    $paragraph_bundle = $parameters['bundle'];

    $this->serviceEntityUseParagraph->initParameters($parent_entity_type, $paragraph_bundle);

    /** @var mixed[] $resultTable */
    $resultTable = $this->serviceEntityUseParagraph->getEntityUsePlaces();

    $count = 1;
    foreach ($resultTable as &$row) {
      $link = "<a href='@url' target='_blank'>Open page</a>";
      $row = [
        'num' => $count,
        'entity_type_parent' => $row['entity_type_parent'],
        'entity_type' => $row['entity_type'],
        'bundle' => $row['bundle'],
        'nid' => $row['nid'],
        'link' => new FormattableMarkup($link, ['@url' => $row['url']]),
        'status' => $row['status'],
      ];
      $count++;
    }

    $headerTable = [
      $this->t('Num.'),
      $this->t('Parent entity type'),
      $this->t('Entity type'),
      $this->t('Bundle'),
      $this->t('ID parent'),
      $this->t('Link'),
      $this->t('Status'),
    ];

    return [
      'header_table' => $headerTable,
      'results_table' => $resultTable,
    ];
  }

  /**
   * Get values from current url query.
   */
  protected function getParametersFromUrl(): array {
    /**@var \Symfony\Component\HttpFoundation\Request $request*/
    $request = $this->requestStack->getCurrentRequest();
    if (!$request instanceof Request) {
      return [];
    }
    /**@var \Symfony\Component\HttpFoundation\ParameterBag $query*/
    $query = $request->query;
    $parameters = [];
    foreach ($query->all() as $key => $value) {
      $parameters[$key] = $value;
    }
    return $parameters;
  }

  /**
   * Create the requirements for the task.
   */
  public function paragraphsInstallActions(): void {
    if (!$this->database->schema()->tableExists(self::XRA_PARAGRAPH_TEMPORARY_TABLE_NAME)) {
      $this->database->schema()->createTable(self::XRA_PARAGRAPH_TEMPORARY_TABLE_NAME, $this->paragraphsHierarchyGetTableConfiguration());
    }
  }

  /**
   * Uninstall the requirements for the task.
   */
  public function paragraphsUninstallActions(): void {
    $this->state->set('xray_audit.paragraphs_hierarchy_temporary_table_creation', 0);
    $this->database->schema()->dropTable(self::XRA_PARAGRAPH_TEMPORARY_TABLE_NAME);
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvHeaders(string $operation): array {
    $headers = parent::prepareCsvHeaders($operation);
    if (empty($headers)) {
      return [];
    }

    if ($operation === 'paragraphs_revisions_count') {
      return [
        'ID',
        'Label',
        'Revisions',
      ];
    }

    return $headers;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCsvData(string $operation, array $data): array {
    $data = parent::prepareCsvData($operation, $data);
    if ($operation === 'paragraphs_revisions_count') {
      foreach ($data as &$row) {
        unset($row['link']);
      }
    }
    return $data;
  }

}
