<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\ContentModel;

use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "vocabulary_content_model",
 *   label = @Translation("Vocabularies reports"),
 *   description = @Translation("Reports on the vocabularies (taxonomies) of this site."),
 *   group = "content_model",
 *   sort = 15,
 *   operations = {
 *      "vocabulary_description" = {
 *          "label" = "Vocabulary descriptions",
 *          "description" = "Generates a report listing all vocabularies (ID, label, description, and language code)",
 *          "download" = TRUE
 *       }
 *    },
 *   dependencies = {"taxonomy"}
 * )
 */
class XrayAuditVocabularyPlugin extends XrayAuditTaskPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    switch ($operation) {
      case 'vocabulary_description':
        return [
          $this->t('Id'),
          $this->t('Label'),
          $this->t('Description'),
          $this->t('Langcode'),
        ];
    }
    return [];
  }

  /**
   * Gets the rows for the 'vocabulary_description' operation.
   *
   * @return array
   *   The table rows.
   */
  private function getVocabularyDescriptionRows(): array {
    $resultTable = [];
    $vocabularies = Vocabulary::loadMultiple();
    foreach ($vocabularies as $key => $vocabulary) {
      $resultTable[] = [
        $key,
        $vocabulary->label(),
        $vocabulary->get('description'),
        $vocabulary->get('langcode'),
      ];
    }
    uasort($resultTable, function ($a, $b) {
      return strcmp($a[0], $b[0]);
    });
    return $resultTable;
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    switch ($operation) {
      case 'vocabulary_description':
        return $this->getVocabularyDescriptionRows();
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    $cid = $this->getPluginId() . ':' . $operation;

    $cached_data = $this->pluginRepository->getCachedData($cid);
    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    $data = [];
    switch ($operation) {
      case 'vocabulary_description':
        $data = [
          'header_table' => $this->getHeaders($operation),
          'results_table' => $this->getRows($operation),
        ];
        break;
    }

    if (!empty($data)) {
      $this->pluginRepository->setCacheTagsInv($cid, $data, ['taxonomy_vocabulary_list']);
    }
    return $data;
  }

}
