<?php

namespace Drupal\xray_audit\Plugin\xray_audit\tasks\Database;

use Drupal\xray_audit\Plugin\XrayAuditTaskPluginBase;

/**
 * Plugin implementation of queries_data_node.
 *
 * @XrayAuditTaskPlugin (
 *   id = "database_general",
 *   label = @Translation("Database information"),
 *   description = @Translation("Database information."),
 *   group = "database",
 *   sort = 1,
 *   local_task = 1,
 *   operations = {
 *      "database_summary" = {
 *          "label" = "Summary",
 *          "description" = "Provides an overview of the database
 *          including the total number of tables and overall size.",
 *          "download" = TRUE
 *        },
 *      "database" = {
 *          "label" = "Tables",
 *          "description" = "Report on the size of database tables
 *          including row counts and storage usage.",
 *          "download" = TRUE
 *       },
 *    },
 * )
 */
class XrayAuditDatabaseGeneralTaskPlugin extends XrayAuditTaskPluginBase {

  /**
   * {@inheritdoc}
   */
  public function getHeaders(string $operation = ''): array {
    switch ($operation) {
      case 'database':
        return [
          $this->t('Name'),
          $this->t('Number of Rows'),
          $this->t('Data length (MB)'),
          $this->t('Index length (MB)'),
          $this->t('Total length (MB)'),
        ];

      case 'database_summary':
        // Headers for the "largest tables" part of the summary.
        return [
          $this->t('Name'),
          $this->t('Number of Rows'),
          $this->t('Size (MB)'),
        ];
    }
    return [];
  }

  /**
   * Gets the rows for the 'database' (table size) operation.
   *
   * @return array
   *   The table rows.
   */
  private function getDatabaseTableSizeRows(): array {
    $resultTable = $this->getInformationAboutTables();
    $max_size_table = $this->xrayAuditConfig->get('size_thresholds.tables');

    foreach ($resultTable as $key => $table) {
      $total_size_cell = ['data' => $table['total_size']];
      if (!empty($max_size_table) && $table['total_size'] > $max_size_table) {
        $total_size_cell['class'] = 'xray-audit-color-error';
      }
      $resultTable[$key] = [
        'name' => $table['name'],
        'rows' => $table['rows'],
        'size' => $table['size'],
        'index_size' => $table['index_size'],
        'total_size' => $total_size_cell,
      ];
    }
    return $resultTable;
  }

  /**
   * Gets the structured data for the 'database_summary' operation.
   *
   * @return array
   *   Structured data for the summary.
   */
  private function getDatabaseSummaryData(): array {
    $tables = $this->getInformationAboutTables();
    $max_size_table = $this->xrayAuditConfig->get('size_thresholds.tables');

    $num_tables = count($tables);
    $total_size = 0;
    foreach ($tables as $table) {
      $total_size += $table['total_size'];
    }

    $largest_tables_rows = array_slice($tables, 0, 20);
    foreach ($largest_tables_rows as &$largest_table_row) {
      $size_data = $largest_table_row['total_size'];
      $largest_table_row = [
        'name' => $largest_table_row['name'],
        'rows' => $largest_table_row['rows'],
      // This structure is for the table cell in buildDataRenderArray.
        'total_size' => [
          'data' => $size_data,
          'class' => (!empty($max_size_table) && $size_data > $max_size_table) ? 'xray-audit-color-error' : '',
        ],
      ];
    }

    return [
      'table_number' => $num_tables,
      'total_size' => $total_size,
      'largest_tables_rows' => $largest_tables_rows,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getRows(string $operation = ''): array {
    switch ($operation) {
      case 'database':
        return $this->getDatabaseTableSizeRows();

      case 'database_summary':
        // For summary, the main data isn't a simple row set.
        // We return the largest_tables part here for potential direct use if needed,
        // but getDataOperationResult will return the full structured data.
        $summary_data = $this->getDatabaseSummaryData();
        return $summary_data['largest_tables_rows'];
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getDataOperationResult(string $operation = ''): array {
    $cid = $this->getPluginId() . ':' . $operation;
    // Using a generic cache tag, can be invalidated manually or via a short lifetime.
    $cache_tags = ['database_tables_status'];

    $cached_data = $this->pluginRepository->getCachedData($cid);
    if (!empty($cached_data) && is_array($cached_data)) {
      return $cached_data;
    }

    $data = [];
    switch ($operation) {
      case 'database':
        $data = [
          'header_table' => $this->getHeaders($operation),
          'results_table' => $this->getRows($operation),
          'extended_data_render' => [
            '#attached' => [
              'library' => [
                'xray_audit/color',
              ],
            ],
          ],
        ];
        break;

      case 'database_summary':
        // For summary, we fetch the structured data directly.
        // The 'header_table' for the largest_tables part is implicitly defined
        // in buildDataRenderArray or can be fetched via getHeaders if needed there.
        $data = $this->getDatabaseSummaryData();
        break;
    }

    if (!empty($data)) {
      $this->pluginRepository->setCacheTagsInv($cid, $data, $cache_tags);
    }
    return $data;
  }

  /**
   * Get information about tables.
   *
   * @return array
   *   Render array.
   */
  protected function getInformationAboutTables(): array {
    $tables = $this->database->query("SHOW TABLE STATUS")->fetchAll();
    $table_info = [];
    foreach ($tables as $table) {
      $data_size = $this->transformFromBytesToMb($table->Data_length);
      $index_size = $this->transformFromBytesToMb($table->Index_length);
      $table_info[] = [
        'name' => $table->Name,
        'rows' => $table->Rows,
        'size' => $data_size,
        'index_size' => $index_size,
        'total_size' => $data_size + $index_size,
      ];
    }

    usort($table_info, function ($a, $b) {
      return $b['total_size'] <=> $a['total_size'];
    });

    return $table_info;
  }

  /**
   * Transform bytes to MB.
   *
   * @param int|float $bytes
   *   Bytes.
   *
   * @return float
   *   MB.
   */
  protected function transformFromBytesToMb($bytes) {
    if (empty($bytes)) {
      return 0;
    }
    $mg = (int) $bytes / (1024 * 1024);
    // I to ensure that the number is rounded to two decimal places.
    return round($mg, 2);
  }

  /**
   * {@inheritdoc}
   */
  public function buildDataRenderArray(array $data, string $operation = '') {
    if ($operation === 'database') {
      return parent::buildDataRenderArray($data, $operation);
    }

    // This is for 'database_summary' operation.
    $build = [];

    // Section: Database Properties Header.
    $build['database_properties_header'] = [
      '#markup' => '<h4>' . $this->t('Database properties') . '</h4>',
       // Restrict allowed HTML tags for security.
      '#allowed_tags' => ['h4'],
    ];

    // Section: Database Properties Content.
    $build['database_properties_content'] = [
      '#theme' => 'item_list',
      '#items' => [
        $this->t('Number of tables: @num_tables', ['@num_tables' => $data['table_number']]),
        $this->t('Total size of the database (MB): @total_size', ['@total_size' => $data['total_size']]),
      ],
      '#attributes' => ['class' => ['database-properties-list']],
    ];

    // Section: Largest Tables Header.
    $build['largest_tables_header'] = [
      '#markup' => '<h4>' . $this->t('The 20 largest tables') . '</h4>',
      '#allowed_tags' => ['h4'],
    ];

    $header = $this->getHeaders($operation);
    $rows = $data['largest_tables_rows'];
    // Section: Largest Tables Table.
    $build['largest_tables'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#weight' => 20,
      'extended_data_render' => [
        '#attached' => [
          'library' => [
            'xray_audit/color',
          ],
        ],
      ],
    ];

    // Process CSV download using the standardized base class method.
    $this->processCsvDownload($operation, $rows, $build);

    return $build;
  }

}
