<?php

namespace Drupal\xray_audit\Utils;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\xray_audit\Plugin\XrayAuditTaskPluginInterface;
use Drupal\Component\Utility\Html;
use Drupal\Core\Url;

/**
 * Helper class for X-Ray Audit batch operations.
 */
class XrayAuditBatchHelper {

  use StringTranslationTrait;

  /**
   * Processes a single X-Ray audit task operation for CSV generation.
   *
   * @param string $plugin_id
   *   The X-Ray audit task plugin ID.
   * @param string $operation_id
   *   The operation ID within the task plugin.
   * @param array $context
   *   The batch context.
   */
  public static function processSingleTaskOperationCsv($plugin_id, $operation_id, array &$context) {
    if (!isset($context['results']['csv_files'])) {
      $context['results']['csv_files'] = [];
    }
    if (!isset($context['results']['errors'])) {
      $context['results']['errors'] = [];
    }

    /** @var \Drupal\xray_audit\Services\PluginRepositoryInterface $plugin_repository */
    $plugin_repository = \Drupal::service('xray_audit.plugin_repository');
    /** @var \Drupal\Core\File\FileSystemInterface $file_system */
    $file_system = \Drupal::service('file_system');
    $messenger = \Drupal::messenger();

    $task_plugin = $plugin_repository->getInstancePluginTask($plugin_id);

    if (!$task_plugin instanceof XrayAuditTaskPluginInterface) {
      $context['results']['errors'][] = t('Could not instantiate task plugin: @plugin_id', ['@plugin_id' => $plugin_id]);
      $messenger->addError(t('Could not instantiate task plugin: @plugin_id', ['@plugin_id' => $plugin_id]));
      return;
    }

    $plugin_definition = $task_plugin->getPluginDefinition();
    $operation_label = $operation_id;
    if (isset($plugin_definition['operations'][$operation_id]['label'])) {
      $operation_label = (string) $plugin_definition['operations'][$operation_id]['label'];
    }
    $task_label = (string) $task_plugin->label();
    $context['message'] = t('Processing: @task_label - @operation_label...', ['@task_label' => $task_label, '@operation_label' => $operation_label]);

    try {
      // Ensure the plugin uses the trait or implements the methods directly.
      if (!method_exists($task_plugin, 'prepareCsvHeaders') || !method_exists($task_plugin, 'prepareCsvData')) {
        $context['results']['errors'][] = t('Plugin @plugin_id does not support CSV export for operation @operation_id (missing required methods).', ['@plugin_id' => $plugin_id, '@operation_id' => $operation_id]);
        return;
      }

      // The prepareCsvData method (from the trait) is expected to fetch its own data.
      // The second argument to prepareCsvData is $data_from_trait, which is not used when handleCsv is not involved.
      $csv_rows = $task_plugin->prepareCsvData($operation_id, []);
      $csv_headers = $task_plugin->prepareCsvHeaders($operation_id);

      if (empty($csv_headers) && empty($csv_rows)) {
        // This can be a valid case if a report is legitimately empty.
        // We might want to create an empty CSV or skip. For now, let's skip and not log as an error unless headers are present but rows are not (or vice-versa).
        $context['message'] = t('Skipping empty report for: @task_label - @operation_label...', ['@task_label' => $task_label, '@operation_label' => $operation_label]);
        return;
      }

      if (empty($csv_headers) || (empty($csv_rows) && !empty($csv_headers))) {
        // Log if headers are present but no rows, or no headers at all (if rows were expected).
        $context['results']['errors'][] = t('Empty or incomplete CSV content for plugin: @plugin_id, operation: @operation_id', ['@plugin_id' => $plugin_id, '@operation_id' => $operation_id]);
        return;
      }

      $temp_dir = 'temporary://xray_reports';
      $file_system->prepareDirectory($temp_dir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

      $safe_plugin_id = Html::cleanCssIdentifier($plugin_id);
      $safe_operation_id = Html::cleanCssIdentifier($operation_id);
      $filename = "{$safe_plugin_id}_{$safe_operation_id}.csv";
      $filepath = $temp_dir . '/' . $filename;

      $handle = fopen($filepath, 'w');
      if ($handle) {
        fputcsv($handle, $csv_headers);
        foreach ($csv_rows as $row) {
          // Ensure row is an array of scalars for fputcsv.
          $processed_row = array_map(function ($value) {
            if (is_array($value) || is_object($value)) {
              if ($value instanceof TranslatableMarkup) {
                return (string) $value;
              }
              // Attempt to get a string representation for Link objects or similar.
              if (is_object($value) && method_exists($value, 'toString')) {
                return $value->toString();
              }
              if (is_array($value) && isset($value['uri'])) {
                return $value['uri'];
              }
              // Fallback for other complex types.
              return json_encode($value);
            }
            // Ensure scalar.
            return (string) $value;
            // Ensure $row is an array.
          }, (array) $row);
          fputcsv($handle, $processed_row);
        }
        fclose($handle);
        $context['results']['csv_files'][] = $filepath;
      }
      else {
        $context['results']['errors'][] = t('Could not open file for writing: @filepath', ['@filepath' => $filepath]);
        $messenger->addError(t('Could not open file for writing: @filepath', ['@filepath' => $filepath]));
      }

    }
    catch (\Exception $e) {
      $context['results']['errors'][] = t('Error processing plugin @plugin_id, operation @operation_id: @message', [
        '@plugin_id' => $plugin_id,
        '@operation_id' => $operation_id,
        '@message' => $e->getMessage(),
      ]);
      $messenger->addError(t('Error processing plugin @plugin_id, operation @operation_id: @message', [
        '@plugin_id' => $plugin_id,
        '@operation_id' => $operation_id,
        '@message' => $e->getMessage(),
      ]));
    }
  }

  /**
   * Finished callback for the X-Ray audit CSV generation batch.
   *
   * @param bool $success
   *   Indicates whether the batch operation was successful.
   * @param array $results
   *   Holds the results of the batch operation.
   * @param array $operations
   *   If $success is FALSE, contains the operations that remained unprocessed.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse|null
   *   A redirect response to the download controller if successful, or null otherwise.
   */
  public static function downloadAllCsvFinished($success, array $results, array $operations) {
    $messenger = \Drupal::messenger();

    if ($success) {
      if (!empty($results['csv_files'])) {
        /** @var \Drupal\Core\File\FileSystemInterface $file_system */
        $file_system = \Drupal::service('file_system');
        /** @var \Drupal\Core\Archiver\ArchiverManager $archiver_manager */
        $archiver_manager = \Drupal::service('plugin.manager.archiver');

        $zip_filename_base = 'xray_audit_all_reports_' . date('Ymd_His');
        // Sanitize.
        $zip_filename_base = preg_replace('/[^a-zA-Z0-9_-]/', '_', $zip_filename_base);
        $zip_filepath = 'temporary://' . $zip_filename_base . '.zip';

        try {
          // Pass the stream wrapper path directly. The archiver will create the file.
          $zip = $archiver_manager->getInstance(['filepath' => $zip_filepath, 'flags' => \ZipArchive::CREATE | \ZipArchive::OVERWRITE]);
          if (!$zip) {
            throw new \Exception('Could not load archiver for zip file: ' . $zip_filepath);
          }

          foreach ($results['csv_files'] as $csv_filepath) {
            // When adding files to zip, realpath is appropriate if the file exists.
            $real_csv_path = $file_system->realpath($csv_filepath);
            if ($real_csv_path && file_exists($real_csv_path)) {
              $zip->add($real_csv_path, basename($csv_filepath));
            }
          }

          // Explicitly close the archive to ensure it's written to disk.
          // Drupal's Zip archiver might do this in its destructor, but being explicit can help.
          // The ArchiverInterface doesn't define a close() method, so we rely on the destructor
          // or specific implementation details if available.
          // For \ZipArchive, close() is standard. Drupal's wrapper should handle it.
          // Let's ensure the $zip object is destructed before we try to use the file.
          // This will trigger the destructor of Drupal\Core\Archiver\Zip.
          unset($zip);

          $real_zip_filepath = $file_system->realpath($zip_filepath);
          if (!$real_zip_filepath || !file_exists($real_zip_filepath)) {
            // It's possible the stream wrapper path is needed if realpath fails for some reason post-creation.
            // However, BinaryFileResponse typically needs a real path.
            throw new \Exception('ZIP file was not created or is not accessible after attempting to close. Path: ' . $zip_filepath . ' (Real path: ' . ($real_zip_filepath ?: 'failed to resolve') . ')');
          }

          // Store the path to the ZIP file in the session for the download controller.
          // Use the stream wrapper path, as realpath might not be consistent across requests if the temp dir changes.
          // The download controller will resolve it to a real path.
          $session = \Drupal::request()->getSession();
          // Store stream wrapper path.
          $session->set('xray_audit_batch_zip_filepath', $zip_filepath);

          // Clean up individual CSV files. The ZIP file will be cleaned by the download controller.
          foreach ($results['csv_files'] as $csv_filepath) {
            $real_csv_path_to_delete = $file_system->realpath($csv_filepath);
            if ($real_csv_path_to_delete && file_exists($real_csv_path_to_delete)) {
              $file_system->delete($real_csv_path_to_delete);
            }
          }
          // Do NOT delete the $real_zip_filepath here. It will be deleted by the download controller.
          // Clean up the temporary directory if it's now empty (excluding the zip file itself).
          $temp_dir_path = $file_system->realpath('temporary://xray_reports');
          if ($temp_dir_path && is_dir($temp_dir_path)) {
            $files_in_temp_dir = array_diff(scandir($temp_dir_path), ['.', '..']);
            // Check if the only remaining file is our zip file.
            if (count($files_in_temp_dir) === 1 && basename(reset($files_in_temp_dir)) === basename($real_zip_filepath)) {
              // If only the zip remains, don't delete the directory yet.
              // If other unexpected files remain, this won't delete the dir.
            }
            elseif (empty($files_in_temp_dir)) {
              $file_system->deleteRecursive($temp_dir_path);
            }
          }

          if (!empty($results['errors'])) {
            $error_messages = [];
            foreach ($results['errors'] as $error) {
              if ($error instanceof TranslatableMarkup) {
                $error_messages[] = (string) $error;
              }
              elseif (is_string($error)) {
                $error_messages[] = $error;
              }
            }
            if (!empty($error_messages)) {
              $messenger->addWarning(t('Some reports encountered errors during generation: @errors', ['@errors' => implode('; ', $error_messages)]));
            }
          }
          $download_url = Url::fromRoute('xray_audit.download_generated_zip_file')->toString();
          $messenger->addStatus(t('All X-Ray audit reports have been generated. <a href=":url">Click here to download your ZIP archive</a>.', [':url' => $download_url]));

          if (!empty($results['errors'])) {
            $error_messages = [];
            foreach ($results['errors'] as $error) {
              if ($error instanceof TranslatableMarkup) {
                $error_messages[] = (string) $error;
              }
              elseif (is_string($error)) {
                $error_messages[] = $error;
              }
            }
            if (!empty($error_messages)) {
              $messenger->addWarning(t('Some reports encountered errors during generation: @errors', ['@errors' => implode('; ', $error_messages)]));
            }
          }
          // Return NULL to allow the 'redirect' from batch_set() to take effect.
          // The user will be redirected to xray_audit.home and see the message with the download link.
          return NULL;

        }
        catch (\Exception $e) {
          $messenger->addError(t('Could not create ZIP archive: @error', ['@error' => $e->getMessage()]));
          \Drupal::logger('xray_audit')->error('Failed to create ZIP archive for X-Ray reports: @error', ['@error' => $e->getMessage()]);
        }

      }
      else {
        $messenger->addWarning(t('No CSV files were generated.'));
        if (!empty($results['errors'])) {
          $error_messages = [];
          foreach ($results['errors'] as $error) {
            if ($error instanceof TranslatableMarkup) {
              $error_messages[] = (string) $error;
            }
            elseif (is_string($error)) {
              $error_messages[] = $error;
            }
          }
          if (!empty($error_messages)) {
            $messenger->addWarning(t('The following errors occurred: @errors', ['@errors' => implode('; ', $error_messages)]));
          }
        }
      }
    }
    else {
      // An error occurred during batch processing.
      $error_operation = reset($operations);
      if ($error_operation) {
        $messenger->addError(t('An error occurred while processing @operation with arguments: @args', [
          '@operation' => is_array($error_operation) && isset($error_operation[0]) ? (is_array($error_operation[0]) ? implode('::', $error_operation[0]) : (string) $error_operation[0]) : 'N/A',
          '@args' => is_array($error_operation) && isset($error_operation[1]) ? print_r($error_operation[1], TRUE) : 'N/A',
        ]));
      }
      else {
        $messenger->addError(t('An unspecified error occurred during batch processing.'));
      }
    }
    // If not successful, or no files, or error in zip creation, return NULL.
    // Batch API will redirect to 'redirect' URL from batch_set (xray_audit.home)
    // where error/warning messages will be displayed.
    return NULL;
  }

}
