<?php

namespace Drupal\Tests\xray_audit\Functional\Controller;

use Drupal\xray_audit\Controller\XrayAuditHomeController;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the XrayAuditHomeController.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 * @group critical
 */
class XrayAuditHomeControllerTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'xray_audit',
  ];

  /**
   * A user with permission to access xray audit.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $privilegedUser;

  /**
   * A user without permission to access xray audit.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $unprivilegedUser;

  /**
   * The plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create users with different permissions.
    $this->privilegedUser = $this->drupalCreateUser([
      'xray_audit access',
      'access content',
    ]);

    $this->unprivilegedUser = $this->drupalCreateUser([
      'access content',
    ]);

    // Get services.
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');
    $this->renderer = $this->container->get('renderer');
  }

  /**
   * Tests that unprivileged users cannot access home route.
   */
  public function testAccessDeniedForUnprivilegedUsers() {
    // Arrange: Login as unprivileged user.
    $this->drupalLogin($this->unprivilegedUser);

    // Act: Try to access home route.
    $this->drupalGet('/admin/reports/xray-audit');

    // Assert: Access should be denied.
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests that privileged users can access home route.
   */
  public function testAccessGrantedForPrivilegedUsers() {
    // Arrange: Login as privileged user.
    $this->drupalLogin($this->privilegedUser);

    // Act: Access home route.
    $this->drupalGet('/admin/reports/xray-audit');

    // Assert: Should be accessible.
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests that home page displays group blocks.
   */
  public function testHomePageDisplaysGroupBlocks() {
    // Arrange: Login as privileged user.
    $this->drupalLogin($this->privilegedUser);

    // Get group definitions.
    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();

    // Act: Access home route.
    $this->drupalGet('/admin/reports/xray-audit');

    // Assert: Page should display group information.
    $this->assertSession()->statusCodeEquals(200);

    // Check if at least some group labels are present (if there are groups).
    if (!empty($group_definitions)) {
      foreach ($group_definitions as $group) {
        if (!empty($group['label'])) {
          $this->assertSession()->pageTextContains((string) $group['label']);
          // At least one group label found.
          break;
        }
      }
    }
  }

  /**
   * Tests build() returns proper render array.
   */
  public function testBuildReturnsRenderArray() {
    // Arrange: Get controller.
    $controller = XrayAuditHomeController::create($this->container);

    // Act: Call build method.
    $render_array = $controller->build();

    // Assert: Should return an array.
    $this->assertIsArray($render_array, 'build() should return an array');
  }

  /**
   * Tests build() sorts groups by sort order.
   */
  public function testBuildSortsGroupsBySortOrder() {
    // Arrange: Get controller and group definitions.
    $controller = XrayAuditHomeController::create($this->container);
    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();

    if (count($group_definitions) < 2) {
      $this->markTestSkipped('Need at least 2 groups to test sorting');
    }

    // Act: Build render array.
    $render_array = $controller->build();

    // Assert: Groups should be in sorted order.
    // We verify this indirectly by checking that the render array has the expected structure.
    $this->assertIsArray($render_array);

    // Count admin blocks (exclude 'form' key).
    $admin_blocks = array_filter($render_array, function ($key) {
      return $key !== 'form';
    }, ARRAY_FILTER_USE_KEY);

    // Should have multiple admin blocks if multiple groups.
    $this->assertGreaterThanOrEqual(1, count($admin_blocks));
  }

  /**
   * Tests build() includes FlushCacheForm.
   */
  public function testBuildIncludesFlushCacheForm() {
    // Arrange: Get controller.
    $controller = XrayAuditHomeController::create($this->container);

    // Act: Build render array.
    $render_array = $controller->build();

    // Assert: Should include form.
    $this->assertArrayHasKey('form', $render_array, 'Render array should include form key');
    $this->assertIsArray($render_array['form'], 'Form should be an array');
  }

  /**
   * Tests buildTaskItems() is used for each group.
   */
  public function testBuildCallsBuildTaskItemsForEachGroup() {
    // Arrange: Get controller.
    $controller = XrayAuditHomeController::create($this->container);
    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();

    if (empty($group_definitions)) {
      $this->markTestSkipped('Need at least 1 group to test buildTaskItems');
    }

    // Act: Build render array.
    $render_array = $controller->build();

    // Assert: Each group should have tasks built (indirectly verified).
    // Count the number of admin blocks.
    $admin_block_count = 0;
    foreach ($render_array as $key => $element) {
      if ($key !== 'form' && isset($element['#theme']) && $element['#theme'] === 'admin_block') {
        $admin_block_count++;
      }
    }

    // Should have admin blocks corresponding to groups.
    $this->assertGreaterThanOrEqual(1, $admin_block_count, 'Should have admin blocks for groups');
  }

  /**
   * Tests home page title is correct.
   */
  public function testHomePageTitle() {
    // Arrange: Login as privileged user.
    $this->drupalLogin($this->privilegedUser);

    // Act: Access home route.
    $this->drupalGet('/admin/reports/xray-audit');

    // Assert: Page title should be correct.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Xray Audit Reports');
  }

  /**
   * Tests that XrayAuditBuildTaskItemsInControllerTrait is used.
   */
  public function testTraitUsage() {
    // Arrange: Get controller.
    $controller = XrayAuditHomeController::create($this->container);

    // Get a group ID to test.
    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();
    if (empty($group_definitions)) {
      $this->markTestSkipped('Need at least 1 group to test trait');
    }

    $first_group = reset($group_definitions);
    $group_id = $first_group['id'];

    // Act: Call buildTaskItems (from trait).
    $task_items = $controller->buildTaskItems($group_id);

    // Assert: Should return an array.
    $this->assertIsArray($task_items, 'buildTaskItems() should return an array');
  }

}
