<?php

namespace Drupal\Tests\xray_audit\Kernel\Controller;

use Drupal\Core\Url;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;
use Drupal\xray_audit\Controller\XrayAuditBuildTaskItemsInControllerTrait;
use Drupal\xray_audit\Services\PluginRepositoryInterface;

/**
 * Tests for XrayAuditBuildTaskItemsInControllerTrait.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditBuildTaskItemsInControllerTraitTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'xray_audit',
  ];

  /**
   * The plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Test implementation of the trait.
   *
   * @var \Drupal\Tests\xray_audit\Kernel\Controller\TestBuildTaskItemsClass
   */
  protected $traitObject;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');
    $this->traitObject = new TestBuildTaskItemsClass($this->pluginRepository);
  }

  /**
   * Tests buildTaskItems with valid group.
   *
   */
  public function testBuildTaskItemsWithValidGroup(): void {
    // Get a valid group plugin ID.
    $group_definitions = $this->pluginRepository->getGroupPluginDefinitions();
    $this->assertNotEmpty($group_definitions, 'Should have at least one group plugin');

    $first_group = reset($group_definitions);
    $group_id = $first_group['id'];

    $result = $this->traitObject->buildTaskItems($group_id);

    $this->assertIsArray($result);
  }

  /**
   * Tests buildTaskItems with invalid group returns error message.
   *
   */
  public function testBuildTaskItemsWithInvalidGroup(): void {
    $result = $this->traitObject->buildTaskItems('non_existent_group');

    $this->assertIsArray($result);
    $this->assertArrayHasKey('#markup', $result);
    $this->assertNotEmpty($result['#markup']);
  }

  /**
   * Tests buildTaskItems with group that has no tasks.
   *
   */
  public function testBuildTaskItemsWithGroupHavingNoTasks(): void {
    // Create a mock group plugin that returns empty task definitions.
    $mock_group = $this->createMock('\Drupal\xray_audit\Plugin\XrayAuditGroupPluginInterface');
    $mock_group->method('getPluginTaskDefinitionsThisGroup')
      ->willReturn([]);

    $mock_repository = $this->createMock(PluginRepositoryInterface::class);
    $mock_repository->method('getInstancePluginGroup')
      ->willReturn($mock_group);

    $test_object = new TestBuildTaskItemsClass($mock_repository);
    $result = $test_object->buildTaskItems('test_group');

    $this->assertIsArray($result);
    $this->assertEmpty($result);
  }

  /**
   * Tests buildTaskItems processes all operations when not local_task.
   *
   */
  public function testBuildTaskItemsProcessesAllOperations(): void {
    $task_definitions = [
      [
        'id' => 'test_task',
        'label' => 'Test Task',
        'description' => 'Test description',
        'local_task' => FALSE,
        'operations' => [
          'op1' => [
            'label' => 'Operation 1',
            'description' => 'Op 1 description',
          ],
          'op2' => [
            'label' => 'Operation 2',
            'description' => 'Op 2 description',
          ],
        ],
      ],
    ];

    $mock_group = $this->createMock('\Drupal\xray_audit\Plugin\XrayAuditGroupPluginInterface');
    $mock_group->method('getPluginTaskDefinitionsThisGroup')
      ->willReturn($task_definitions);

    $mock_repository = $this->createMock(PluginRepositoryInterface::class);
    $mock_repository->method('getInstancePluginGroup')
      ->willReturn($mock_group);
    $mock_repository->method('getOnePluginTaskRouteInfo')
      ->willReturn(['route_name' => 'test.route']);

    $test_object = new TestBuildTaskItemsClass($mock_repository);
    $result = $test_object->buildTaskItems('test_group');

    $this->assertCount(2, $result);
  }

  /**
   * Tests buildTaskItems processes only first operation when local_task.
   *
   */
  public function testBuildTaskItemsProcessesOnlyFirstOperationForLocalTask(): void {
    $task_definitions = [
      [
        'id' => 'test_task',
        'label' => 'Test Task',
        'description' => 'Test description',
        'local_task' => TRUE,
        'operations' => [
          'op1' => [
            'label' => 'Operation 1',
            'description' => 'Op 1 description',
          ],
          'op2' => [
            'label' => 'Operation 2',
            'description' => 'Op 2 description',
          ],
        ],
      ],
    ];

    $mock_group = $this->createMock('\Drupal\xray_audit\Plugin\XrayAuditGroupPluginInterface');
    $mock_group->method('getPluginTaskDefinitionsThisGroup')
      ->willReturn($task_definitions);

    $mock_repository = $this->createMock(PluginRepositoryInterface::class);
    $mock_repository->method('getInstancePluginGroup')
      ->willReturn($mock_group);
    $mock_repository->method('getOnePluginTaskRouteInfo')
      ->willReturn(['route_name' => 'test.route']);

    $test_object = new TestBuildTaskItemsClass($mock_repository);
    $result = $test_object->buildTaskItems('test_group');

    // Should only have 1 item for local_task case.
    $this->assertCount(1, $result);
  }

  /**
   * Tests buildAllOperations skips operations with not_show flag.
   *
   */
  public function testBuildAllOperationsSkipsNotShowOperations(): void {
    $task_definitions = [
      [
        'id' => 'test_task',
        'label' => 'Test Task',
        'local_task' => FALSE,
        'operations' => [
          'op1' => [
            'label' => 'Operation 1',
            'not_show' => FALSE,
          ],
          'op2' => [
            'label' => 'Operation 2',
            'not_show' => TRUE,
          ],
          'op3' => [
            'label' => 'Operation 3',
          ],
        ],
      ],
    ];

    $mock_group = $this->createMock('\Drupal\xray_audit\Plugin\XrayAuditGroupPluginInterface');
    $mock_group->method('getPluginTaskDefinitionsThisGroup')
      ->willReturn($task_definitions);

    $mock_repository = $this->createMock(PluginRepositoryInterface::class);
    $mock_repository->method('getInstancePluginGroup')
      ->willReturn($mock_group);
    $mock_repository->method('getOnePluginTaskRouteInfo')
      ->willReturn(['route_name' => 'test.route']);

    $test_object = new TestBuildTaskItemsClass($mock_repository);
    $result = $test_object->buildTaskItems('test_group');

    // Should have 2 items (op1 and op3, skipping op2).
    $this->assertCount(2, $result);
  }

  /**
   * Tests buildTaskItem returns correctly structured array.
   *
   */
  public function testBuildTaskItemStructure(): void {
    $title = 'Test Title';
    $route_name = 'test.route';
    $description = 'Test Description';

    $result = $this->traitObject->buildTaskItem($title, $route_name, $description);

    $this->assertIsArray($result);
    $this->assertArrayHasKey('title', $result);
    $this->assertArrayHasKey('url', $result);
    $this->assertArrayHasKey('localized_options', $result);
    $this->assertArrayHasKey('description', $result);

    $this->assertEquals($title, $result['title']);
    $this->assertInstanceOf(Url::class, $result['url']);
    $this->assertIsArray($result['localized_options']);
    $this->assertEquals($description, $result['description']);
  }

  /**
   * Tests getTaskRouteName returns correct route name.
   *
   */
  public function testGetTaskRouteNameReturnsCorrectRouteName(): void {
    $task_id = 'test_task';
    $operation_id = 'test_operation';
    $expected_route_name = 'xray_audit.task.test';

    $mock_repository = $this->createMock(PluginRepositoryInterface::class);
    $mock_repository->method('getOnePluginTaskRouteInfo')
      ->with($task_id, $operation_id)
      ->willReturn(['route_name' => $expected_route_name]);

    $test_object = new TestBuildTaskItemsClass($mock_repository);
    $result = $test_object->getTaskRouteName($task_id, $operation_id);

    $this->assertEquals($expected_route_name, $result);
  }

}

/**
 * Test class that implements the trait for testing purposes.
 */
class TestBuildTaskItemsClass {
  use XrayAuditBuildTaskItemsInControllerTrait {
    buildTaskItem as public;
    getTaskRouteName as public;
  }

  /**
   * The plugin repository.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Constructor.
   *
   * @param \Drupal\xray_audit\Services\PluginRepositoryInterface $plugin_repository
   *   The plugin repository.
   */
  public function __construct(PluginRepositoryInterface $plugin_repository) {
    $this->pluginRepository = $plugin_repository;
  }

  /**
   * Mock translation method.
   *
   * @param string $string
   *   The string to translate.
   *
   * @return string
   *   The translated string.
   */
  protected function t($string) {
    return $string;
  }

}
