<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentAccessControl;

use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;
use Drupal\user\Entity\Role;

/**
 * Comprehensive tests for XrayAuditQueryTaskRolesPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditQueryTaskRolesPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The roles plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentAccessControl\XrayAuditQueryTaskRolesPlugin
   */
  protected $plugin;

  /**
   * Test roles created for testing.
   *
   * @var \Drupal\user\Entity\Role[]
   */
  protected $testRoles = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Get task plugin manager.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('queries_data_roles');

    // Create test roles with various permission states.
    $this->createTestRoles();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clean up test roles.
    foreach ($this->testRoles as $role) {
      if ($role) {
        $role->delete();
      }
    }
    parent::tearDown();
  }

  /**
   * Creates test roles with various permission states.
   */
  protected function createTestRoles() {
    // Role with multiple permissions.
    $editor = Role::create([
      'id' => 'editor',
      'label' => 'Editor',
    ]);
    $editor->grantPermission('access content');
    $editor->grantPermission('create article content');
    $editor->grantPermission('edit own article content');
    $editor->grantPermission('delete own article content');
    $editor->save();
    $this->testRoles['editor'] = $editor;

    // Role with single permission.
    $reviewer = Role::create([
      'id' => 'reviewer',
      'label' => 'Reviewer',
    ]);
    $reviewer->grantPermission('access content');
    $reviewer->save();
    $this->testRoles['reviewer'] = $reviewer;

    // Role without permissions (empty).
    $viewer = Role::create([
      'id' => 'viewer',
      'label' => 'Viewer',
    ]);
    $viewer->save();
    $this->testRoles['viewer'] = $viewer;

    // Role with ID that sorts first alphabetically.
    $admin = Role::create([
      'id' => 'admin',
      'label' => 'Administrator',
    ]);
    $admin->grantPermission('administer site configuration');
    $admin->grantPermission('access administration pages');
    $admin->save();
    $this->testRoles['admin'] = $admin;
  }

  // -------------------------------------------------------------------------
  // rolesDescription()
  // -------------------------------------------------------------------------

  /**
   * Tests rolesDescription() returns proper structure.
   *
   */
  public function testRolesDescriptionReturnsProperStructure() {
    // Act - rolesDescription() now returns just the rows, use getDataOperationResult() instead.
    $result = $this->plugin->getDataOperationResult('role_list');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header.
    $this->assertCount(2, $result['header_table']);
    $header = $result['header_table'];
    $this->assertNotEmpty($header[0], 'First header column should be Id');
    $this->assertNotEmpty($header[1], 'Second header column should be Label');

    // Verify results table is an array.
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests rolesDescription() includes all roles.
   *
   */
  public function testRolesDescriptionIncludesAllRoles() {
    // Act - rolesDescription() now returns just the rows array.
    $results_table = $this->plugin->rolesDescription();

    // Assert.
    // Should include default roles.
    $this->assertArrayHasKey('anonymous', $results_table, 'Should include anonymous role');
    $this->assertArrayHasKey('authenticated', $results_table, 'Should include authenticated role');

    // Should include our custom test roles.
    $this->assertArrayHasKey('editor', $results_table, 'Should include editor role');
    $this->assertArrayHasKey('reviewer', $results_table, 'Should include reviewer role');
    $this->assertArrayHasKey('viewer', $results_table, 'Should include viewer role');
    $this->assertArrayHasKey('admin', $results_table, 'Should include admin role');

    // Verify each role has proper structure [id, label].
    foreach ($results_table as $role_id => $role_data) {
      $this->assertIsArray($role_data);
      $this->assertCount(2, $role_data, "Role $role_id should have 2 values: id and label");
      $this->assertEquals($role_id, $role_data[0], "First value should be role ID");
      $this->assertIsString($role_data[1], "Second value should be label string");
    }
  }

  // -------------------------------------------------------------------------
  // permissionsPerRole()
  // -------------------------------------------------------------------------

  /**
   * Tests permissionsPerRole() includes all roles.
   */
  public function testPermissionsPerRoleIncludesAllRoles() {
    // Act - permissionsPerRole() now returns just the rows array.
    $results_table = $this->plugin->permissionsPerRole();

    // Assert.
    // Should include default roles.
    $this->assertArrayHasKey('anonymous', $results_table, 'Should include anonymous role');
    $this->assertArrayHasKey('authenticated', $results_table, 'Should include authenticated role');

    // Should include our custom test roles.
    $this->assertArrayHasKey('editor', $results_table, 'Should include editor role');
    $this->assertArrayHasKey('reviewer', $results_table, 'Should include reviewer role');
    $this->assertArrayHasKey('viewer', $results_table, 'Should include viewer role');
    $this->assertArrayHasKey('admin', $results_table, 'Should include admin role');

    // Verify each role has proper structure [label, permissions_string].
    foreach ($results_table as $role_id => $role_data) {
      $this->assertIsArray($role_data);
      $this->assertCount(2, $role_data, "Role $role_id should have 2 values: label and permissions");
      $this->assertIsString($role_data[0], "First value should be label string");
      $this->assertIsString($role_data[1], "Second value should be permissions string");
    }
  }

  /**
   * Tests permissionsPerRole() lists permissions correctly.
   */
  public function testPermissionsPerRoleListsPermissions() {
    // Act - permissionsPerRole() now returns just the rows array.
    $results_table = $this->plugin->permissionsPerRole();

    // Assert.
    // Verify editor role has permissions (4 permissions granted).
    $this->assertArrayHasKey('editor', $results_table);
    $editor_data = $results_table['editor'];
    $this->assertEquals('Editor', $editor_data[0], 'Editor label should match');

    $editor_permissions = $editor_data[1];
    $this->assertIsString($editor_permissions);
    $this->assertNotEmpty($editor_permissions, 'Editor should have permissions');

    // Verify permissions are comma-separated if multiple permissions exist.
    $permission_array = explode(', ', $editor_permissions);
    $this->assertGreaterThanOrEqual(1, count($permission_array), 'Should have at least one permission');

    // If multiple permissions, verify comma-separated format.
    if (count($permission_array) > 1) {
      $this->assertStringContainsString(', ', $editor_permissions, 'Multiple permissions should be comma-separated');
    }

    // Verify at least one of our granted permissions is present.
    // Note: Some permissions might not be available in test environment.
    $expected_permissions = [
      'access content',
      'create article content',
      'edit own article content',
      'delete own article content',
    ];
    $found_permission = FALSE;
    foreach ($expected_permissions as $expected) {
      if (strpos($editor_permissions, $expected) !== FALSE) {
        $found_permission = TRUE;
        break;
      }
    }
    $this->assertTrue($found_permission, 'Editor should have at least one granted permission');

    // Verify reviewer role has single permission.
    $this->assertArrayHasKey('reviewer', $results_table);
    $reviewer_data = $results_table['reviewer'];
    $reviewer_permissions = $reviewer_data[1];
    $this->assertEquals('access content', $reviewer_permissions, 'Reviewer should have single permission');

    // Verify admin role has permissions.
    $this->assertArrayHasKey('admin', $results_table);
    $admin_data = $results_table['admin'];
    $admin_permissions = $admin_data[1];
    $this->assertNotEmpty($admin_permissions, 'Admin should have permissions');
    $this->assertStringContainsString('administer site configuration', $admin_permissions);
  }

  // -------------------------------------------------------------------------
  // DATA STRUCTURE & INTEGRATION TESTS
  // -------------------------------------------------------------------------

}
