<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentDisplay;

use Drupal\Core\Render\RenderContext;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;
use Drupal\xray_audit\Plugin\xray_audit\tasks\ContentDisplay\XrayAuditEntityDisplaysPlugin;
use Drupal\Core\Entity\Entity\EntityViewDisplay;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Comprehensive TDD tests for XrayAuditEntityDisplaysPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditEntityDisplaysPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'link',
    'taxonomy',
    'file',
    'image',
    'options',
    'datetime',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The entity displays plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentDisplay\XrayAuditEntityDisplaysPlugin
   */
  protected $plugin;

  /**
   * The entity display architecture service.
   *
   * @var \Drupal\xray_audit\Services\EntityDisplayArchitectureInterface
   */
  protected $entityDisplayArchitectureService;

  /**
   * The plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install additional schemas for taxonomy and file support.
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('file');
    $this->installSchema('file', ['file_usage']);
    $this->installConfig(['field', 'node', 'taxonomy', 'file', 'image', 'system']);

    // Get services from container.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');
    $this->entityDisplayArchitectureService = $this->container->get('xray_audit.entity_display_architecture');
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');

    // Create plugin instance for testing.
    $this->plugin = $this->taskPluginManager->createInstance('entity_displays');

    // Set up comprehensive test data.
    $this->setUpTestContentTypes();
    $this->setUpTestFields();
    $this->setUpTestDisplayModes();
    $this->setUpTestTaxonomy();
  }

  /**
   * Helper to execute getDataOperationResult within a render context.
   *
   * The EntityDisplayArchitecture service uses the renderer which requires
   * a render context. This helper wraps the operation execution.
   *
   * @param string $operation
   *   The operation name.
   *
   * @return array
   *   The operation result data.
   */
  protected function getDataWithRenderContext(string $operation): array {
    $renderer = $this->container->get('renderer');
    $data = [];
    $renderer->executeInRenderContext(new RenderContext(), function () use ($operation, &$data) {
      $data = $this->plugin->getDataOperationResult($operation);
    });
    return $data;
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    parent::tearDown();
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Sets up test content types.
   *
   * Creates multiple content types with different configurations to test
   * display mode analysis across various scenarios.
   */
  protected function setUpTestContentTypes(): void {
    // Create article content type.
    $this->createContentType('article', 'Article');

    // Create page content type.
    $this->createContentType('page', 'Page');

    // Create blog content type for additional testing.
    $this->createContentType('blog', 'Blog');
  }

  /**
   * Sets up test fields on content types.
   *
   * Creates various field types to test display formatter configurations.
   */
  protected function setUpTestFields(): void {
    // Create text field on article.
    $this->createTestField('field_summary', 'node', 'article', 'text_long');

    // Create image field on article.
    $this->createTestField('field_image', 'node', 'article', 'image');

    // Create link field on article.
    $this->createTestField('field_link', 'node', 'article', 'link');

    // Create text field on page.
    $this->createTestField('field_body', 'node', 'page', 'text_long');

    // Create date field on page.
    $this->createTestField('field_date', 'node', 'page', 'datetime');

    // Create text field on blog.
    $this->createTestField('field_content', 'node', 'blog', 'text_long');
  }

  /**
   * Sets up test view display modes.
   *
   * Configures various display modes (default, teaser, full) with different
   * field formatter configurations to test comprehensive display analysis.
   */
  protected function setUpTestDisplayModes(): void {
    // Configure default display for article with proper ID.
    $display = EntityViewDisplay::create([
      'id' => 'node.article.default',
      'targetEntityType' => 'node',
      'bundle' => 'article',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $display->setComponent('field_summary', [
      'type' => 'text_default',
      'weight' => 1,
      'label' => 'above',
      'settings' => [],
    ]);
    $display->setComponent('field_image', [
      'type' => 'image',
      'weight' => 2,
      'label' => 'hidden',
      'settings' => [
        'image_style' => 'large',
        'image_link' => '',
      ],
    ]);
    $display->setComponent('field_link', [
      'type' => 'link',
      'weight' => 3,
      'label' => 'inline',
      'settings' => [
        'trim_length' => 80,
        'url_only' => FALSE,
        'url_plain' => FALSE,
        'rel' => '',
        'target' => '',
      ],
    ]);
    $display->save();

    // Configure teaser display for article.
    $teaser_display = EntityViewDisplay::create([
      'id' => 'node.article.teaser',
      'targetEntityType' => 'node',
      'bundle' => 'article',
      'mode' => 'teaser',
      'status' => TRUE,
    ]);
    $teaser_display->setComponent('field_summary', [
      'type' => 'text_trimmed',
      'weight' => 1,
      'label' => 'hidden',
      'settings' => [
        'trim_length' => 200,
      ],
    ]);
    $teaser_display->setComponent('field_image', [
      'type' => 'image',
      'weight' => 0,
      'label' => 'hidden',
      'settings' => [
        'image_style' => 'thumbnail',
        'image_link' => 'content',
      ],
    ]);
    $teaser_display->save();

    // Configure default display for page.
    $page_display = EntityViewDisplay::create([
      'id' => 'node.page.default',
      'targetEntityType' => 'node',
      'bundle' => 'page',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $page_display->setComponent('field_body', [
      'type' => 'text_default',
      'weight' => 1,
      'label' => 'hidden',
      'settings' => [],
    ]);
    $page_display->setComponent('field_date', [
      'type' => 'datetime_default',
      'weight' => 2,
      'label' => 'above',
      'settings' => [
        'format_type' => 'medium',
      ],
    ]);
    $page_display->save();

    // Configure default display for blog (minimal configuration).
    $blog_display = EntityViewDisplay::create([
      'id' => 'node.blog.default',
      'targetEntityType' => 'node',
      'bundle' => 'blog',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $blog_display->setComponent('field_content', [
      'type' => 'text_default',
      'weight' => 1,
      'label' => 'above',
      'settings' => [],
    ]);
    $blog_display->save();
  }

  /**
   * Sets up test taxonomy vocabulary and terms.
   *
   * Creates taxonomy vocabulary with fields for testing taxonomy display modes.
   */
  protected function setUpTestTaxonomy(): void {
    // Create tags vocabulary.
    $vocabulary = Vocabulary::create([
      'vid' => 'tags',
      'name' => 'Tags',
      'description' => 'Test tags vocabulary',
    ]);
    $vocabulary->save();

    // Create categories vocabulary.
    $categories = Vocabulary::create([
      'vid' => 'categories',
      'name' => 'Categories',
      'description' => 'Test categories vocabulary',
    ]);
    $categories->save();

    // Add description field to tags vocabulary.
    $this->createTestField('field_description', 'taxonomy_term', 'tags', 'text_long');

    // Configure display for tags taxonomy.
    $taxonomy_display = EntityViewDisplay::create([
      'id' => 'taxonomy_term.tags.default',
      'targetEntityType' => 'taxonomy_term',
      'bundle' => 'tags',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $taxonomy_display->setComponent('field_description', [
      'type' => 'text_default',
      'weight' => 1,
      'label' => 'above',
      'settings' => [],
    ]);
    $taxonomy_display->save();

    // Configure display for categories taxonomy (no custom fields).
    $categories_display = EntityViewDisplay::create([
      'id' => 'taxonomy_term.categories.default',
      'targetEntityType' => 'taxonomy_term',
      'bundle' => 'categories',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $categories_display->save();
  }

  /**
   * Get plugin definition test.
   */
  public function testPluginDefinesAllOperations(): void {
    $definition = $this->plugin->getPluginDefinition();
    $operations = $definition['operations'];

    // Assert operations we can test (node and taxonomy) are defined.
    $testable_operations = [
      'node_display_mode',
      'taxonomy_display_mode',
    ];

    foreach ($testable_operations as $operation) {
      $this->assertArrayHasKey(
        $operation,
        $operations,
        "Plugin should define '{$operation}' operation."
      );
    }

    // Assert node operation metadata.
    $this->assertEquals(
      'Node display configurations',
      (string) $operations['node_display_mode']['label'],
      'Node operation should have correct label.'
    );
    $this->assertArrayHasKey(
      'dependencies',
      $operations['node_display_mode'],
      'Node operation should declare dependencies.'
    );
    $this->assertContains(
      'node',
      $operations['node_display_mode']['dependencies'],
      'Node operation should depend on node module.'
    );

    // Assert taxonomy operation metadata.
    $this->assertEquals(
      'Taxonomy display configurations',
      (string) $operations['taxonomy_display_mode']['label'],
      'Taxonomy operation should have correct label.'
    );
    $this->assertContains(
      'taxonomy',
      $operations['taxonomy_display_mode']['dependencies'],
      'Taxonomy operation should depend on taxonomy module.'
    );

    // Note: Other operations (paragraph_display_mode, media_display_mode,
    // block_display_mode) are defined in the plugin but require additional
    // modules that may not be installed in the test environment.
  }

  /**
   * Get data operation result.
   */
  public function testNodeDisplayModeOperationReturnsCompleteData(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');

    // Assert data structure has required keys.
    $this->assertIsArray($data, 'Node display mode operation should return array.');
    $this->assertArrayHasKey('summary_bundle_display', $data, 'Data should contain summary_bundle_display.');
    $this->assertArrayHasKey('summary_display_bundles', $data, 'Data should contain summary_display_bundles.');
    $this->assertArrayHasKey('main_table', $data, 'Data should contain main_table.');

    // Assert main_table structure.
    $this->assertArrayHasKey('headers', $data['main_table'], 'Main table should have headers.');
    $this->assertArrayHasKey('rows', $data['main_table'], 'Main table should have rows.');
    $this->assertArrayHasKey('computed', $data['main_table'], 'Main table should have computed fields.');
  }

  /**
   * Check display mode summary with all bundles.
   */
  public function testNodeDisplayModeSummaryIncludesAllBundles(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $summary = $data['summary_bundle_display'];

    // Assert content types with displays are included.
    // Note: Bundles without displays are excluded by the service.
    $this->assertNotEmpty($summary, 'Summary should contain bundle data.');

    // Verify at least one bundle with displays exists.
    $this->assertGreaterThan(0, count($summary), 'At least one bundle should have displays configured.');

    // Validate structure of first bundle in summary.
    $first_bundle = reset($summary);
    if ($first_bundle) {
      $this->assertIsArray($first_bundle, 'Bundle data should be array.');
      $this->assertArrayHasKey('label', $first_bundle, 'Bundle should have label.');
      $this->assertArrayHasKey('displays', $first_bundle, 'Bundle should have displays.');
      $this->assertArrayHasKey('count', $first_bundle, 'Bundle should have count.');

      // Verify types.
      $this->assertIsString($first_bundle['label'], 'Label should be string.');
      $this->assertIsString($first_bundle['displays'], 'Displays should be string.');
      $this->assertIsInt($first_bundle['count'], 'Count should be integer.');
      $this->assertGreaterThan(0, $first_bundle['count'], 'Count should be positive.');
    }
  }

  /**
   * Check display mode summary grouped by display.
   */
  public function testNodeDisplayModeSummaryGroupsByDisplay(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $summary = $data['summary_display_bundles'];

    // Assert default display is listed.
    $this->assertArrayHasKey('default', $summary, 'Summary should include default display.');

    $default_summary = $summary['default'];
    $this->assertEquals('default', $default_summary['display'], 'Display key should match.');
    $this->assertIsString($default_summary['bundles'], 'Bundles should be string list.');
    $this->assertIsInt($default_summary['count'], 'Bundle count should be integer.');

    // Assert multiple bundles use default display.
    $this->assertGreaterThanOrEqual(2, $default_summary['count'], 'Multiple bundles should use default display.');

    // Verify article is in the bundles list.
    $this->assertStringContainsString('article', $default_summary['bundles'], 'Article should use default display.');
  }

  /**
   * Display mode main table contains field data.
   */
  public function testNodeDisplayModeMainTableContainsFieldData(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $main_table = $data['main_table'];

    // Assert headers include label and fields columns.
    $headers = $main_table['headers'];
    $this->assertArrayHasKey('label', $headers, 'Headers should include label column.');
    $this->assertArrayHasKey('fields', $headers, 'Headers should include fields column.');

    // Assert headers include display mode columns.
    $this->assertArrayHasKey('default', $headers, 'Headers should include default display column.');

    // Assert rows contain data for each bundle.
    $rows = $main_table['rows'];
    $this->assertNotEmpty($rows, 'Main table should have rows.');
    $this->assertGreaterThanOrEqual(3, count($rows), 'Should have rows for article, page, and blog.');

    // Find article row and validate structure.
    $article_row = NULL;
    foreach ($rows as $row) {
      if (isset($row['label']) && strpos($row['label'], 'Article') !== FALSE) {
        $article_row = $row;
        break;
      }
    }

    $this->assertNotNull($article_row, 'Article row should exist in main table.');
    $this->assertArrayHasKey('fields', $article_row, 'Article row should have fields column.');
    $this->assertArrayHasKey('default', $article_row, 'Article row should have default display column.');
  }

  /**
   * Node display mode included field formatter details.
   */
  public function testNodeDisplayModeIncludesFieldFormatterDetails(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $main_table = $data['main_table'];
    $rows = $main_table['rows'];

    // Find article row to check formatter details.
    $article_row = NULL;
    foreach ($rows as $row) {
      if (isset($row['label']) && strpos($row['label'], 'Article') !== FALSE) {
        $article_row = $row;
        break;
      }
    }

    $this->assertNotNull($article_row, 'Article row must exist for formatter testing.');

    // Assert default display column contains formatter information.
    $this->assertArrayHasKey('default', $article_row, 'Article should have default display data.');

    $default_display = $article_row['default'];
    $this->assertNotEmpty($default_display, 'Default display should have content.');

    // Verify display data is FormattableMarkup (contains field and formatter info).
    $this->assertInstanceOf(
      'Drupal\Component\Render\FormattableMarkup',
      $default_display,
      'Display cell should be FormattableMarkup with field formatter details.'
    );

    // Convert to string to check content.
    $display_content = (string) $default_display;
    $this->assertNotEmpty($display_content, 'Display content should not be empty.');
  }

  /**
   * Node display mode includes tease display mode.
   */
  public function testNodeDisplayModeIncludesTeaserDisplayMode(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $main_table = $data['main_table'];
    $headers = $main_table['headers'];

    // Assert teaser display column exists.
    $this->assertArrayHasKey('teaser', $headers, 'Headers should include teaser display column.');

    // Find article row and verify teaser display data.
    $rows = $main_table['rows'];
    $article_row = NULL;
    foreach ($rows as $row) {
      if (isset($row['label']) && strpos($row['label'], 'Article') !== FALSE) {
        $article_row = $row;
        break;
      }
    }

    $this->assertNotNull($article_row, 'Article row should exist.');
    $this->assertArrayHasKey('teaser', $article_row, 'Article should have teaser display configuration.');

    $teaser_display = $article_row['teaser'];
    $this->assertNotEmpty($teaser_display, 'Teaser display should have content.');
  }

  /**
   * Node display mode identifies computed fields.
   */
  public function testNodeDisplayModeIdentifiesComputedFields(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $main_table = $data['main_table'];

    // Assert computed key exists.
    $this->assertArrayHasKey('computed', $main_table, 'Main table should track computed fields.');
    $this->assertIsString($main_table['computed'], 'Computed should be a string.');

    // Note: Computed fields are typically pseudo-fields like "links", "extra_field", etc.
    // The actual computed list depends on entity display configuration.
  }

  // ============================================================================
  // CATEGORY 4: TAXONOMY DISPLAY MODE OPERATION
  // TDD Phase: RED-GREEN-REFACTOR
  // Tests for taxonomy_display_mode operation.
  // ============================================================================

  /**
   *
   * RED: Test fails if taxonomy operation doesn't return data.
   * GREEN: Taxonomy display mode operation returns complete structure.
   * REFACTOR: Validate taxonomy-specific data structure.
   */
  public function testTaxonomyDisplayModeOperationReturnsData(): void {
    $data = $this->getDataWithRenderContext('taxonomy_display_mode');

    // Assert data structure.
    $this->assertIsArray($data, 'Taxonomy display mode operation should return array.');
    $this->assertArrayHasKey('summary_bundle_display', $data, 'Data should contain summary_bundle_display.');
    $this->assertArrayHasKey('summary_display_bundles', $data, 'Data should contain summary_display_bundles.');
    $this->assertArrayHasKey('main_table', $data, 'Data should contain main_table.');
  }

  /**
   *
   * RED: Test fails if taxonomy vocabularies are not included.
   * GREEN: All created vocabularies appear in results.
   * REFACTOR: Verify tags and categories vocabularies.
   */
  public function testTaxonomyDisplayModeIncludesAllVocabularies(): void {
    $data = $this->getDataWithRenderContext('taxonomy_display_mode');
    $summary = $data['summary_bundle_display'];

    // Assert vocabularies are included.
    $this->assertArrayHasKey('tags', $summary, 'Summary should include tags vocabulary.');
    $this->assertArrayHasKey('categories', $summary, 'Summary should include categories vocabulary.');

    // Validate tags vocabulary data.
    $tags_summary = $summary['tags'];
    $this->assertEquals('Tags', $tags_summary['label'], 'Tags label should match.');
    $this->assertGreaterThan(0, $tags_summary['count'], 'Tags should have at least one display mode.');
  }

  /**
   *
   * RED: Test fails if taxonomy fields are not included in display.
   * GREEN: Taxonomy term displays include configured fields.
   * REFACTOR: Verify field_description in tags vocabulary display.
   */
  public function testTaxonomyDisplayModeIncludesVocabularyFields(): void {
    $data = $this->getDataWithRenderContext('taxonomy_display_mode');
    $main_table = $data['main_table'];
    $rows = $main_table['rows'];

    // Find tags row.
    $tags_row = NULL;
    foreach ($rows as $row) {
      if (isset($row['label']) && strpos($row['label'], 'Tags') !== FALSE) {
        $tags_row = $row;
        break;
      }
    }

    $this->assertNotNull($tags_row, 'Tags row should exist in main table.');
    $this->assertArrayHasKey('fields', $tags_row, 'Tags row should have fields column.');

    // Verify fields content.
    $fields_content = $tags_row['fields'];
    $this->assertNotEmpty($fields_content, 'Tags should have field data.');
  }

  /**
   *
   * RED: Test fails if taxonomy displays use wrong entity type.
   * GREEN: Operation correctly uses taxonomy_vocabulary and taxonomy_term.
   * REFACTOR: Verify getData is called with correct parameters.
   */
  public function testTaxonomyDisplayModeUsesCorrectEntityType(): void {
    // This test verifies the operation calls getData with correct parameters.
    // We can't directly test the parameters, but we can verify the result
    // contains taxonomy data and not node data.
    $data = $this->getDataWithRenderContext('taxonomy_display_mode');
    $summary = $data['summary_bundle_display'];

    // Assert contains taxonomy bundles, not node bundles.
    $this->assertArrayHasKey('tags', $summary, 'Should contain taxonomy vocabulary.');
    $this->assertArrayNotHasKey('article', $summary, 'Should not contain node bundles.');
  }

  /**
   *
   * RED: Test fails if vocabulary without fields fails gracefully.
   * GREEN: Empty vocabularies are handled without errors.
   * REFACTOR: Verify categories vocabulary (no custom fields).
   */
  public function testTaxonomyDisplayModeHandlesVocabularyWithoutFields(): void {
    $data = $this->getDataWithRenderContext('taxonomy_display_mode');
    $summary = $data['summary_bundle_display'];

    // Assert categories vocabulary is present.
    $this->assertArrayHasKey('categories', $summary, 'Categories vocabulary should be included.');

    $categories_summary = $summary['categories'];
    $this->assertEquals('Categories', $categories_summary['label'], 'Categories label should match.');

    // Categories may have display modes even without custom fields.
    $this->assertGreaterThanOrEqual(0, $categories_summary['count'], 'Categories should handle no-field scenario.');
  }

  // ============================================================================
  // CATEGORY 5: DATA STRUCTURE VALIDATION
  // TDD Phase: RED-GREEN-REFACTOR
  // Tests that validate the exact structure of returned data.
  // ============================================================================

  /**
   *
   * RED: Test fails if summary_bundle_display structure is incorrect.
   * GREEN: Summary has exact required keys and value types.
   * REFACTOR: Validate every bundle summary entry structure.
   */
  public function testSummaryBundleDisplayStructureIsValid(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $summary = $data['summary_bundle_display'];

    $this->assertIsArray($summary, 'Summary bundle display should be array.');

    // Only validate if we have bundle data.
    if (!empty($summary)) {
      foreach ($summary as $bundle_id => $bundle_data) {
        $this->assertIsString($bundle_id, "Bundle ID should be string, got {$bundle_id}.");

        $this->assertArrayHasKey('label', $bundle_data, "Bundle {$bundle_id} should have label.");
        $this->assertIsString($bundle_data['label'], "Bundle {$bundle_id} label should be string.");

        $this->assertArrayHasKey('displays', $bundle_data, "Bundle {$bundle_id} should have displays.");
        $this->assertIsString($bundle_data['displays'], "Bundle {$bundle_id} displays should be string.");

        $this->assertArrayHasKey('count', $bundle_data, "Bundle {$bundle_id} should have count.");
        $this->assertIsInt($bundle_data['count'], "Bundle {$bundle_id} count should be integer.");
        $this->assertGreaterThanOrEqual(0, $bundle_data['count'], "Bundle {$bundle_id} count should be non-negative.");
      }
    }
  }

  /**
   *
   * RED: Test fails if summary_display_bundles structure is incorrect.
   * GREEN: Display summary has exact required keys and value types.
   * REFACTOR: Validate every display summary entry structure.
   */
  public function testSummaryDisplayBundlesStructureIsValid(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $summary = $data['summary_display_bundles'];

    $this->assertIsArray($summary, 'Summary display bundles should be array.');

    foreach ($summary as $display_id => $display_data) {
      $this->assertIsString($display_id, "Display ID should be string, got {$display_id}.");

      $this->assertArrayHasKey('display', $display_data, "Display {$display_id} should have display key.");
      $this->assertIsString($display_data['display'], "Display {$display_id} display value should be string.");
      $this->assertEquals($display_id, $display_data['display'], "Display {$display_id} display value should match key.");

      $this->assertArrayHasKey('bundles', $display_data, "Display {$display_id} should have bundles.");
      $this->assertIsString($display_data['bundles'], "Display {$display_id} bundles should be string.");

      $this->assertArrayHasKey('count', $display_data, "Display {$display_id} should have count.");
      $this->assertIsInt($display_data['count'], "Display {$display_id} count should be integer.");
      $this->assertGreaterThanOrEqual(0, $display_data['count'], "Display {$display_id} count should be non-negative.");
    }
  }

  /**
   *
   * RED: Test fails if main_table structure is incomplete.
   * GREEN: Main table has headers, rows, and computed keys with correct types.
   * REFACTOR: Validate nested structure of main table.
   */
  public function testMainTableStructureIsValid(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $main_table = $data['main_table'];

    // Validate top-level structure.
    $this->assertIsArray($main_table, 'Main table should be array.');
    $this->assertArrayHasKey('headers', $main_table, 'Main table should have headers.');
    $this->assertArrayHasKey('rows', $main_table, 'Main table should have rows.');
    $this->assertArrayHasKey('computed', $main_table, 'Main table should have computed.');

    // Validate headers.
    $headers = $main_table['headers'];
    $this->assertIsArray($headers, 'Headers should be array.');
    $this->assertArrayHasKey('label', $headers, 'Headers should include label.');
    $this->assertArrayHasKey('fields', $headers, 'Headers should include fields.');

    // Validate rows.
    $rows = $main_table['rows'];
    $this->assertIsArray($rows, 'Rows should be array.');

    foreach ($rows as $row_index => $row) {
      $this->assertIsArray($row, "Row {$row_index} should be array.");
      $this->assertArrayHasKey('label', $row, "Row {$row_index} should have label.");
      $this->assertArrayHasKey('fields', $row, "Row {$row_index} should have fields.");
    }

    // Validate computed.
    $this->assertIsString($main_table['computed'], 'Computed should be string.');
  }

  /**
   *
   * RED: Test fails if headers don't include display mode columns.
   * GREEN: Headers dynamically include all display modes found.
   * REFACTOR: Verify headers match available display modes.
   */
  public function testMainTableHeadersIncludeDisplayModes(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $headers = $data['main_table']['headers'];

    // Assert static columns.
    $this->assertArrayHasKey('label', $headers, 'Headers must include label column.');
    $this->assertArrayHasKey('fields', $headers, 'Headers must include fields column.');

    // Assert dynamic display mode columns.
    $this->assertArrayHasKey('default', $headers, 'Headers must include default display.');

    // Verify header values are human-readable labels.
    $this->assertEquals('Type', $headers['label'], 'Label header should be "Type".');
    $this->assertEquals('Fields', $headers['fields'], 'Fields header should be "Fields".');
    $this->assertEquals('Default', $headers['default'], 'Default display header should be "Default".');
  }

  /**
   *
   * RED: Test fails if rows have inconsistent structure.
   * GREEN: All rows have matching keys corresponding to headers.
   * REFACTOR: Verify row keys align with header keys.
   */
  public function testMainTableRowsMatchHeaderStructure(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $headers = $data['main_table']['headers'];
    $rows = $data['main_table']['rows'];

    foreach ($rows as $row_index => $row) {
      // Each row should have at minimum: label and fields.
      $this->assertArrayHasKey('label', $row, "Row {$row_index} should have label key.");
      $this->assertArrayHasKey('fields', $row, "Row {$row_index} should have fields key.");

      // Row should have keys for display modes that apply to this bundle.
      // Not all bundles have all display modes, so we check keys exist in headers.
      foreach (array_keys($row) as $row_key) {
        if (!in_array($row_key, ['label', 'fields'])) {
          $this->assertArrayHasKey(
            $row_key,
            $headers,
            "Row {$row_index} has key '{$row_key}' which should exist in headers."
          );
        }
      }
    }
  }

  // ============================================================================
  // CATEGORY 6: EDGE CASES AND ERROR CONDITIONS
  // TDD Phase: RED-GREEN-REFACTOR
  // Tests that verify robust handling of unusual scenarios.
  // ============================================================================

  /**
   *
   * RED: Test fails if invalid operation causes error.
   * GREEN: Invalid operation returns empty array gracefully.
   * REFACTOR: Verify defensive programming for operation parameter.
   */
  public function testInvalidOperationReturnsEmptyArray(): void {
    $data = $this->getDataWithRenderContext('invalid_operation');

    $this->assertIsArray($data, 'Invalid operation should return array.');
    $this->assertEmpty($data, 'Invalid operation should return empty array.');
  }

  /**
   *
   * RED: Test fails if empty operation parameter causes error.
   * GREEN: Empty operation returns empty array.
   * REFACTOR: Verify default case in switch statement.
   */
  public function testEmptyOperationReturnsEmptyArray(): void {
    $data = $this->getDataWithRenderContext('');

    $this->assertIsArray($data, 'Empty operation should return array.');
    $this->assertEmpty($data, 'Empty operation should return empty array.');
  }

  /**
   *
   * RED: Test fails if system with no content types fails.
   * GREEN: Operation handles no bundles gracefully.
   * REFACTOR: Test with fresh install (no custom content types).
   */
  public function testOperationHandlesNoBundlesGracefully(): void {
    // Note: Our setUp creates content types, so we'd need to test with clean state.
    // For now, we verify the data structure is valid even with bundles.
    // In a real scenario, you'd create a separate test with no setUp content types.
    $data = $this->getDataWithRenderContext('node_display_mode');

    // Even with bundles, structure should be valid.
    $this->assertIsArray($data, 'Operation should return array even with empty bundles.');
    $this->assertArrayHasKey('summary_bundle_display', $data, 'Should have summary structure.');
  }

  /**
   *
   * RED: Test fails if content type without fields breaks operation.
   * GREEN: Bundles without custom fields are handled correctly.
   * REFACTOR: Verify empty field list scenario.
   */
  public function testOperationHandlesBundleWithoutCustomFields(): void {
    // Create a content type with no custom fields.
    $this->createContentType('minimal', 'Minimal Type');

    // Configure minimal display (no fields).
    $minimal_display = EntityViewDisplay::create([
      'id' => 'node.minimal.default',
      'targetEntityType' => 'node',
      'bundle' => 'minimal',
      'mode' => 'default',
      'status' => TRUE,
    ]);
    $minimal_display->save();

    // Execute operation.
    $data = $this->getDataWithRenderContext('node_display_mode');
    $summary = $data['summary_bundle_display'];

    // Assert minimal type is included (if display was properly configured).
    if (isset($summary['minimal'])) {
      $minimal_summary = $summary['minimal'];
      $this->assertEquals('Minimal Type', $minimal_summary['label'], 'Minimal label should match.');
    }
    else {
      // Verify that at least operation completes without errors.
      $this->assertIsArray($summary, 'Summary should be an array even with minimal bundle.');
    }
  }

  /**
   *
   * RED: Test fails if display with hidden fields causes issues.
   * GREEN: Hidden fields are properly handled in display analysis.
   * REFACTOR: Create field explicitly set to hidden.
   */
  public function testOperationHandlesHiddenFields(): void {
    // Create field and set to hidden in display.
    $this->createTestField('field_hidden', 'node', 'article', 'text');

    $display = EntityViewDisplay::load('node.article.default');
    $display->removeComponent('field_hidden');
    $display->save();

    // Execute operation.
    $data = $this->getDataWithRenderContext('node_display_mode');

    // Operation should complete successfully.
    $this->assertIsArray($data, 'Operation should handle hidden fields.');
    $this->assertArrayHasKey('main_table', $data, 'Main table should be present.');
  }

  /**
   *
   * RED: Test fails if multiple content types with same display modes fail.
   * GREEN: Multiple bundles sharing display modes are correctly aggregated.
   * REFACTOR: Verify display_bundles summary aggregates correctly.
   */
  public function testOperationAggregatesMultipleBundlesPerDisplay(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $display_summary = $data['summary_display_bundles'];

    // Default display should be used by multiple bundles.
    $this->assertArrayHasKey('default', $display_summary, 'Default display should exist.');

    $default_data = $display_summary['default'];
    $bundles_list = $default_data['bundles'];

    // Should contain multiple bundle names.
    $this->assertGreaterThan(1, substr_count($bundles_list, ','), 'Default display should have multiple bundles.');
  }

  // ============================================================================
  // CATEGORY 7: BUILD DATA RENDER ARRAY
  // TDD Phase: RED-GREEN-REFACTOR
  // Tests for buildDataRenderArray method that creates renderable output.
  // ============================================================================

  /**
   *
   * RED: Test fails if render array structure is invalid.
   * GREEN: Render array contains all required elements.
   * REFACTOR: Validate complete render array structure.
   */
  public function testBuildDataRenderArrayReturnsValidStructure(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $build = $this->plugin->buildDataRenderArray($data, 'node_display_mode');

    // Assert render array is array.
    $this->assertIsArray($build, 'Build should be array.');

    // Assert main components exist.
    $this->assertArrayHasKey('summary_table_introduction', $build, 'Should have summary introduction.');
    $this->assertArrayHasKey('summary_bundle_display', $build, 'Should have summary bundle display table.');
    $this->assertArrayHasKey('summary_display_bundles', $build, 'Should have summary display bundles table.');
    $this->assertArrayHasKey('main_table_introduction', $build, 'Should have main table introduction.');
    $this->assertArrayHasKey('main_table', $build, 'Should have main table.');
  }

  /**
   *
   * RED: Test fails if summary tables don't use table theme.
   * GREEN: Summary tables are renderable with correct theme.
   * REFACTOR: Verify table render element structure.
   */
  public function testBuildDataRenderArraySummaryTablesUseTableTheme(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $build = $this->plugin->buildDataRenderArray($data, 'node_display_mode');

    // Verify summary_bundle_display table.
    $bundle_table = $build['summary_bundle_display'];
    $this->assertArrayHasKey('#theme', $bundle_table, 'Bundle table should have theme.');
    $this->assertEquals('table', $bundle_table['#theme'], 'Bundle table should use table theme.');
    $this->assertArrayHasKey('#header', $bundle_table, 'Bundle table should have header.');
    $this->assertArrayHasKey('#rows', $bundle_table, 'Bundle table should have rows.');

    // Verify summary_display_bundles table.
    $display_table = $build['summary_display_bundles'];
    $this->assertArrayHasKey('#theme', $display_table, 'Display table should have theme.');
    $this->assertEquals('table', $display_table['#theme'], 'Display table should use table theme.');
    $this->assertArrayHasKey('#header', $display_table, 'Display table should have header.');
    $this->assertArrayHasKey('#rows', $display_table, 'Display table should have rows.');
  }

  /**
   *
   * RED: Test fails if main table doesn't include proper headers.
   * GREEN: Main table has all headers from data structure.
   * REFACTOR: Verify header array matches data headers.
   */
  public function testBuildDataRenderArrayMainTableHasHeaders(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $build = $this->plugin->buildDataRenderArray($data, 'node_display_mode');

    $main_table = $build['main_table'];
    $this->assertArrayHasKey('#header', $main_table, 'Main table should have headers.');

    $headers = $main_table['#header'];
    $this->assertIsArray($headers, 'Headers should be array.');

    // Verify expected headers.
    $header_values = array_values($headers);
    $this->assertContains('Type', $header_values, 'Headers should include Type.');
    $this->assertContains('Fields', $header_values, 'Headers should include Fields.');
    $this->assertContains('Default', $header_values, 'Headers should include Default display.');
  }

  /**
   *
   * RED: Test fails if main table rows aren't properly structured.
   * GREEN: Main table rows have correct structure with data and class.
   * REFACTOR: Verify row array structure for rendering.
   */
  public function testBuildDataRenderArrayMainTableRowsAreStructured(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $build = $this->plugin->buildDataRenderArray($data, 'node_display_mode');

    $main_table = $build['main_table'];
    $this->assertArrayHasKey('#rows', $main_table, 'Main table should have rows.');

    $rows = $main_table['#rows'];
    $this->assertIsArray($rows, 'Rows should be array.');
    $this->assertNotEmpty($rows, 'Rows should not be empty.');

    // Verify first row structure.
    $first_row = reset($rows);
    $this->assertArrayHasKey('data', $first_row, 'Row should have data key.');
    $this->assertArrayHasKey('class', $first_row, 'Row should have class key.');
    $this->assertContains('xray-audit__row', $first_row['class'], 'Row should have xray-audit__row class.');

    // Verify row data structure.
    $row_data = $first_row['data'];
    $this->assertIsArray($row_data, 'Row data should be array.');
  }

  /**
   *
   * RED: Test fails if main table doesn't have required CSS classes.
   * GREEN: Main table has xray_audit CSS classes for styling.
   * REFACTOR: Verify attributes array structure.
   */
  public function testBuildDataRenderArrayMainTableHasRequiredClasses(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $build = $this->plugin->buildDataRenderArray($data, 'node_display_mode');

    $main_table = $build['main_table'];
    $this->assertArrayHasKey('#attributes', $main_table, 'Main table should have attributes.');

    $attributes = $main_table['#attributes'];
    $this->assertArrayHasKey('class', $attributes, 'Attributes should have class key.');

    $classes = $attributes['class'];
    $this->assertContains('xray-audit__table', $classes, 'Should have xray-audit__table class.');
    $this->assertContains('xray-audit__sticky-header', $classes, 'Should have sticky-header class.');
  }

  /**
   *
   * RED: Test fails if library attachment is missing.
   * GREEN: Main table attaches xray_audit JavaScript library.
   * REFACTOR: Verify attached libraries structure.
   */
  public function testBuildDataRenderArrayAttachesLibrary(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $build = $this->plugin->buildDataRenderArray($data, 'node_display_mode');

    $main_table = $build['main_table'];
    $this->assertArrayHasKey('#attached', $main_table, 'Main table should attach libraries.');

    $attached = $main_table['#attached'];
    $this->assertArrayHasKey('library', $attached, 'Attached should have library key.');
    $this->assertContains('xray_audit/xray_audit', $attached['library'], 'Should attach xray_audit library.');
  }

  /**
   *
   * RED: Test fails if computed fields markup is not included.
   * GREEN: Computed fields are displayed when present.
   * REFACTOR: Verify computed field markup element.
   */
  public function testBuildDataRenderArrayIncludesComputedFieldsWhenPresent(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');

    // Add computed field to data to ensure it appears.
    $data['main_table']['computed'] = 'test_computed_field';

    $build = $this->plugin->buildDataRenderArray($data, 'node_display_mode');

    // If computed fields exist, should have computed element.
    $this->assertArrayHasKey('computed', $build, 'Should have computed element when computed fields exist.');

    $computed = $build['computed'];
    $this->assertArrayHasKey('#markup', $computed, 'Computed should have markup.');
    $this->assertStringContainsString('test_computed_field', (string) $computed['#markup'], 'Markup should contain computed field name.');
  }

  /**
   *
   * RED: Test fails if weights are not set for proper ordering.
   * GREEN: All render elements have weights for correct display order.
   * REFACTOR: Verify weight ordering creates logical flow.
   */
  public function testBuildDataRenderArrayElementsHaveWeights(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $build = $this->plugin->buildDataRenderArray($data, 'node_display_mode');

    // Verify key elements have weights.
    $this->assertArrayHasKey('#weight', $build['summary_table_introduction'], 'Summary intro should have weight.');
    $this->assertArrayHasKey('#weight', $build['summary_bundle_display'], 'Summary bundle table should have weight.');
    $this->assertArrayHasKey('#weight', $build['summary_display_bundles'], 'Summary display table should have weight.');
    $this->assertArrayHasKey('#weight', $build['main_table_introduction'], 'Main table intro should have weight.');
    $this->assertArrayHasKey('#weight', $build['main_table'], 'Main table should have weight.');

    // Verify logical weight ordering.
    $this->assertLessThan(
      $build['main_table']['#weight'],
      $build['summary_table_introduction']['#weight'],
      'Summary should come before main table.'
    );

    // Verify CSV download button weight if it exists.
    if (isset($build['download_button'])) {
      $this->assertArrayHasKey('#weight', $build['download_button'], 'Download button should have weight.');
      $this->assertGreaterThan(
        $build['main_table']['#weight'],
        $build['download_button']['#weight'],
        'Download button should come after main table.'
      );
      if (isset($build['computed'])) {
        $this->assertGreaterThan(
          $build['computed']['#weight'],
          $build['download_button']['#weight'],
          'Download button should come after computed fields.'
        );
      }
    }
  }

  /**
   *
   * RED: Test fails if CSV download button is not positioned correctly.
   * GREEN: CSV download button appears after Field Display Configuration table.
   * REFACTOR: Verify button position matches user expectations.
   */
  public function testCsvDownloadButtonPositionedAfterMainTable(): void {
    $data = $this->getDataWithRenderContext('node_display_mode');
    $build = $this->plugin->buildDataRenderArray($data, 'node_display_mode');

    // Assert download button exists (operation has download = TRUE).
    $this->assertArrayHasKey('download_button', $build, 'Download button should exist for node_display_mode operation.');

    // Assert button is positioned after main elements.
    $button_weight = $build['download_button']['#weight'];
    $this->assertEquals(40, $button_weight, 'Download button weight should be 40.');

    // Verify button comes after all main content.
    $this->assertGreaterThan($build['summary_table_introduction']['#weight'], $button_weight, 'Button after summary intro.');
    $this->assertGreaterThan($build['summary_bundle_display']['#weight'], $button_weight, 'Button after summary bundle table.');
    $this->assertGreaterThan($build['summary_display_bundles']['#weight'], $button_weight, 'Button after summary display table.');
    $this->assertGreaterThan($build['main_table_introduction']['#weight'], $button_weight, 'Button after main table intro.');
    $this->assertGreaterThan($build['main_table']['#weight'], $button_weight, 'Button after main table.');

    if (isset($build['computed'])) {
      $this->assertGreaterThan($build['computed']['#weight'], $button_weight, 'Button after computed fields.');
    }

    // Verify button has wrapper div for styling.
    $this->assertArrayHasKey('#prefix', $build['download_button'], 'Button should have prefix for wrapper div.');
    $this->assertArrayHasKey('#suffix', $build['download_button'], 'Button should have suffix for wrapper div.');
    $this->assertStringContainsString('xray-audit__download-section', $build['download_button']['#prefix'], 'Button should have download section class.');
  }

  /**
   *
   * RED: Test fails if empty data causes render errors.
   * GREEN: Empty data is handled gracefully in render array.
   * REFACTOR: Verify defensive programming for edge cases.
   */
  public function testBuildDataRenderArrayHandlesEmptyData(): void {
    $empty_data = [
      'summary_bundle_display' => [],
      'summary_display_bundles' => [],
      'main_table' => [
        'headers' => [],
        'rows' => [],
        'computed' => '',
      ],
    ];

    $build = $this->plugin->buildDataRenderArray($empty_data, 'node_display_mode');

    // Should return valid render array even with empty data.
    $this->assertIsArray($build, 'Should return array for empty data.');
    $this->assertArrayHasKey('main_table', $build, 'Should have main table element.');
  }

  // ============================================================================
  // CATEGORY 8: INTEGRATION AND COMPREHENSIVE SCENARIOS
  // TDD Phase: RED-GREEN-REFACTOR
  // Tests that verify complete workflows and realistic usage patterns.
  // ============================================================================

  /**
   *
   * RED: Test fails if end-to-end workflow has issues.
   * GREEN: Complete workflow from data retrieval to render works.
   * REFACTOR: Verify realistic plugin usage scenario.
   */
  public function testCompleteWorkflowFromDataToRender(): void {
    // Execute operation to get data.
    $data = $this->getDataWithRenderContext('node_display_mode');

    $this->assertIsArray($data, 'Data retrieval should succeed.');
    $this->assertNotEmpty($data, 'Data should not be empty.');

    // Build render array from data.
    $build = $this->plugin->buildDataRenderArray($data, 'node_display_mode');

    $this->assertIsArray($build, 'Render array should be valid.');
    $this->assertNotEmpty($build, 'Render array should have elements.');

    // Verify render array is actually renderable (has render elements).
    $this->assertTrue(
      isset($build['main_table']['#theme']),
      'Render array should contain renderable elements.'
    );
  }

  /**
   *
   * RED: Test fails if switching between operations causes state issues.
   * GREEN: Plugin handles multiple operation calls correctly.
   * REFACTOR: Verify plugin is stateless between operations.
   */
  public function testPluginHandlesMultipleOperationCallsCorrectly(): void {
    // Call node operation.
    $node_data = $this->getDataWithRenderContext('node_display_mode');
    $this->assertArrayHasKey('article', $node_data['summary_bundle_display'], 'Node data should have article.');

    // Call taxonomy operation.
    $taxonomy_data = $this->getDataWithRenderContext('taxonomy_display_mode');
    $this->assertArrayHasKey('tags', $taxonomy_data['summary_bundle_display'], 'Taxonomy data should have tags.');

    // Call node operation again - should not have taxonomy data.
    $node_data_again = $this->getDataWithRenderContext('node_display_mode');
    $this->assertArrayHasKey('article', $node_data_again['summary_bundle_display'], 'Second node call should still have article.');
    $this->assertArrayNotHasKey('tags', $node_data_again['summary_bundle_display'], 'Second node call should not have tags.');
  }

  /**
   * Integration test: Verify plugin works with real entity display service.
   *
   * This test validates that the plugin correctly integrates with the
   * EntityDisplayArchitecture service to retrieve and format display data.
   *
   * RED: Test fails if service integration is broken.
   * GREEN: Plugin successfully uses service to retrieve data.
   * REFACTOR: Verify service method calls and data flow.
   */
  public function testPluginIntegrationWithEntityDisplayService(): void {
    // This is an integration test - we verify the plugin uses the real service.
    $data = $this->getDataWithRenderContext('node_display_mode');

    // Verify data comes from EntityDisplayArchitecture service.
    // The service should provide summary and main table data.
    $this->assertArrayHasKey('summary_bundle_display', $data, 'Service should provide bundle summary.');
    $this->assertArrayHasKey('summary_display_bundles', $data, 'Service should provide display summary.');
    $this->assertArrayHasKey('main_table', $data, 'Service should provide main table data.');

    // Verify data quality indicates service processing.
    $main_table = $data['main_table'];
    $this->assertArrayHasKey('headers', $main_table, 'Service should structure headers.');
    $this->assertArrayHasKey('rows', $main_table, 'Service should structure rows.');
  }

}
