<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentMetric;

use Drupal\block_content\Entity\BlockContent;
use Drupal\block_content\Entity\BlockContentType;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Comprehensive tests for XrayAuditQueryTaskBlockPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditQueryTaskBlockPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'block',
    'block_content',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The block plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentMetric\XrayAuditQueryTaskBlockPlugin
   */
  protected $plugin;

  /**
   * Test block content types created for testing.
   *
   * @var \Drupal\block_content\Entity\BlockContentType[]
   */
  protected $testBlockTypes = [];

  /**
   * Test block content entities created for testing.
   *
   * @var \Drupal\block_content\Entity\BlockContent[]
   */
  protected $testBlocks = [];

  /**
   * Plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install block_content entity schema and configuration.
    $this->installEntitySchema('block_content');
    $this->installConfig(['block_content']);

    // Get task plugin manager.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');

    // Get plugin repository service for cache testing.
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('queries_data_blocks');

    // Create test block types and blocks.
    $this->createTestBlockTypes();
    $this->createTestBlocks();
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clean up test blocks.
    foreach ($this->testBlocks as $block) {
      if ($block) {
        $block->delete();
      }
    }

    // Clean up test block types.
    foreach ($this->testBlockTypes as $type) {
      if ($type) {
        $type->delete();
      }
    }

    // Clear all cache.
    $this->pluginRepository->clearAllCache();

    parent::tearDown();
  }

  /**
   * Creates test block content types.
   */
  protected function createTestBlockTypes() {
    // Create basic block type.
    $basic = BlockContentType::create([
      'id' => 'basic',
      'label' => 'Basic block',
      'revision' => FALSE,
    ]);
    $basic->save();
    $this->testBlockTypes['basic'] = $basic;

    // Create custom block type.
    $custom = BlockContentType::create([
      'id' => 'custom',
      'label' => 'Custom block',
      'revision' => FALSE,
    ]);
    $custom->save();
    $this->testBlockTypes['custom'] = $custom;

    // Create footer block type.
    $footer = BlockContentType::create([
      'id' => 'footer',
      'label' => 'Footer block',
      'revision' => FALSE,
    ]);
    $footer->save();
    $this->testBlockTypes['footer'] = $footer;
  }

  /**
   * Creates test block content entities with various types and languages.
   */
  protected function createTestBlocks() {
    // Create 3 basic blocks in English.
    for ($i = 1; $i <= 3; $i++) {
      $block = BlockContent::create([
        'type' => 'basic',
        'info' => "Basic block $i EN",
        'langcode' => 'en',
      ]);
      $block->save();
      $this->testBlocks[] = $block;
    }

    // Create 2 custom blocks in English.
    for ($i = 1; $i <= 2; $i++) {
      $block = BlockContent::create([
        'type' => 'custom',
        'info' => "Custom block $i EN",
        'langcode' => 'en',
      ]);
      $block->save();
      $this->testBlocks[] = $block;
    }

    // Create 1 custom block in Spanish.
    $block = BlockContent::create([
      'type' => 'custom',
      'info' => 'Custom block ES',
      'langcode' => 'es',
    ]);
    $block->save();
    $this->testBlocks[] = $block;

    // Create 1 footer block in French.
    $block = BlockContent::create([
      'type' => 'footer',
      'info' => 'Footer block FR',
      'langcode' => 'fr',
    ]);
    $block->save();
    $this->testBlocks[] = $block;

    // Create 2 basic blocks in Spanish.
    for ($i = 1; $i <= 2; $i++) {
      $block = BlockContent::create([
        'type' => 'basic',
        'info' => "Basic block $i ES",
        'langcode' => 'es',
      ]);
      $block->save();
      $this->testBlocks[] = $block;
    }
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  // ---------------------------------------------------------------------------
  // getDataOperationResult()
  // ---------------------------------------------------------------------------

  /**
   * Tests getDataOperationResult() with 'block_count_type' operation.
   */
  public function testGetDataOperationResultBlockCountType() {
    // Act.
    $result = $this->plugin->getDataOperationResult('block_count_type');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure (Type, Label, Total).
    $this->assertCount(3, $result['header_table']);

    // Verify results contain block data.
    $this->assertNotEmpty($result['results_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests getDataOperationResult() with 'block_count_type_langcode' operation.
   */
  public function testGetDataOperationResultBlockCountTypeLangcode() {
    // Act.
    $result = $this->plugin->getDataOperationResult('block_count_type_langcode');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure (Type, Label, Langcode, Total).
    $this->assertCount(4, $result['header_table']);

    // Verify results contain block data grouped by language.
    $this->assertNotEmpty($result['results_table']);
    $this->assertIsArray($result['results_table']);
  }

  // ---------------------------------------------------------------------------
  // blocksCountByType()
  // ---------------------------------------------------------------------------

  /**
   * Tests block_count_type operation returns proper structure.
   */
  public function testBlocksCountByTypeReturnsProperStructure() {
    // Act.
    $result = $this->plugin->getDataOperationResult('block_count_type');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure.
    $this->assertCount(3, $result['header_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests block_count_type operation includes all block types.
   */
  public function testBlocksCountByTypeIncludesAllBlockTypes() {
    // Act.
    $result = $this->plugin->getDataOperationResult('block_count_type');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Extract type IDs from results.
    $type_ids = array_column($results_table, 0);

    // We created 3 types: basic, custom, footer.
    $this->assertContains('basic', $type_ids, 'Results should include basic type');
    $this->assertContains('custom', $type_ids, 'Results should include custom type');
    $this->assertContains('footer', $type_ids, 'Results should include footer type');
  }

  /**
   * Tests blocksCountByType() counts correctly.
   */
  public function testBlocksCountByTypeCountsCorrectly() {
    // Act.
    $result = $this->plugin->getDataOperationResult('block_count_type');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Build associative array of type => count (excluding TOTAL row).
    $counts = [];
    foreach ($results_table as $row) {
      $first_element = (string) $row[0];
      if (strpos($first_element, 'TOTAL') !== FALSE) {
        break;
      }
      $counts[$row[0]] = $row[2];
    }

    // We created:
    // - 5 basic blocks (3 EN + 2 ES)
    // - 3 custom blocks (2 EN + 1 ES)
    // - 1 footer block (1 FR)
    $this->assertEquals(5, $counts['basic'], 'Basic type should have 5 blocks');
    $this->assertEquals(3, $counts['custom'], 'Custom type should have 3 blocks');
    $this->assertEquals(1, $counts['footer'], 'Footer type should have 1 block');
  }

  /**
   * Tests blocksCountByType() results are sorted by count (descending).
   */
  public function testBlocksCountByTypeSortedByCount() {
    // Act.
    $result = $this->plugin->getDataOperationResult('block_count_type');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Extract counts (excluding TOTAL row).
    $counts = [];
    foreach ($results_table as $row) {
      $first_element = (string) $row[0];
      if (strpos($first_element, 'TOTAL') !== FALSE) {
        break;
      }
      $counts[] = $row[2];
    }

    // Verify results are sorted by count (descending).
    $sorted_counts = $counts;
    rsort($sorted_counts, SORT_NUMERIC);
    $this->assertEquals($sorted_counts, $counts, 'Results should be sorted by count in descending order');
  }

  // ---------------------------------------------------------------------------
  // blocksCountByTypeAndLanguage()
  // ---------------------------------------------------------------------------

  /**
   * Tests blocksCountByTypeAndLanguage() returns proper structure.
   */
  public function testBlocksCountByTypeAndLanguageReturnsProperStructure() {
    // Act.
    $result = $this->plugin->getDataOperationResult('block_count_type_langcode');

    // Assert.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure (Type, Label, Langcode, Total).
    $this->assertCount(4, $result['header_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests blocksCountByTypeAndLanguage() groups by language.
   */
  public function testBlocksCountByTypeAndLanguageGroupsByLanguage() {
    // Act.
    $result = $this->plugin->getDataOperationResult('block_count_type_langcode');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Each row should have 4 elements: [type, label, langcode, count].
    foreach ($results_table as $row) {
      $this->assertCount(4, $row, 'Each row should have 4 elements');
      $this->assertIsString($row[2], 'Third element should be langcode string');
    }

    // Verify we have multiple language codes.
    $langcodes = array_unique(array_column($results_table, 2));
    $this->assertGreaterThan(1, count($langcodes), 'Should have multiple language codes');
    $this->assertContains('en', $langcodes, 'Should include English blocks');
    $this->assertContains('es', $langcodes, 'Should include Spanish blocks');
  }

  /**
   * Tests blocksCountByTypeAndLanguage() counts correctly per language.
   */
  public function testBlocksCountByTypeAndLanguageCountsCorrectly() {
    // Act.
    $result = $this->plugin->getDataOperationResult('block_count_type_langcode');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Build associative array of type+langcode => count.
    $counts = [];
    foreach ($results_table as $row) {
      $key = $row[0] . ':' . $row[2];
      $counts[$key] = $row[3];
    }

    // We created:
    // - basic:en = 3
    // - basic:es = 2
    // - custom:en = 2
    // - custom:es = 1
    // - footer:fr = 1.
    $this->assertEquals(3, $counts['basic:en'], 'Basic EN should have 3 blocks');
    $this->assertEquals(2, $counts['basic:es'], 'Basic ES should have 2 blocks');
    $this->assertEquals(2, $counts['custom:en'], 'Custom EN should have 2 blocks');
    $this->assertEquals(1, $counts['custom:es'], 'Custom ES should have 1 block');
    $this->assertEquals(1, $counts['footer:fr'], 'Footer FR should have 1 block');
  }

  /**
   * Tests blocksCountByTypeAndLanguage() handles multiple languages.
   */
  public function testBlocksCountByTypeAndLanguageHandlesMultipleLanguages() {
    // Act.
    $result = $this->plugin->getDataOperationResult('block_count_type_langcode');

    // Assert.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Extract unique langcodes.
    $langcodes = array_unique(array_column($results_table, 2));

    // We created blocks in 3 languages: en, es, fr.
    $this->assertCount(3, $langcodes, 'Should have exactly 3 languages');
    $this->assertContains('en', $langcodes);
    $this->assertContains('es', $langcodes);
    $this->assertContains('fr', $langcodes);
  }

}
