<?php

namespace Drupal\Tests\xray_audit\Kernel\Plugin\Tasks\ContentModel;

use Drupal\block_content\Entity\BlockContentType;
use Drupal\node\Entity\NodeType;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;

/**
 * Comprehensive tests for XrayAuditQueryTaskContentEntitiesPlugin.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class XrayAuditQueryTaskContentEntitiesPluginTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'field',
    'text',
    'filter',
    'taxonomy',
    'block_content',
    'xray_audit',
  ];

  /**
   * The task plugin manager.
   *
   * @var \Drupal\xray_audit\Plugin\XrayAuditTaskPluginManager
   */
  protected $taskPluginManager;

  /**
   * The content entities plugin instance.
   *
   * @var \Drupal\xray_audit\Plugin\xray_audit\tasks\ContentModel\XrayAuditQueryTaskContentEntitiesPlugin
   */
  protected $plugin;

  /**
   * Plugin repository service.
   *
   * @var \Drupal\xray_audit\Services\PluginRepositoryInterface
   */
  protected $pluginRepository;

  /**
   * Entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Test node types created for testing.
   *
   * @var \Drupal\node\Entity\NodeType[]
   */
  protected $testNodeTypes = [];

  /**
   * Test vocabularies created for testing.
   *
   * @var \Drupal\taxonomy\Entity\Vocabulary[]
   */
  protected $testVocabularies = [];

  /**
   * Test block content types created for testing.
   *
   * @var \Drupal\block_content\Entity\BlockContentType[]
   */
  protected $testBlockContentTypes = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install required entity schemas for content entity types.
    $this->installEntitySchema('node');
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('block_content');
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['node', 'filter', 'system', 'taxonomy', 'block_content']);

    // Get task plugin manager.
    $this->taskPluginManager = $this->container->get('plugin_manager.xray_audit_task');

    // Get plugin repository service.
    $this->pluginRepository = $this->container->get('xray_audit.plugin_repository');

    // Get entity type manager.
    $this->entityTypeManager = $this->container->get('entity_type.manager');

    // Create plugin instance.
    $this->plugin = $this->taskPluginManager->createInstance('content_entities');

    // Create test entity bundles.
    $this->createTestEntityBundles();
  }

  /**
   * {@inheritdoc}
   *
   * Clean up all test data to ensure test isolation.
   */
  protected function tearDown(): void {
    // Clean up test node types.
    foreach ($this->testNodeTypes as $type) {
      if ($type) {
        $type->delete();
      }
    }

    // Clean up test vocabularies.
    foreach ($this->testVocabularies as $vocabulary) {
      if ($vocabulary) {
        $vocabulary->delete();
      }
    }

    // Clean up test block content types.
    foreach ($this->testBlockContentTypes as $type) {
      if ($type) {
        $type->delete();
      }
    }

    // Clear all cache.
    $this->pluginRepository->clearAllCache();

    parent::tearDown();
  }

  /**
   * Creates test entity bundles for various content entity types.
   */
  protected function createTestEntityBundles(): void {
    // Create node types (2 types).
    $article = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $article->save();
    $this->testNodeTypes['article'] = $article;

    $page = NodeType::create([
      'type' => 'page',
      'name' => 'Page',
    ]);
    $page->save();
    $this->testNodeTypes['page'] = $page;

    // Create vocabularies (2 vocabularies).
    $tags = Vocabulary::create([
      'vid' => 'tags',
      'name' => 'Tags',
    ]);
    $tags->save();
    $this->testVocabularies['tags'] = $tags;

    $categories = Vocabulary::create([
      'vid' => 'categories',
      'name' => 'Categories',
    ]);
    $categories->save();
    $this->testVocabularies['categories'] = $categories;

    // Create block content types (2 types).
    $basic_block = BlockContentType::create([
      'id' => 'basic',
      'label' => 'Basic block',
    ]);
    $basic_block->save();
    $this->testBlockContentTypes['basic'] = $basic_block;

    $custom_block = BlockContentType::create([
      'id' => 'custom',
      'label' => 'Custom block',
    ]);
    $custom_block->save();
    $this->testBlockContentTypes['custom'] = $custom_block;
  }

  /**
   * Helper method to invoke protected methods using reflection.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name.
   * @param array $parameters
   *   The method parameters.
   *
   * @return mixed
   *   The method return value.
   */
  protected function invokeProtectedMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass($object);
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

  /**
   * Tests getDataOperationResult() with 'content_entity_types' operation.
   */
  public function testGetDataOperationResultContentEntityTypes(): void {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_types');
    // Assert: Verify basic structure.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure (id, Group, Provider, Class).
    $this->assertCount(4, $result['header_table']);

    // Verify results contain entity data.
    $this->assertNotEmpty($result['results_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests getDataOperationResult() with invalid operation returns empty.
   */
  public function testGetDataOperationResultInvalidOperation(): void {
    // Act.
    $result = $this->plugin->getDataOperationResult('invalid_operation_name');
    // Assert: Should return empty array for invalid operation.
    $this->assertIsArray($result);
    $this->assertEmpty($result, 'Invalid operation should return empty array');
  }

  /**
   * Tests contentEntities() returns proper structure.
   */
  public function testContentEntitiesReturnsProperStructure(): void {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_types');
    // Assert: Verify structure.
    $this->assertIsArray($result);
    $this->assertArrayHasKey('header_table', $result);
    $this->assertArrayHasKey('results_table', $result);

    // Verify header structure has 4 columns (id, Group, Provider, Class).
    $this->assertCount(4, $result['header_table']);
    $this->assertIsArray($result['results_table']);
  }

  /**
   * Tests contentEntities() includes core content entity types.
   */
  public function testContentEntitiesIncludesCoreEntities(): void {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_types');

    // Assert: Verify core content entities are included.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Extract entity type IDs.
    $entity_type_ids = array_keys($results_table);

    // Core content entities that should be present.
    $expected_entities = ['node', 'taxonomy_term', 'block_content', 'user'];

    foreach ($expected_entities as $expected_entity) {
      $this->assertContains(
        $expected_entity,
        $entity_type_ids,
        "Results should include $expected_entity entity type"
      );
    }
  }

  /**
   * Tests contentEntities() filters by content group only.
   */
  public function testContentEntitiesFiltersContentGroupOnly(): void {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_types');

    // Assert: Verify only content entities are included.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Verify each entity is a content entity (group === 'content').
    foreach ($results_table as $key => $entity_data) {
      // Verify entity group by loading definition.
      $definition = $this->entityTypeManager->getDefinition($key);
      $this->assertEquals(
        'content',
        $definition->getGroup(),
        "Entity $key should be in content group"
      );
    }
  }

  /**
   * Tests contentEntities() includes correct entity data fields.
   */
  public function testContentEntitiesIncludesCorrectDataFields(): void {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_types');

    // Assert: Verify data fields.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Verify each row has required fields.
    foreach ($results_table as $key => $entity_data) {
      $this->assertArrayHasKey('key', $entity_data, "Row $key should have 'key' field");
      $this->assertArrayHasKey('group', $entity_data, "Row $key should have 'group' field");
      $this->assertArrayHasKey('provider', $entity_data, "Row $key should have 'provider' field");
      $this->assertArrayHasKey('class', $entity_data, "Row $key should have 'class' field");

      // Verify values are not empty.
      $this->assertNotEmpty($entity_data['key'], "Key should not be empty");
      $this->assertEquals('content', $entity_data['group'], "Group should be 'content'");
      $this->assertNotEmpty($entity_data['provider'], "Provider should not be empty");
      $this->assertNotEmpty($entity_data['class'], "Class should not be empty");
    }
  }

  /**
   * Tests contentEntities() provider values are correct.
   */
  public function testContentEntitiesProviderValues(): void {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_types');

    // Assert: Verify provider values.
    $results_table = $result['results_table'];
    $this->assertNotEmpty($results_table);

    // Check specific known entities and their providers.
    $expected_providers = [
      'node' => 'node',
      'taxonomy_term' => 'taxonomy',
      'block_content' => 'block_content',
      'user' => 'user',
    ];

    foreach ($expected_providers as $entity_type => $expected_provider) {
      if (isset($results_table[$entity_type])) {
        $this->assertEquals(
          $expected_provider,
          $results_table[$entity_type]['provider'],
          "Entity type $entity_type should have provider $expected_provider"
        );
      }
    }
  }

  /**
   * Tests header table has correct column labels.
   */
  public function testHeaderTableCorrectLabels(): void {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_types');
    // Assert: Verify header labels.
    $header_table = $result['header_table'];
    $this->assertCount(4, $header_table);

    // Headers should be translatable strings (id, Group, Provider, Class).
    // We can't check exact strings due to translation, but verify they exist.
    foreach ($header_table as $header) {
      $this->assertNotEmpty($header, 'Header label should not be empty');
    }
  }

  /**
   * Tests data structure is compatible with CSV export.
   */
  public function testDataStructureCompatibleWithCsvExport(): void {
    // Act.
    $result = $this->plugin->getDataOperationResult('content_entity_types');

    // Assert: Verify CSV compatibility.
    $header_table = $result['header_table'];
    $results_table = $result['results_table'];

    $header_count = count($header_table);

    // Each row should have same number of fields as header.
    foreach ($results_table as $key => $row) {
      $this->assertCount(
        $header_count,
        $row,
        "Row $key should have same field count as header"
      );
    }
  }

}
