<?php

namespace Drupal\Tests\xray_audit\Kernel\Services;

use Drupal\node\Entity\Node;
use Drupal\Tests\xray_audit\Kernel\XrayAuditKernelTestBase;
use Drupal\xray_audit\Services\EntityUseNode;

/**
 * Tests the EntityUseNode service.
 *
 * @codingStandardsIgnoreFile
 * @group xray_audit
 */
class EntityUseNodeTest extends XrayAuditKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'xray_audit',
  ];

  /**
   * The entity use node service under test.
   *
   * @var \Drupal\xray_audit\Services\EntityUseNode
   */
  protected $entityUseNode;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Clear static storage before each test.
    EntityUseNode::$storage = [];

    // Get the service under test.
    $this->entityUseNode = $this->container->get('xray_audit.entity_use_node');
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    // Clear static storage after each test.
    EntityUseNode::$storage = [];
    parent::tearDown();
  }

  /**
   * Tests countEntityUses() with no nodes.
   */
  public function testCountEntityUsesNoNodes() {
    // Arrange: Create a content type but no nodes.
    $this->createContentType('article', 'Article');

    // Act.
    $result = $this->entityUseNode->countEntityUses();

    // Assert.
    $this->assertIsArray($result);
    $this->assertNotEmpty($result);

    // Find article in results.
    $article_result = NULL;
    foreach ($result as $item) {
      if ($item['bundle'] === 'article') {
        $article_result = $item;
        break;
      }
    }

    $this->assertNotNull($article_result);
    $this->assertEquals(0, $article_result['count']);
    $this->assertEquals('node', $article_result['entity']);
  }

  /**
   * Tests countEntityUses() with published nodes.
   */
  public function testCountEntityUsesWithPublishedNodes() {
    // Arrange: Create content type and published nodes.
    $this->createContentType('page', 'Page');

    Node::create([
      'type' => 'page',
      'title' => 'Published Page 1',
      'status' => 1,
    ])->save();

    Node::create([
      'type' => 'page',
      'title' => 'Published Page 2',
      'status' => 1,
    ])->save();

    Node::create([
      'type' => 'page',
      'title' => 'Unpublished Page',
      'status' => 0,
    ])->save();

    // Act.
    $result = $this->entityUseNode->countEntityUses();

    // Assert.
    $page_result = NULL;
    foreach ($result as $item) {
      if ($item['bundle'] === 'page') {
        $page_result = $item;
        break;
      }
    }

    $this->assertNotNull($page_result);
    // Should count only published nodes.
    $this->assertEquals(2, $page_result['count']);
  }

  /**
   * Tests countEntityUses() with specific bundle initialization.
   */
  public function testCountEntityUsesWithSpecificBundle() {
    // Arrange: Create two content types with nodes.
    $this->createContentType('article', 'Article');
    $this->createContentType('page', 'Page');

    Node::create(['type' => 'article', 'title' => 'Article 1', 'status' => 1])->save();
    Node::create(['type' => 'page', 'title' => 'Page 1', 'status' => 1])->save();

    // Initialize with specific bundle.
    $this->entityUseNode->initParameters('node', 'article');

    // Act.
    $result = $this->entityUseNode->countEntityUses();

    // Assert: Should only return article count.
    $this->assertCount(1, $result);
    $this->assertEquals('article', $result[0]['bundle']);
    $this->assertEquals(1, $result[0]['count']);
  }

  /**
   * Tests getEntityUsePlaces() with published and unpublished nodes.
   */
  public function testGetEntityUsePlacesWithNodes() {
    // Arrange: Create content type with nodes.
    $this->createContentType('article', 'Article');

    $published = Node::create([
      'type' => 'article',
      'title' => 'Published Article',
      'status' => 1,
    ]);
    $published->save();

    $unpublished = Node::create([
      'type' => 'article',
      'title' => 'Unpublished Article',
      'status' => 0,
    ]);
    $unpublished->save();

    // Act.
    $result = $this->entityUseNode->getEntityUsePlaces();

    // Assert.
    $this->assertIsArray($result);

    // The method executes entityUsageData for published and unpublished.
    // Result structure depends on processResults method from EntityUseBase.
    // Verify method executed successfully and returns array.
    $this->assertIsArray($result, 'getEntityUsePlaces should return array');
  }

  /**
   * Tests countEntityUses() with multiple content types.
   */
  public function testCountEntityUsesMultipleTypes() {
    // Arrange: Create multiple types with different node counts.
    $this->createContentType('blog', 'Blog');
    $this->createContentType('news', 'News');

    // Create 3 blog posts.
    for ($i = 0; $i < 3; $i++) {
      Node::create([
        'type' => 'blog',
        'title' => "Blog Post $i",
        'status' => 1,
      ])->save();
    }

    // Create 5 news articles.
    for ($i = 0; $i < 5; $i++) {
      Node::create([
        'type' => 'news',
        'title' => "News Article $i",
        'status' => 1,
      ])->save();
    }

    // Act.
    $result = $this->entityUseNode->countEntityUses();

    // Assert.
    $blog_count = NULL;
    $news_count = NULL;

    foreach ($result as $item) {
      if ($item['bundle'] === 'blog') {
        $blog_count = $item['count'];
      }
      if ($item['bundle'] === 'news') {
        $news_count = $item['count'];
      }
    }

    $this->assertEquals(3, $blog_count);
    $this->assertEquals(5, $news_count);
  }

  /**
   * Tests countEntityUses() result structure.
   */
  public function testCountEntityUsesResultStructure() {
    // Arrange.
    $this->createContentType('structured', 'Structured');
    Node::create(['type' => 'structured', 'title' => 'Test', 'status' => 1])->save();

    // Act.
    $result = $this->entityUseNode->countEntityUses();

    // Assert: Check result structure.
    $this->assertNotEmpty($result);
    $first_result = reset($result);

    $this->assertArrayHasKey('parent_entity_type', $first_result);
    $this->assertArrayHasKey('entity', $first_result);
    $this->assertArrayHasKey('bundle', $first_result);
    $this->assertArrayHasKey('count', $first_result);
    $this->assertArrayHasKey('site', $first_result);

    $this->assertEquals('node', $first_result['parent_entity_type']);
    $this->assertEquals('node', $first_result['entity']);
  }

  /**
   * Tests getEntityUsePlaces() with specific bundle.
   */
  public function testGetEntityUsePlacesSpecificBundle() {
    // Arrange.
    $this->createContentType('specific', 'Specific');
    $this->createContentType('other', 'Other');

    Node::create(['type' => 'specific', 'title' => 'Specific Node', 'status' => 1])->save();
    Node::create(['type' => 'other', 'title' => 'Other Node', 'status' => 1])->save();

    $this->entityUseNode->initParameters('node', 'specific');

    // Act.
    $result = $this->entityUseNode->getEntityUsePlaces();

    // Assert: Should only have results for 'specific' bundle.
    foreach ($result as $item) {
      $this->assertEquals('specific', $item['bundle']);
    }
  }

}
